/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.gc;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import io.opentelemetry.api.trace.Span;
import io.opentelemetry.context.Scope;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.gc.GcCandidate;
import org.apache.accumulo.core.gc.Reference;
import org.apache.accumulo.core.metadata.schema.Ample;
import org.apache.accumulo.core.metadata.schema.MetadataSchema;
import org.apache.accumulo.core.trace.TraceUtil;
import org.apache.accumulo.gc.GarbageCollectionEnvironment;
import org.apache.accumulo.server.replication.StatusUtil;
import org.apache.accumulo.server.replication.proto.Replication;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollectionAlgorithm {
    private static final Logger log = LoggerFactory.getLogger(GarbageCollectionAlgorithm.class);

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private String makeRelative(String path, int expectedLen) {
        Object relPath = path;
        if (((String)relPath).startsWith("../")) {
            relPath = ((String)relPath).substring(3);
        }
        while (((String)relPath).endsWith("/")) {
            relPath = ((String)relPath).substring(0, ((String)relPath).length() - 1);
        }
        while (((String)relPath).startsWith("/")) {
            relPath = ((String)relPath).substring(1);
        }
        String[] tokens = (String[])Arrays.stream(((String)relPath).split("/")).filter(Predicate.not(""::equals)).toArray(String[]::new);
        if (tokens.length > 3 && path.contains(":")) {
            if (tokens[tokens.length - 4].equals("tables") && (expectedLen == 0 || expectedLen == 3)) {
                relPath = tokens[tokens.length - 3] + "/" + tokens[tokens.length - 2] + "/" + tokens[tokens.length - 1];
            } else {
                if (!tokens[tokens.length - 3].equals("tables") || expectedLen != 0 && expectedLen != 2) throw new IllegalArgumentException("Failed to make path relative. Bad reference: " + path);
                relPath = tokens[tokens.length - 2] + "/" + tokens[tokens.length - 1];
            }
        } else if (!(tokens.length != 3 || expectedLen != 0 && expectedLen != 3 || path.contains(":"))) {
            relPath = tokens[0] + "/" + tokens[1] + "/" + tokens[2];
        } else {
            if (tokens.length != 2 || expectedLen != 0 && expectedLen != 2 || path.contains(":")) throw new IllegalArgumentException("Failed to make path relative. Bad reference: " + path);
            relPath = tokens[0] + "/" + tokens[1];
        }
        log.trace("{} -> {} expectedLen = {}", new Object[]{path, relPath, expectedLen});
        return relPath;
    }

    private SortedMap<String, GcCandidate> makeRelative(Collection<GcCandidate> candidates) {
        TreeMap<String, GcCandidate> ret = new TreeMap<String, GcCandidate>();
        for (GcCandidate candidate : candidates) {
            String relPath;
            try {
                relPath = this.makeRelative(candidate.getPath(), 0);
            }
            catch (IllegalArgumentException iae) {
                log.warn("Ignoring invalid deletion candidate {}", (Object)candidate);
                continue;
            }
            ret.put(relPath, candidate);
        }
        return ret;
    }

    private void removeCandidatesInUse(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) throws InterruptedException {
        ArrayList<GcCandidate> candidateEntriesToBeDeleted = new ArrayList<GcCandidate>();
        Set<TableId> tableIdsBefore = gce.getCandidateTableIDs();
        HashSet<TableId> tableIdsSeen = new HashSet<TableId>();
        Iterator iter = gce.getReferences().iterator();
        while (iter.hasNext()) {
            String dir;
            GcCandidate gcT;
            Reference ref = (Reference)iter.next();
            tableIdsSeen.add(ref.getTableId());
            if (ref.isDirectory()) {
                MetadataSchema.TabletsSection.ServerColumnFamily.validateDirCol((String)ref.getMetadataEntry());
                Object dir2 = "/" + String.valueOf(ref.getTableId()) + "/" + ref.getMetadataEntry();
                GcCandidate gcTemp = (GcCandidate)candidateMap.remove(dir2 = this.makeRelative((String)dir2, 2));
                if (gcTemp == null) continue;
                log.debug("Directory Candidate was still in use by dir ref: {}", dir2);
                continue;
            }
            Object reference = ref.getMetadataEntry();
            if (((String)reference).startsWith("/")) {
                log.debug("Candidate {} has a relative path, prepend tableId {}", reference, (Object)ref.getTableId());
                reference = "/" + String.valueOf(ref.getTableId()) + ref.getMetadataEntry();
            } else if (!((String)reference).contains(":") && !((String)reference).startsWith("../")) {
                throw new RuntimeException("Bad file reference " + (String)reference);
            }
            String relativePath = this.makeRelative((String)reference, 3);
            GcCandidate gcTemp = (GcCandidate)candidateMap.remove(relativePath);
            if (gcTemp != null) {
                log.debug("File Candidate was still in use: {}", (Object)relativePath);
                if (!ref.isScan()) {
                    candidateEntriesToBeDeleted.add(gcTemp);
                }
            }
            if ((gcT = (GcCandidate)candidateMap.remove(dir = relativePath.substring(0, relativePath.lastIndexOf(47)))) == null) continue;
            log.debug("Directory Candidate was still in use by file ref: {}", (Object)relativePath);
        }
        Set<TableId> tableIdsAfter = gce.getCandidateTableIDs();
        this.ensureAllTablesChecked(Collections.unmodifiableSet(tableIdsBefore), Collections.unmodifiableSet(tableIdsSeen), Collections.unmodifiableSet(tableIdsAfter));
        if (gce.canRemoveInUseCandidates()) {
            gce.deleteGcCandidates(candidateEntriesToBeDeleted, Ample.GcCandidateType.INUSE);
        }
    }

    private long removeBlipCandidates(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) throws TableNotFoundException {
        long blipCount = 0L;
        boolean checkForBulkProcessingFiles = candidateMap.keySet().stream().anyMatch(relativePath -> relativePath.toLowerCase(Locale.ENGLISH).contains("b-"));
        if (checkForBulkProcessingFiles) {
            try (Stream<String> blipStream = gce.getBlipPaths();){
                Iterator blipiter = blipStream.iterator();
                while (blipiter.hasNext()) {
                    ++blipCount;
                    String blipPath = (String)blipiter.next();
                    blipPath = this.makeRelative(blipPath, 2);
                    Iterator<String> tailIter = candidateMap.tailMap(blipPath).keySet().iterator();
                    int count = 0;
                    while (tailIter.hasNext() && tailIter.next().startsWith(blipPath)) {
                        ++count;
                        tailIter.remove();
                    }
                    if (count <= 0) continue;
                    log.debug("Folder has bulk processing flag: {}", (Object)blipPath);
                }
            }
        }
        return blipCount;
    }

    @VisibleForTesting
    protected void ensureAllTablesChecked(Set<TableId> tableIdsBefore, Set<TableId> tableIdsSeen, Set<TableId> tableIdsAfter) {
        HashSet<TableId> tableIdsMustHaveSeen = new HashSet<TableId>(tableIdsBefore);
        tableIdsMustHaveSeen.retainAll(tableIdsAfter);
        if (tableIdsMustHaveSeen.isEmpty() && !tableIdsSeen.isEmpty()) {
            throw new RuntimeException("Garbage collection will not proceed because table ids were seen in the metadata table and none were seen Zookeeper. This can have two causes. First, total number of tables going to/from zero during a GC cycle will cause this. Second, it could be caused by corruption of the metadata table and/or Zookeeper. Only the second cause is problematic, but there is no way to distinguish between the two causes so this GC cycle will not proceed. The first cause should be transient and one would not expect to see this message repeated in subsequent GC cycles.");
        }
        tableIdsMustHaveSeen.removeAll(tableIdsSeen);
        if (!tableIdsMustHaveSeen.isEmpty()) {
            throw new IllegalStateException("Saw table IDs in ZK that were not in metadata table:  " + String.valueOf(tableIdsMustHaveSeen) + " TableIDs before GC: " + String.valueOf(tableIdsBefore) + ", TableIDs during GC: " + String.valueOf(tableIdsSeen) + ", TableIDs after GC: " + String.valueOf(tableIdsAfter));
        }
    }

    protected void confirmDeletesFromReplication(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) {
        Iterator<Map.Entry<String, Replication.Status>> replicationNeededIterator = gce.getReplicationNeededIterator();
        Iterator<Map.Entry<String, GcCandidate>> candidateMapIterator = candidateMap.entrySet().iterator();
        PeekingIterator pendingReplication = Iterators.peekingIterator(replicationNeededIterator);
        PeekingIterator candidates = Iterators.peekingIterator(candidateMapIterator);
        while (pendingReplication.hasNext() && candidates.hasNext()) {
            String fullPathCandidate;
            Map.Entry pendingReplica = (Map.Entry)pendingReplication.peek();
            Map.Entry candidate = (Map.Entry)candidates.peek();
            String filePendingReplication = (String)pendingReplica.getKey();
            int comparison = filePendingReplication.compareTo(fullPathCandidate = ((GcCandidate)candidate.getValue()).getPath());
            if (comparison < 0) {
                pendingReplication.next();
                continue;
            }
            if (comparison > 1) {
                candidates.next();
                continue;
            }
            candidates.next();
            pendingReplication.next();
            boolean safeToRemove = StatusUtil.isSafeForRemoval((Replication.Status)((Replication.Status)pendingReplica.getValue()));
            if (safeToRemove) continue;
            candidates.remove();
        }
    }

    private void cleanUpDeletedTableDirs(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) throws InterruptedException, IOException {
        HashSet<TableId> tableIdsWithDeletes = new HashSet<TableId>();
        for (String delete : candidateMap.keySet()) {
            String[] tokens = delete.split("/");
            if (tokens.length != 2) continue;
            TableId tableId = TableId.of((String)delete.split("/")[0]);
            tableIdsWithDeletes.add(tableId);
        }
        Set<TableId> tableIdsInZookeeper = gce.getTableIDs().keySet();
        tableIdsWithDeletes.removeAll(tableIdsInZookeeper);
        for (TableId delTableId : tableIdsWithDeletes) {
            gce.deleteTableDirIfEmpty(delTableId);
        }
    }

    private long confirmDeletesTrace(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) throws InterruptedException, TableNotFoundException {
        long blips = 0L;
        Span confirmDeletesSpan = TraceUtil.startSpan(this.getClass(), (String)"confirmDeletes");
        try (Scope scope = confirmDeletesSpan.makeCurrent();){
            blips = this.removeBlipCandidates(gce, candidateMap);
            this.removeCandidatesInUse(gce, candidateMap);
            this.confirmDeletesFromReplication(gce, candidateMap);
        }
        catch (Exception e) {
            TraceUtil.setException((Span)confirmDeletesSpan, (Throwable)e, (boolean)true);
            throw e;
        }
        finally {
            confirmDeletesSpan.end();
        }
        return blips;
    }

    private void deleteConfirmedCandidates(GarbageCollectionEnvironment gce, SortedMap<String, GcCandidate> candidateMap) throws InterruptedException, IOException, TableNotFoundException {
        Span deleteSpan = TraceUtil.startSpan(this.getClass(), (String)"deleteFiles");
        try (Scope deleteScope = deleteSpan.makeCurrent();){
            gce.deleteConfirmedCandidates(candidateMap);
        }
        catch (Exception e) {
            TraceUtil.setException((Span)deleteSpan, (Throwable)e, (boolean)true);
            throw e;
        }
        finally {
            deleteSpan.end();
        }
        this.cleanUpDeletedTableDirs(gce, candidateMap);
    }

    public long collect(GarbageCollectionEnvironment gce) throws InterruptedException, TableNotFoundException, IOException {
        Iterator<GcCandidate> candidatesIter = gce.getCandidates();
        long totalBlips = 0L;
        int batchCount = 0;
        while (candidatesIter.hasNext()) {
            List<GcCandidate> batchOfCandidates;
            Span candidatesSpan = TraceUtil.startSpan(this.getClass(), (String)"getCandidates");
            try (Scope candidatesScope = candidatesSpan.makeCurrent();){
                batchOfCandidates = gce.readCandidatesThatFitInMemory(candidatesIter);
            }
            catch (Exception e) {
                TraceUtil.setException((Span)candidatesSpan, (Throwable)e, (boolean)true);
                throw e;
            }
            finally {
                candidatesSpan.end();
            }
            totalBlips = this.deleteBatch(gce, batchOfCandidates, ++batchCount);
        }
        return totalBlips;
    }

    private long deleteBatch(GarbageCollectionEnvironment gce, List<GcCandidate> currentBatch, int batchCount) throws InterruptedException, TableNotFoundException, IOException {
        long origSize = currentBatch.size();
        gce.incrementCandidatesStat(origSize);
        log.info("Batch {} total deletion candidates: {}", (Object)batchCount, (Object)origSize);
        SortedMap<String, GcCandidate> candidateMap = this.makeRelative(currentBatch);
        long blips = this.confirmDeletesTrace(gce, candidateMap);
        gce.incrementInUseStat(origSize - (long)candidateMap.size());
        this.deleteConfirmedCandidates(gce, candidateMap);
        return blips;
    }
}

