// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.catalog;

import org.apache.doris.analysis.UserIdentity;
import org.apache.doris.common.AnalysisException;
import org.apache.doris.common.ExceptionChecker;
import org.apache.doris.common.FeConstants;
import org.apache.doris.datasource.CatalogIf;
import org.apache.doris.datasource.ExternalDatabase;
import org.apache.doris.datasource.test.TestExternalCatalog;
import org.apache.doris.datasource.test.TestExternalDatabase;
import org.apache.doris.datasource.test.TestExternalTable;
import org.apache.doris.mysql.privilege.Auth;
import org.apache.doris.nereids.parser.NereidsParser;
import org.apache.doris.nereids.trees.plans.commands.CreateCatalogCommand;
import org.apache.doris.nereids.trees.plans.commands.CreateUserCommand;
import org.apache.doris.nereids.trees.plans.commands.DropCatalogCommand;
import org.apache.doris.nereids.trees.plans.commands.GrantTablePrivilegeCommand;
import org.apache.doris.nereids.trees.plans.commands.refresh.RefreshDatabaseCommand;
import org.apache.doris.nereids.trees.plans.logical.LogicalPlan;
import org.apache.doris.qe.ConnectContext;
import org.apache.doris.qe.StmtExecutor;
import org.apache.doris.utframe.TestWithFeService;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

import java.util.List;
import java.util.Map;

public class RefreshDbTest extends TestWithFeService {
    private static Env env;
    private ConnectContext rootCtx;

    @Override
    protected void runBeforeAll() throws Exception {
        FeConstants.runningUnitTest = true;
        rootCtx = createDefaultCtx();
        env = Env.getCurrentEnv();

        // 1. create test catalog
        String createStmt = "create catalog test1 properties(\n"
                + "    \"type\" = \"test\",\n"
                + "    \"catalog_provider.class\" "
                + "= \"org.apache.doris.catalog.RefreshTableTest$RefreshTableProvider\"\n"
                + ");";

        NereidsParser nereidsParser = new NereidsParser();
        LogicalPlan logicalPlan = nereidsParser.parseSingle(createStmt);
        if (logicalPlan instanceof CreateCatalogCommand) {
            ((CreateCatalogCommand) logicalPlan).run(rootCtx, null);
        }
    }

    @Override
    protected void runAfterAll() throws Exception {
        super.runAfterAll();
        rootCtx.setThreadLocalInfo();
        NereidsParser nereidsParser = new NereidsParser();
        LogicalPlan logicalPlan = nereidsParser.parseSingle("drop catalog test1");
        if (logicalPlan instanceof DropCatalogCommand) {
            ((DropCatalogCommand) logicalPlan).run(rootCtx, null);
        }
    }

    @Test
    public void testRefreshDatabase() throws Exception {
        CatalogIf test1 = env.getCatalogMgr().getCatalog("test1");
        TestExternalDatabase db1 = (TestExternalDatabase) test1.getDbNullable("db1");
        long l1 = db1.getLastUpdateTime();
        Assertions.assertTrue(l1 == 0);
        TestExternalTable table = db1.getTable("tbl11").get();
        long l2 = db1.getLastUpdateTime();
        Assertions.assertTrue(l2 > l1);
        Assertions.assertFalse(table.isObjectCreated());
        table.makeSureInitialized();
        Assertions.assertTrue(table.isObjectCreated());

        RefreshDatabaseCommand refreshDatabaseCommand = new RefreshDatabaseCommand("test1", "db1", null);
        refreshDatabaseCommand.run(connectContext, null);

        long l3 = db1.getLastUpdateTime();
        Assertions.assertTrue(l3 == l2);
        // the table will be recreated after refresh.
        // so we need to get table again
        table = db1.getTable("tbl11").get();
        Assertions.assertFalse(table.isObjectCreated());
        test1.getDbNullable("db1").getTables();
        Assertions.assertFalse(table.isObjectCreated());
        refreshDatabaseCommand.run(connectContext, null);

        Assertions.assertFalse(((ExternalDatabase) test1.getDbNullable("db1")).isInitialized());
        table.makeSureInitialized();
        long l4 = db1.getLastUpdateTime();
        Assertions.assertTrue(l4 > l3);
        Assertions.assertTrue(((ExternalDatabase) test1.getDbNullable("db1")).isInitialized());
        refreshDatabaseCommand.run(connectContext, null);
    }

    @Test
    public void testRefreshPriv() throws Exception {
        Auth auth = Env.getCurrentEnv().getAuth();
        // create user1
        NereidsParser nereidsParser = new NereidsParser();
        String sql = "create user 'user1'@'%' identified by 'pwd1';";
        LogicalPlan parsed = nereidsParser.parseSingle(sql);
        StmtExecutor stmtExecutor = new StmtExecutor(rootCtx, sql);
        if (parsed instanceof CreateUserCommand) {
            ((CreateUserCommand) parsed).run(rootCtx, stmtExecutor);
        }

        // mock login user1
        UserIdentity user1 = new UserIdentity("user1", "%");
        user1.analyze();
        ConnectContext user1Ctx = createCtx(user1, "127.0.0.1");
        RefreshDatabaseCommand command = (RefreshDatabaseCommand) parseStmt(
                "refresh database test1.db1", user1Ctx);
        ExceptionChecker.expectThrowsWithMsg(AnalysisException.class,
                "Access denied",
                () -> command.run(rootCtx, stmtExecutor));
        ConnectContext.remove();

        // add drop priv to user1
        rootCtx.setThreadLocalInfo();
        LogicalPlan logicalPlan1 = nereidsParser.parseSingle("grant drop_priv on test1.db1.* to 'user1'@'%';");
        Assertions.assertTrue(logicalPlan1 instanceof GrantTablePrivilegeCommand);
        GrantTablePrivilegeCommand command1 = (GrantTablePrivilegeCommand) logicalPlan1;
        command1.validate();
        auth.grantTablePrivilegeCommand(command1);
        ConnectContext.remove();

        // user1 can do refresh table
        user1Ctx.setThreadLocalInfo();
        RefreshDatabaseCommand command2 = (RefreshDatabaseCommand) parseStmt(
                "refresh database test1.db1", user1Ctx);
        ExceptionChecker.expectThrowsNoException(
                () -> command2.run(rootCtx, stmtExecutor));
    }

    public static class RefreshTableProvider implements TestExternalCatalog.TestCatalogProvider {
        public static final Map<String, Map<String, List<Column>>> MOCKED_META;

        static {
            MOCKED_META = Maps.newHashMap();
            Map<String, List<Column>> tblSchemaMap1 = Maps.newHashMap();
            // db1
            tblSchemaMap1.put("tbl11", Lists.newArrayList(
                new Column("a11", PrimitiveType.BIGINT),
                new Column("a12", PrimitiveType.STRING),
                new Column("a13", PrimitiveType.FLOAT)));
            tblSchemaMap1.put("tbl12", Lists.newArrayList(
                new Column("b21", PrimitiveType.BIGINT),
                new Column("b22", PrimitiveType.STRING),
                new Column("b23", PrimitiveType.FLOAT)));
            MOCKED_META.put("db1", tblSchemaMap1);
            // db2
            Map<String, List<Column>> tblSchemaMap2 = Maps.newHashMap();
            tblSchemaMap2.put("tbl21", Lists.newArrayList(
                new Column("c11", PrimitiveType.BIGINT),
                new Column("c12", PrimitiveType.STRING),
                new Column("c13", PrimitiveType.FLOAT)));
            MOCKED_META.put("db2", tblSchemaMap2);
        }

        @Override
        public Map<String, Map<String, List<Column>>> getMetadata() {
            return MOCKED_META;
        }
    }
}
