/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.mapred;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.RawComparator;
import org.apache.hadoop.io.WritableUtils;
import org.apache.hadoop.io.serializer.Serializer;
import org.apache.hadoop.mapred.RawKeyValueIterator;
import org.apache.hadoop.util.Progress;
import org.apache.uniffle.common.serializer.SerializerInstance;
import org.apache.uniffle.shaded.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SortWriteBuffer<K, V>
extends OutputStream {
    private static final Logger LOG = LoggerFactory.getLogger(SortWriteBuffer.class);
    private long copyTime = 0L;
    private final List<WrappedBuffer> buffers = Lists.newArrayList();
    private final List<Record<K>> records = Lists.newArrayList();
    private int dataLength = 0;
    private long sortTime = 0L;
    private final RawComparator<K> comparator;
    private long maxSegmentSize;
    private int partitionId;
    private Serializer<K> keySerializer;
    private Serializer<V> valSerializer;
    private int currentOffset = 0;
    private int currentIndex = 0;
    private final boolean useUniffleSerializer;
    private final SerializerInstance serializerInstance;
    private DataOutputStream dataOutputStream;

    public SortWriteBuffer(int partitionId, RawComparator<K> comparator, long maxSegmentSize, boolean useUniffleSerializer, Serializer<K> keySerializer, Serializer<V> valueSerializer, SerializerInstance serializerInstance) {
        this.partitionId = partitionId;
        this.comparator = comparator;
        this.maxSegmentSize = maxSegmentSize;
        this.useUniffleSerializer = useUniffleSerializer;
        this.keySerializer = keySerializer;
        this.valSerializer = valueSerializer;
        this.serializerInstance = serializerInstance;
        if (useUniffleSerializer) {
            this.dataOutputStream = new DataOutputStream(this);
        }
    }

    public int addRecord(K key, V value) throws IOException {
        if (!this.useUniffleSerializer) {
            this.keySerializer.open((OutputStream)this);
            this.valSerializer.open((OutputStream)this);
        }
        int lastOffSet = this.currentOffset;
        int lastIndex = this.currentIndex;
        int lastDataLength = this.dataLength;
        int keyIndex = lastIndex;
        if (this.useUniffleSerializer) {
            this.serializerInstance.serialize(key, this.dataOutputStream);
        } else {
            this.keySerializer.serialize(key);
        }
        int keyLength = this.dataLength - lastDataLength;
        int keyOffset = lastOffSet;
        if (this.compact(lastIndex, lastOffSet, keyLength)) {
            keyOffset = lastOffSet;
            keyIndex = lastIndex;
        }
        lastDataLength = this.dataLength;
        if (this.useUniffleSerializer) {
            this.serializerInstance.serialize(value, this.dataOutputStream);
        } else {
            this.valSerializer.serialize(value);
        }
        int valueLength = this.dataLength - lastDataLength;
        this.records.add(new Record(keyIndex, keyOffset, keyLength, valueLength));
        return keyLength + valueLength;
    }

    public void clear() {
        this.buffers.clear();
        this.records.clear();
    }

    public synchronized void sort() {
        long startSort = System.currentTimeMillis();
        this.records.sort((o1, o2) -> this.comparator.compare(this.buffers.get(o1.getKeyIndex()).getBuffer(), o1.getKeyOffSet(), o1.getKeyLength(), this.buffers.get(o2.getKeyIndex()).getBuffer(), o2.getKeyOffSet(), o2.getKeyLength()));
        long finishSort = System.currentTimeMillis();
        this.sortTime += finishSort - startSort;
    }

    public synchronized byte[] getData() {
        int extraSize = 0;
        for (Record<K> record : this.records) {
            extraSize += WritableUtils.getVIntSize((long)record.getKeyLength());
            extraSize += WritableUtils.getVIntSize((long)record.getValueLength());
        }
        extraSize += WritableUtils.getVIntSize((long)-1L);
        byte[] data = new byte[this.dataLength + (extraSize += WritableUtils.getVIntSize((long)-1L))];
        int offset = 0;
        long startCopy = System.currentTimeMillis();
        for (Record<K> record : this.records) {
            offset = this.writeDataInt(data, offset, record.getKeyLength());
            offset = this.writeDataInt(data, offset, record.getValueLength());
            int recordLength = record.getKeyLength() + record.getValueLength();
            int copyOffset = record.getKeyOffSet();
            int copyIndex = record.getKeyIndex();
            while (recordLength > 0) {
                byte[] srcBytes = this.buffers.get(copyIndex).getBuffer();
                int length = copyOffset + recordLength;
                int copyLength = recordLength;
                if (length > srcBytes.length) {
                    copyLength = srcBytes.length - copyOffset;
                }
                System.arraycopy(srcBytes, copyOffset, data, offset, copyLength);
                copyOffset = 0;
                ++copyIndex;
                recordLength -= copyLength;
                offset += copyLength;
            }
        }
        offset = this.writeDataInt(data, offset, -1L);
        this.writeDataInt(data, offset, -1L);
        this.copyTime += System.currentTimeMillis() - startCopy;
        return data;
    }

    private boolean compact(int lastIndex, int lastOffset, int dataLength) {
        if (lastIndex != this.currentIndex) {
            int i;
            if (LOG.isDebugEnabled()) {
                LOG.debug("compact lastIndex {}, currentIndex {}, lastOffset {} currentOffset {} dataLength {}", new Object[]{lastIndex, this.currentIndex, lastOffset, this.currentOffset, dataLength});
            }
            WrappedBuffer buffer = new WrappedBuffer(lastOffset + dataLength);
            int offset = 0;
            for (i = lastIndex; i < this.currentIndex; ++i) {
                byte[] sourceBuffer = this.buffers.get(i).getBuffer();
                System.arraycopy(sourceBuffer, 0, buffer.getBuffer(), offset, sourceBuffer.length);
                offset += sourceBuffer.length;
            }
            System.arraycopy(this.buffers.get(this.currentIndex).getBuffer(), 0, buffer.getBuffer(), offset, this.currentOffset);
            for (i = this.currentIndex; i >= lastIndex; --i) {
                this.buffers.remove(i);
            }
            this.buffers.add(buffer);
            this.currentOffset = 0;
            WrappedBuffer anotherBuffer = new WrappedBuffer((int)this.maxSegmentSize);
            this.buffers.add(anotherBuffer);
            this.currentIndex = this.buffers.size() - 1;
            return true;
        }
        return false;
    }

    private int writeDataInt(byte[] data, int offset, long dataInt) {
        if (dataInt >= -112L && dataInt <= 127L) {
            data[offset] = (byte)dataInt;
            ++offset;
        } else {
            int len = -112;
            if (dataInt < 0L) {
                dataInt ^= 0xFFFFFFFFFFFFFFFFL;
                len = -120;
            }
            long tmp = dataInt;
            while (tmp != 0L) {
                tmp >>= 8;
                --len;
            }
            data[offset] = (byte)len;
            ++offset;
            for (int idx = len = len < -120 ? -(len + 120) : -(len + 112); idx != 0; --idx) {
                int shiftBits = (idx - 1) * 8;
                long mask = 255L << shiftBits;
                data[offset] = (byte)((dataInt & mask) >> shiftBits);
                ++offset;
            }
        }
        return offset;
    }

    public int getDataLength() {
        return this.dataLength;
    }

    public long getCopyTime() {
        return this.copyTime;
    }

    public long getSortTime() {
        return this.sortTime;
    }

    public int getPartitionId() {
        return this.partitionId;
    }

    @Override
    public void write(int b) throws IOException {
        if (this.buffers.isEmpty()) {
            this.buffers.add(new WrappedBuffer((int)this.maxSegmentSize));
        }
        if ((long)(1 + this.currentOffset) > this.maxSegmentSize) {
            ++this.currentIndex;
            this.currentOffset = 0;
            this.buffers.add(new WrappedBuffer((int)this.maxSegmentSize));
        }
        WrappedBuffer buffer = this.buffers.get(this.currentIndex);
        buffer.getBuffer()[this.currentOffset] = (byte)b;
        ++this.currentOffset;
        ++this.dataLength;
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        if (this.buffers.isEmpty()) {
            this.buffers.add(new WrappedBuffer((int)this.maxSegmentSize));
        }
        int bufferNum = (int)((long)(this.currentOffset + len) / this.maxSegmentSize);
        for (int i = 0; i < bufferNum; ++i) {
            this.buffers.add(new WrappedBuffer((int)this.maxSegmentSize));
        }
        int index = this.currentIndex;
        int offset = this.currentOffset;
        int srcPos = 0;
        while (len > 0) {
            int copyLength = 0;
            if ((long)(offset + len) >= this.maxSegmentSize) {
                copyLength = (int)(this.maxSegmentSize - (long)offset);
                this.currentOffset = 0;
            } else {
                copyLength = len;
                this.currentOffset += len;
            }
            System.arraycopy(b, srcPos, this.buffers.get(index).getBuffer(), offset, copyLength);
            offset = 0;
            srcPos += copyLength;
            ++index;
            len -= copyLength;
            this.dataLength += copyLength;
        }
        this.currentIndex += bufferNum;
    }

    public static class SortBufferIterator<K, V>
    implements RawKeyValueIterator {
        private final SortWriteBuffer<K, V> sortWriteBuffer;
        private final Iterator<Record<K>> iterator;
        private final DataInputBuffer keyBuffer = new DataInputBuffer();
        private final DataInputBuffer valueBuffer = new DataInputBuffer();
        private Record<K> currentRecord;

        public SortBufferIterator(SortWriteBuffer<K, V> sortWriteBuffer) {
            this.sortWriteBuffer = sortWriteBuffer;
            this.iterator = ((SortWriteBuffer)sortWriteBuffer).records.iterator();
        }

        private byte[] fetchDataFromBuffers(int index, int offset, int length) {
            while (offset >= ((WrappedBuffer)((SortWriteBuffer)this.sortWriteBuffer).buffers.get(index)).getSize()) {
                offset -= ((WrappedBuffer)((SortWriteBuffer)this.sortWriteBuffer).buffers.get(index)).getSize();
                ++index;
            }
            byte[] data = new byte[length];
            int copyDestPos = 0;
            while (length > 0) {
                WrappedBuffer currentBuffer = (WrappedBuffer)((SortWriteBuffer)this.sortWriteBuffer).buffers.get(index);
                byte[] currentBufferData = currentBuffer.getBuffer();
                int currentBufferCapacity = currentBuffer.getSize();
                int copyLength = Math.min(currentBufferCapacity - offset, length);
                System.arraycopy(currentBufferData, offset, data, copyDestPos, copyLength);
                length -= copyLength;
                copyDestPos += copyLength;
                ++index;
                offset = 0;
            }
            return data;
        }

        public DataInputBuffer getKey() {
            int keyIndex = this.currentRecord.getKeyIndex();
            int keyOffset = this.currentRecord.getKeyOffSet();
            int keyLength = this.currentRecord.getKeyLength();
            byte[] keyData = this.fetchDataFromBuffers(keyIndex, keyOffset, keyLength);
            this.keyBuffer.reset(keyData, 0, keyLength);
            return this.keyBuffer;
        }

        public DataInputBuffer getValue() {
            int keyIndex = this.currentRecord.getKeyIndex();
            int valueOffset = this.currentRecord.getKeyOffSet() + this.currentRecord.getKeyLength();
            int valueLength = this.currentRecord.getValueLength();
            byte[] valueData = this.fetchDataFromBuffers(keyIndex, valueOffset, valueLength);
            this.valueBuffer.reset(valueData, 0, valueLength);
            return this.valueBuffer;
        }

        public boolean next() {
            if (this.iterator.hasNext()) {
                this.currentRecord = this.iterator.next();
                return true;
            }
            return false;
        }

        public void close() throws IOException {
        }

        public Progress getProgress() {
            return new Progress();
        }
    }

    private static final class WrappedBuffer {
        private byte[] buffer;
        private int size;

        WrappedBuffer(int size) {
            this.buffer = new byte[size];
            this.size = size;
        }

        public byte[] getBuffer() {
            return this.buffer;
        }

        public int getSize() {
            return this.size;
        }
    }

    private static final class Record<K> {
        private final int keyIndex;
        private final int keyOffSet;
        private final int keyLength;
        private final int valueLength;

        Record(int keyIndex, int keyOffset, int keyLength, int valueLength) {
            this.keyIndex = keyIndex;
            this.keyOffSet = keyOffset;
            this.keyLength = keyLength;
            this.valueLength = valueLength;
        }

        public int getKeyIndex() {
            return this.keyIndex;
        }

        public int getKeyOffSet() {
            return this.keyOffSet;
        }

        public int getKeyLength() {
            return this.keyLength;
        }

        public int getValueLength() {
            return this.valueLength;
        }
    }
}

