﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/appflow/AppflowRequest.h>
#include <aws/appflow/Appflow_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Appflow {
namespace Model {

/**
 */
class DeleteFlowRequest : public AppflowRequest {
 public:
  AWS_APPFLOW_API DeleteFlowRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteFlow"; }

  AWS_APPFLOW_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> The specified name of the flow. Spaces are not allowed. Use underscores (_)
   * or hyphens (-) only. </p>
   */
  inline const Aws::String& GetFlowName() const { return m_flowName; }
  inline bool FlowNameHasBeenSet() const { return m_flowNameHasBeenSet; }
  template <typename FlowNameT = Aws::String>
  void SetFlowName(FlowNameT&& value) {
    m_flowNameHasBeenSet = true;
    m_flowName = std::forward<FlowNameT>(value);
  }
  template <typename FlowNameT = Aws::String>
  DeleteFlowRequest& WithFlowName(FlowNameT&& value) {
    SetFlowName(std::forward<FlowNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates whether Amazon AppFlow should delete the flow, even if it is
   * currently in use. </p>
   */
  inline bool GetForceDelete() const { return m_forceDelete; }
  inline bool ForceDeleteHasBeenSet() const { return m_forceDeleteHasBeenSet; }
  inline void SetForceDelete(bool value) {
    m_forceDeleteHasBeenSet = true;
    m_forceDelete = value;
  }
  inline DeleteFlowRequest& WithForceDelete(bool value) {
    SetForceDelete(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_flowName;

  bool m_forceDelete{false};
  bool m_flowNameHasBeenSet = false;
  bool m_forceDeleteHasBeenSet = false;
};

}  // namespace Model
}  // namespace Appflow
}  // namespace Aws
