﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudfront/CloudFront_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace CloudFront {
namespace Model {

/**
 * <p>An alias (also called a CNAME) and the CloudFront standard distribution and
 * Amazon Web Services account ID that it's associated with. The standard
 * distribution and account IDs are partially hidden, which allows you to identify
 * the standard distributions and accounts that you own, and helps to protect the
 * information of ones that you don't own.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudfront-2020-05-31/ConflictingAlias">AWS
 * API Reference</a></p>
 */
class ConflictingAlias {
 public:
  AWS_CLOUDFRONT_API ConflictingAlias() = default;
  AWS_CLOUDFRONT_API ConflictingAlias(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_CLOUDFRONT_API ConflictingAlias& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_CLOUDFRONT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>An alias (also called a CNAME).</p>
   */
  inline const Aws::String& GetAlias() const { return m_alias; }
  inline bool AliasHasBeenSet() const { return m_aliasHasBeenSet; }
  template <typename AliasT = Aws::String>
  void SetAlias(AliasT&& value) {
    m_aliasHasBeenSet = true;
    m_alias = std::forward<AliasT>(value);
  }
  template <typename AliasT = Aws::String>
  ConflictingAlias& WithAlias(AliasT&& value) {
    SetAlias(std::forward<AliasT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The (partially hidden) ID of the CloudFront standard distribution associated
   * with the alias.</p>
   */
  inline const Aws::String& GetDistributionId() const { return m_distributionId; }
  inline bool DistributionIdHasBeenSet() const { return m_distributionIdHasBeenSet; }
  template <typename DistributionIdT = Aws::String>
  void SetDistributionId(DistributionIdT&& value) {
    m_distributionIdHasBeenSet = true;
    m_distributionId = std::forward<DistributionIdT>(value);
  }
  template <typename DistributionIdT = Aws::String>
  ConflictingAlias& WithDistributionId(DistributionIdT&& value) {
    SetDistributionId(std::forward<DistributionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The (partially hidden) ID of the Amazon Web Services account that owns the
   * standard distribution that's associated with the alias.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  ConflictingAlias& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_alias;

  Aws::String m_distributionId;

  Aws::String m_accountId;
  bool m_aliasHasBeenSet = false;
  bool m_distributionIdHasBeenSet = false;
  bool m_accountIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFront
}  // namespace Aws
