﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 */
class GetTokensFromRefreshTokenRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API GetTokensFromRefreshTokenRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetTokensFromRefreshToken"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A valid refresh token that can authorize the request for new tokens. When
   * refresh token rotation is active in the requested app client, this token is
   * invalidated after the request is complete and after an optional grace
   * period.</p>
   */
  inline const Aws::String& GetRefreshToken() const { return m_refreshToken; }
  inline bool RefreshTokenHasBeenSet() const { return m_refreshTokenHasBeenSet; }
  template <typename RefreshTokenT = Aws::String>
  void SetRefreshToken(RefreshTokenT&& value) {
    m_refreshTokenHasBeenSet = true;
    m_refreshToken = std::forward<RefreshTokenT>(value);
  }
  template <typename RefreshTokenT = Aws::String>
  GetTokensFromRefreshTokenRequest& WithRefreshToken(RefreshTokenT&& value) {
    SetRefreshToken(std::forward<RefreshTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The app client that issued the refresh token to the user who wants to request
   * new tokens.</p>
   */
  inline const Aws::String& GetClientId() const { return m_clientId; }
  inline bool ClientIdHasBeenSet() const { return m_clientIdHasBeenSet; }
  template <typename ClientIdT = Aws::String>
  void SetClientId(ClientIdT&& value) {
    m_clientIdHasBeenSet = true;
    m_clientId = std::forward<ClientIdT>(value);
  }
  template <typename ClientIdT = Aws::String>
  GetTokensFromRefreshTokenRequest& WithClientId(ClientIdT&& value) {
    SetClientId(std::forward<ClientIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The client secret of the requested app client, if the client has a
   * secret.</p>
   */
  inline const Aws::String& GetClientSecret() const { return m_clientSecret; }
  inline bool ClientSecretHasBeenSet() const { return m_clientSecretHasBeenSet; }
  template <typename ClientSecretT = Aws::String>
  void SetClientSecret(ClientSecretT&& value) {
    m_clientSecretHasBeenSet = true;
    m_clientSecret = std::forward<ClientSecretT>(value);
  }
  template <typename ClientSecretT = Aws::String>
  GetTokensFromRefreshTokenRequest& WithClientSecret(ClientSecretT&& value) {
    SetClientSecret(std::forward<ClientSecretT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When you enable device remembering, Amazon Cognito issues a device key that
   * you can use for device authentication that bypasses multi-factor authentication
   * (MFA). To implement <code>GetTokensFromRefreshToken</code> in a user pool with
   * device remembering, you must capture the device key from the initial
   * authentication request. If your application doesn't provide the key of a
   * registered device, Amazon Cognito issues a new one. You must provide the
   * confirmed device key in this request if device remembering is enabled in your
   * user pool.</p> <p>For more information about device remembering, see <a
   * href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html">Working
   * with devices</a>.</p>
   */
  inline const Aws::String& GetDeviceKey() const { return m_deviceKey; }
  inline bool DeviceKeyHasBeenSet() const { return m_deviceKeyHasBeenSet; }
  template <typename DeviceKeyT = Aws::String>
  void SetDeviceKey(DeviceKeyT&& value) {
    m_deviceKeyHasBeenSet = true;
    m_deviceKey = std::forward<DeviceKeyT>(value);
  }
  template <typename DeviceKeyT = Aws::String>
  GetTokensFromRefreshTokenRequest& WithDeviceKey(DeviceKeyT&& value) {
    SetDeviceKey(std::forward<DeviceKeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of custom key-value pairs that you can provide as input for certain
   * custom workflows that this action triggers.</p> <p>You create custom workflows
   * by assigning Lambda functions to user pool triggers. When you use the
   * <code>GetTokensFromRefreshToken</code> API action, Amazon Cognito invokes the
   * Lambda function the pre token generation trigger.</p> <p>For more information,
   * see <a
   * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">
   * Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.</p>
   *  <p>When you use the <code>ClientMetadata</code> parameter, note that
   * Amazon Cognito won't do the following:</p> <ul> <li> <p>Store the
   * <code>ClientMetadata</code> value. This data is available only to Lambda
   * triggers that are assigned to a user pool to support custom workflows. If your
   * user pool configuration doesn't include triggers, the
   * <code>ClientMetadata</code> parameter serves no purpose.</p> </li> <li>
   * <p>Validate the <code>ClientMetadata</code> value.</p> </li> <li> <p>Encrypt the
   * <code>ClientMetadata</code> value. Don't send sensitive information in this
   * parameter.</p> </li> </ul>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetClientMetadata() const { return m_clientMetadata; }
  inline bool ClientMetadataHasBeenSet() const { return m_clientMetadataHasBeenSet; }
  template <typename ClientMetadataT = Aws::Map<Aws::String, Aws::String>>
  void SetClientMetadata(ClientMetadataT&& value) {
    m_clientMetadataHasBeenSet = true;
    m_clientMetadata = std::forward<ClientMetadataT>(value);
  }
  template <typename ClientMetadataT = Aws::Map<Aws::String, Aws::String>>
  GetTokensFromRefreshTokenRequest& WithClientMetadata(ClientMetadataT&& value) {
    SetClientMetadata(std::forward<ClientMetadataT>(value));
    return *this;
  }
  template <typename ClientMetadataKeyT = Aws::String, typename ClientMetadataValueT = Aws::String>
  GetTokensFromRefreshTokenRequest& AddClientMetadata(ClientMetadataKeyT&& key, ClientMetadataValueT&& value) {
    m_clientMetadataHasBeenSet = true;
    m_clientMetadata.emplace(std::forward<ClientMetadataKeyT>(key), std::forward<ClientMetadataValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_refreshToken;

  Aws::String m_clientId;

  Aws::String m_clientSecret;

  Aws::String m_deviceKey;

  Aws::Map<Aws::String, Aws::String> m_clientMetadata;
  bool m_refreshTokenHasBeenSet = false;
  bool m_clientIdHasBeenSet = false;
  bool m_clientSecretHasBeenSet = false;
  bool m_deviceKeyHasBeenSet = false;
  bool m_clientMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
