﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune/Neptune_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Neptune {
namespace Model {

/**
 * <p>Describes an Amazon Identity and Access Management (IAM) role that is
 * associated with a DB cluster.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/neptune-2014-10-31/DBClusterRole">AWS
 * API Reference</a></p>
 */
class DBClusterRole {
 public:
  AWS_NEPTUNE_API DBClusterRole() = default;
  AWS_NEPTUNE_API DBClusterRole(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_NEPTUNE_API DBClusterRole& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role that is associated with the DB
   * cluster.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  DBClusterRole& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the state of association between the IAM role and the DB cluster.
   * The Status property returns one of the following values:</p> <ul> <li> <p>
   * <code>ACTIVE</code> - the IAM role ARN is associated with the DB cluster and can
   * be used to access other Amazon services on your behalf.</p> </li> <li> <p>
   * <code>PENDING</code> - the IAM role ARN is being associated with the DB
   * cluster.</p> </li> <li> <p> <code>INVALID</code> - the IAM role ARN is
   * associated with the DB cluster, but the DB cluster is unable to assume the IAM
   * role in order to access other Amazon services on your behalf.</p> </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  DBClusterRole& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the feature associated with the Amazon Identity and Access
   * Management (IAM) role. For the list of supported feature names, see
   * <a>DescribeDBEngineVersions</a>. </p>
   */
  inline const Aws::String& GetFeatureName() const { return m_featureName; }
  inline bool FeatureNameHasBeenSet() const { return m_featureNameHasBeenSet; }
  template <typename FeatureNameT = Aws::String>
  void SetFeatureName(FeatureNameT&& value) {
    m_featureNameHasBeenSet = true;
    m_featureName = std::forward<FeatureNameT>(value);
  }
  template <typename FeatureNameT = Aws::String>
  DBClusterRole& WithFeatureName(FeatureNameT&& value) {
    SetFeatureName(std::forward<FeatureNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleArn;

  Aws::String m_status;

  Aws::String m_featureName;
  bool m_roleArnHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_featureNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
