﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSMRequest.h>
#include <aws/ssm/SSM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSM {
namespace Model {

/**
 */
class GetMaintenanceWindowExecutionTaskInvocationRequest : public SSMRequest {
 public:
  AWS_SSM_API GetMaintenanceWindowExecutionTaskInvocationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetMaintenanceWindowExecutionTaskInvocation"; }

  AWS_SSM_API Aws::String SerializePayload() const override;

  AWS_SSM_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the maintenance window execution for which the task is a part.</p>
   */
  inline const Aws::String& GetWindowExecutionId() const { return m_windowExecutionId; }
  inline bool WindowExecutionIdHasBeenSet() const { return m_windowExecutionIdHasBeenSet; }
  template <typename WindowExecutionIdT = Aws::String>
  void SetWindowExecutionId(WindowExecutionIdT&& value) {
    m_windowExecutionIdHasBeenSet = true;
    m_windowExecutionId = std::forward<WindowExecutionIdT>(value);
  }
  template <typename WindowExecutionIdT = Aws::String>
  GetMaintenanceWindowExecutionTaskInvocationRequest& WithWindowExecutionId(WindowExecutionIdT&& value) {
    SetWindowExecutionId(std::forward<WindowExecutionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the specific task in the maintenance window task that should be
   * retrieved. </p>
   */
  inline const Aws::String& GetTaskId() const { return m_taskId; }
  inline bool TaskIdHasBeenSet() const { return m_taskIdHasBeenSet; }
  template <typename TaskIdT = Aws::String>
  void SetTaskId(TaskIdT&& value) {
    m_taskIdHasBeenSet = true;
    m_taskId = std::forward<TaskIdT>(value);
  }
  template <typename TaskIdT = Aws::String>
  GetMaintenanceWindowExecutionTaskInvocationRequest& WithTaskId(TaskIdT&& value) {
    SetTaskId(std::forward<TaskIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The invocation ID to retrieve.</p>
   */
  inline const Aws::String& GetInvocationId() const { return m_invocationId; }
  inline bool InvocationIdHasBeenSet() const { return m_invocationIdHasBeenSet; }
  template <typename InvocationIdT = Aws::String>
  void SetInvocationId(InvocationIdT&& value) {
    m_invocationIdHasBeenSet = true;
    m_invocationId = std::forward<InvocationIdT>(value);
  }
  template <typename InvocationIdT = Aws::String>
  GetMaintenanceWindowExecutionTaskInvocationRequest& WithInvocationId(InvocationIdT&& value) {
    SetInvocationId(std::forward<InvocationIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_windowExecutionId;

  Aws::String m_taskId;

  Aws::String m_invocationId;
  bool m_windowExecutionIdHasBeenSet = false;
  bool m_taskIdHasBeenSet = false;
  bool m_invocationIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
