﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/inspector2/Inspector2Request.h>
#include <aws/inspector2/Inspector2_EXPORTS.h>
#include <aws/inspector2/model/Service.h>

#include <utility>

namespace Aws {
namespace Inspector2 {
namespace Model {

/**
 */
class ListAccountPermissionsRequest : public Inspector2Request {
 public:
  AWS_INSPECTOR2_API ListAccountPermissionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAccountPermissions"; }

  AWS_INSPECTOR2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The service scan type to check permissions for.</p>
   */
  inline Service GetService() const { return m_service; }
  inline bool ServiceHasBeenSet() const { return m_serviceHasBeenSet; }
  inline void SetService(Service value) {
    m_serviceHasBeenSet = true;
    m_service = value;
  }
  inline ListAccountPermissionsRequest& WithService(Service value) {
    SetService(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results the response can return. If your request would
   * return more than the maximum the response will return a <code>nextToken</code>
   * value, use this value when you call the action again to get the remaining
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAccountPermissionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token to use for paginating results that are returned in the response. Set
   * the value of this parameter to null for the first request to a list action. If
   * your response returns more than the <code>maxResults</code> maximum value it
   * will also return a <code>nextToken</code> value. For subsequent calls, use the
   * NextToken value returned from the previous request to continue listing results
   * after the first page.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAccountPermissionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Service m_service{Service::NOT_SET};
  bool m_serviceHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector2
}  // namespace Aws
