/*
 * Decompiled with CFR 0.152.
 */
package org.sleuthkit.datamodel;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Table;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.sleuthkit.datamodel.TskCoreException;

final class WindowsAccountUtils {
    static final String SPECIAL_WINDOWS_BACK_UP_POSTFIX = ".bak";
    static final String WINDOWS_NULL_SID = "S-1-0-0";
    private static final Set<String> GROUP_SIDS = ImmutableSet.of((Object)"S-1-1-0", (Object)"S-1-2-0", (Object)"S-1-2-1", (Object)"S-1-3-1", (Object)"S-1-3-4", (Object)"S-1-5-1", (Object[])new String[]{"S-1-5-2", "S-1-5-3", "S-1-5-4", "S-1-5-6", "S-1-5-7", "S-1-5-9", "S-1-5-11", "S-1-5-12", "S-1-5-13", "S-1-5-14", "S-1-5-15", "S-1-5-80-0", "S-1-5-83-0", "S-1-5-90-0"});
    private static final Set<String> GROUP_SID_PREFIX = ImmutableSet.of((Object)"S-1-5-32", (Object)"S-1-5-87");
    private static final String NTAUTHORITY_SID_PREFIX = "S-1-5";
    private static final String NTAUTHORITY_REALM_NAME = "NT AUTHORITY";
    private static final Set<String> DOMAIN_GROUP_SID_SUFFIX = ImmutableSet.of((Object)"-512", (Object)"-513", (Object)"-514", (Object)"-515", (Object)"-516", (Object)"-517", (Object[])new String[]{"-518", "-519", "-520", "-526", "-527", "-533", "-498", "-521", "-571", "-572", "-522"});
    private static final Map<String, WellKnownSidInfo> SPECIAL_SIDS_MAP = ImmutableMap.builder().put((Object)"S-1-5-17", (Object)new WellKnownSidInfo(true, "S-1-5", "NT AUTHORITY", "IUSR", "IIS Default Account")).put((Object)"S-1-5-18", (Object)new WellKnownSidInfo(true, "S-1-5", "NT AUTHORITY", "SYSTEM", "Local System Account")).put((Object)"S-1-5-19", (Object)new WellKnownSidInfo(true, "S-1-5", "NT AUTHORITY", "LOCAL SERVICE", "Local Service Account")).put((Object)"S-1-5-20", (Object)new WellKnownSidInfo(true, "S-1-5", "NT AUTHORITY", "NETWORK SERVICE", "Network Service Account")).build();
    private static final Map<String, WellKnownSidInfo> SPECIAL_SID_PREFIXES_MAP = ImmutableMap.builder().put((Object)"S-1-5-80", (Object)new WellKnownSidInfo(false, "S-1-5-80", "NT SERVICE", "", "NT Service Virtual Account")).put((Object)"S-1-5-82", (Object)new WellKnownSidInfo(false, "S-1-5-82", "IIS APPPOOL", "", "IIS AppPool Virtual Account")).put((Object)"S-1-5-83", (Object)new WellKnownSidInfo(false, "S-1-5-83", "NT VIRTUAL MACHINE", "", "Virtual Machine Virtual Account")).put((Object)"S-1-5-90", (Object)new WellKnownSidInfo(false, "S-1-5-90", "Window Manager", "", "Windows Manager Virtual Account")).put((Object)"S-1-5-94", (Object)new WellKnownSidInfo(false, "S-1-5-94", "WinRM Virtual Users", "", "Windows Remoting Virtual Account")).put((Object)"S-1-5-96", (Object)new WellKnownSidInfo(false, "S-1-5-96", "Font Driver Host", "", "Font Driver Host Virtual Account")).build();
    private static final Pattern WINDOWS_SPECIAL_ACCOUNT_PREFIX_REGEX = Pattern.compile("^[sS]\\-1\\-5\\-(\\d+)\\-");
    private static final Table<String, String, String> SPECIAL_ACCOUNTS_TO_SID_MAP = HashBasedTable.create();
    private static final Map<String, String> REALM_NAME_TO_ENGLISH_MAP;
    private static final Map<String, String> LOGINNAME_TO_ENGLISH_MAP;

    WindowsAccountUtils() {
    }

    static boolean isWindowsWellKnownSid(String sid) {
        Integer domainIdentifier;
        String tempSID = WindowsAccountUtils.stripWindowsBackupPostfix(sid);
        if (SPECIAL_SIDS_MAP.containsKey(tempSID)) {
            return true;
        }
        for (String specialPrefix : SPECIAL_SID_PREFIXES_MAP.keySet()) {
            if (!tempSID.startsWith(specialPrefix)) continue;
            return true;
        }
        Matcher match = WINDOWS_SPECIAL_ACCOUNT_PREFIX_REGEX.matcher(tempSID);
        return match.find() && (domainIdentifier = Integer.valueOf(match.group(1))) != null && domainIdentifier >= 80 && domainIdentifier <= 111;
    }

    static boolean isWindowsWellKnownAccountName(String loginName, String realmName) {
        String resolvedRealmName = WindowsAccountUtils.toWellknownEnglishRealmName(realmName);
        String resolvedLoginName = WindowsAccountUtils.toWellknownEnglishLoginName(loginName);
        if (StringUtils.isBlank((CharSequence)resolvedRealmName) || StringUtils.isBlank((CharSequence)resolvedLoginName)) {
            return false;
        }
        return SPECIAL_ACCOUNTS_TO_SID_MAP.contains((Object)resolvedRealmName.toUpperCase(), (Object)resolvedLoginName.toUpperCase());
    }

    private static String getWindowsWellKnownSidRealmAddr(String sid) throws TskCoreException {
        Integer domainIdentifier;
        String tempSID = WindowsAccountUtils.stripWindowsBackupPostfix(sid);
        if (SPECIAL_SIDS_MAP.containsKey(tempSID)) {
            return SPECIAL_SIDS_MAP.get(tempSID).getRealmAddr();
        }
        for (Map.Entry<String, WellKnownSidInfo> specialPrefixEntry : SPECIAL_SID_PREFIXES_MAP.entrySet()) {
            if (!tempSID.startsWith(specialPrefixEntry.getKey())) continue;
            return specialPrefixEntry.getValue().getRealmAddr();
        }
        Matcher match = WINDOWS_SPECIAL_ACCOUNT_PREFIX_REGEX.matcher(tempSID);
        if (match.find() && (domainIdentifier = Integer.valueOf(match.group(1))) != null && domainIdentifier >= 80 && domainIdentifier <= 111) {
            String realmAddr = String.format("%s-%d", NTAUTHORITY_SID_PREFIX, domainIdentifier);
            return realmAddr;
        }
        return "";
    }

    private static WellKnownSidInfo getWindowsWellKnownInfo(String sid) {
        String tempSID = WindowsAccountUtils.stripWindowsBackupPostfix(sid);
        if (SPECIAL_SIDS_MAP.containsKey(tempSID)) {
            return SPECIAL_SIDS_MAP.get(tempSID);
        }
        for (Map.Entry<String, WellKnownSidInfo> specialPrefixEntry : SPECIAL_SID_PREFIXES_MAP.entrySet()) {
            if (!tempSID.startsWith(specialPrefixEntry.getKey())) continue;
            return specialPrefixEntry.getValue();
        }
        return null;
    }

    static String getWindowsWellKnownSidFullName(String sid) {
        WellKnownSidInfo wellKnownSidInfo = WindowsAccountUtils.getWindowsWellKnownInfo(sid);
        return Objects.nonNull(wellKnownSidInfo) ? wellKnownSidInfo.getDescription() : "";
    }

    static String getWindowsWellKnownSidRealmName(String sid) {
        if (StringUtils.isNotBlank((CharSequence)sid) && sid.equals(NTAUTHORITY_SID_PREFIX)) {
            return NTAUTHORITY_REALM_NAME;
        }
        WellKnownSidInfo wellKnownSidInfo = WindowsAccountUtils.getWindowsWellKnownInfo(sid);
        return Objects.nonNull(wellKnownSidInfo) ? wellKnownSidInfo.getRealmName() : null;
    }

    static String getWindowsWellKnownSidLoginName(String sid) {
        WellKnownSidInfo wellKnownSidInfo = WindowsAccountUtils.getWindowsWellKnownInfo(sid);
        return Objects.nonNull(wellKnownSidInfo) ? wellKnownSidInfo.getLoginName() : null;
    }

    static String getWindowsWellKnownAccountSid(String loginName, String realmName) {
        String resolvedRealmName = WindowsAccountUtils.toWellknownEnglishRealmName(realmName);
        String resolvedLoginName = WindowsAccountUtils.toWellknownEnglishLoginName(loginName);
        if (StringUtils.isBlank((CharSequence)resolvedRealmName) || StringUtils.isBlank((CharSequence)resolvedLoginName)) {
            return null;
        }
        return (String)SPECIAL_ACCOUNTS_TO_SID_MAP.get((Object)resolvedRealmName.toUpperCase(), (Object)resolvedLoginName.toUpperCase());
    }

    static String toWellknownEnglishRealmName(String name) {
        return StringUtils.isNotBlank((CharSequence)name) ? REALM_NAME_TO_ENGLISH_MAP.getOrDefault(name.toUpperCase(), name) : null;
    }

    static String toWellknownEnglishLoginName(String name) {
        return StringUtils.isNotBlank((CharSequence)name) ? LOGINNAME_TO_ENGLISH_MAP.getOrDefault(name.toUpperCase(), name) : null;
    }

    static boolean isWindowsUserSid(String sid) {
        String tempSID = WindowsAccountUtils.stripWindowsBackupPostfix(sid);
        if (GROUP_SIDS.contains(tempSID)) {
            return false;
        }
        for (String prefix : GROUP_SID_PREFIX) {
            if (!tempSID.startsWith(prefix)) continue;
            return false;
        }
        if (tempSID.startsWith(NTAUTHORITY_SID_PREFIX)) {
            for (String suffix : DOMAIN_GROUP_SID_SUFFIX) {
                if (!tempSID.endsWith(suffix)) continue;
                return false;
            }
        }
        return true;
    }

    public static String getWindowsRealmAddress(String sid) throws TskCoreException {
        String realmAddr;
        String tempSID = WindowsAccountUtils.stripWindowsBackupPostfix(sid);
        if (WindowsAccountUtils.isWindowsWellKnownSid(tempSID)) {
            realmAddr = WindowsAccountUtils.getWindowsWellKnownSidRealmAddr(sid);
        } else {
            if (StringUtils.countMatches((CharSequence)tempSID, (CharSequence)"-") < 3) {
                throw new TskCoreException(String.format("Invalid SID %s for a host/domain", tempSID));
            }
            realmAddr = sid.substring(0, tempSID.lastIndexOf(45));
        }
        return realmAddr;
    }

    private static String stripWindowsBackupPostfix(String sid) {
        String tempSID = sid;
        if (tempSID.endsWith(SPECIAL_WINDOWS_BACK_UP_POSTFIX)) {
            tempSID = tempSID.replace(SPECIAL_WINDOWS_BACK_UP_POSTFIX, "");
        }
        return tempSID;
    }

    static {
        SPECIAL_ACCOUNTS_TO_SID_MAP.put((Object)NTAUTHORITY_REALM_NAME, (Object)"SYSTEM", (Object)"S-1-5-18");
        SPECIAL_ACCOUNTS_TO_SID_MAP.put((Object)NTAUTHORITY_REALM_NAME, (Object)"LOCAL SERVICE", (Object)"S-1-5-19");
        SPECIAL_ACCOUNTS_TO_SID_MAP.put((Object)NTAUTHORITY_REALM_NAME, (Object)"NETWORK SERVICE", (Object)"S-1-5-20");
        REALM_NAME_TO_ENGLISH_MAP = ImmutableMap.builder().put((Object)NTAUTHORITY_REALM_NAME, (Object)NTAUTHORITY_REALM_NAME).put((Object)"NT-AUTORIT\u00c4T", (Object)NTAUTHORITY_REALM_NAME).put((Object)"AUTORITE NT", (Object)NTAUTHORITY_REALM_NAME).put((Object)"NT INSTANS", (Object)NTAUTHORITY_REALM_NAME).build();
        LOGINNAME_TO_ENGLISH_MAP = ImmutableMap.builder().put((Object)"SYSTEM", (Object)"SYSTEM").put((Object)"SYST\u00c8ME", (Object)"SYSTEM").put((Object)"LOCAL SERVICE", (Object)"LOCAL SERVICE").put((Object)"LOKALER DIENST", (Object)"LOCAL SERVICE").put((Object)"SERVICE LOCAL", (Object)"LOCAL SERVICE").put((Object)"SERVIZIO LOCALE", (Object)"LOCAL SERVICE").put((Object)"SERVICIO LOC", (Object)"LOCAL SERVICE").put((Object)"NETWORK SERVICE", (Object)"NETWORK SERVICE").put((Object)"NETZWERKDIENST", (Object)"NETWORK SERVICE").put((Object)"N\u00c4TVERKSTJ\u00c4NST", (Object)"NETWORK SERVICE").put((Object)"SERVICE R\u00c9SEAU", (Object)"NETWORK SERVICE").put((Object)"SERVIZIO DI RETE", (Object)"NETWORK SERVICE").put((Object)"SERVICIO DE RED", (Object)"NETWORK SERVICE").build();
    }

    public static class WellKnownSidInfo {
        private final String realmAddr;
        private final boolean isUserSID;
        private final String realmName;
        private final String loginName;
        private final String description;

        WellKnownSidInfo(boolean isUserSID, String addr, String realmName, String loginName, String description) {
            this.realmAddr = addr;
            this.isUserSID = isUserSID;
            this.realmName = realmName;
            this.loginName = this.isUserSID ? loginName : "";
            this.description = description;
        }

        public String getRealmAddr() {
            return this.realmAddr;
        }

        public boolean isIsUserSID() {
            return this.isUserSID;
        }

        public String getRealmName() {
            return this.realmName;
        }

        public String getLoginName() {
            return this.loginName;
        }

        public String getDescription() {
            return this.description;
        }
    }
}

