use Test2::V0;
use Bitcoin::Crypto::BIP85;
use Bitcoin::Crypto qw(btc_extprv);
use Bitcoin::Crypto::Util qw(to_format);

# Data from https://github.com/bitcoin/bips/blob/master/bip-0085.mediawiki
# extra data generated by https://iancoleman.io/bip39/

subtest 'should derive_entropy' => sub {
	my $bip85 = Bitcoin::Crypto::BIP85->new(
		key => btc_extprv->from_serialized(
			[
				base58 =>
					'xprv9s21ZrQH143K2LBWUUQRFXhucrQqBpKdRRxNVq2zBqsx8HVqFk2uYo8kmbaLLHRdqtQpUm98uKfu3vca1LqdGhUtyoFnCNkfmXRyPXLjbKb'
			]
		),
	);

	is to_format [hex => $bip85->derive_entropy(q{m/83696968'/0'/0'})],
		'efecfbccffea313214232d29e71563d941229afb4338c21f9517c41aaa0d16f00b83d2a09ef747e7a64e8e2bd5a14869e693da66ce94ac2da570ab7ee48618f7',
		'entropy index 0 derived ok';

	is to_format [hex => $bip85->derive_entropy(q{m/83696968'/0'/0'}, 32)],
		'efecfbccffea313214232d29e71563d941229afb4338c21f9517c41aaa0d16f0',
		'truncated entropy index 0 ok';

	is to_format [hex => $bip85->derive_entropy(q{m/83696968'/0'/0'}, 80)],
		'b78b1ee6b345eae6836c2d53d33c64cdaf9a696487be81b03e822dc84b3f1cd883d7559e53d175f243e4c349e822a957bbff9224bc5dde9492ef54e8a439f6bc8c7355b87a925a37ee405a7502991111',
		'stretched entropy index 0 ok';

	is to_format [hex => $bip85->derive_entropy(q{m/83696968'/0'/1'})],
		'70c6e3e8ebee8dc4c0dbba66076819bb8c09672527c4277ca8729532ad711872218f826919f6b67218adde99018a6df9095ab2b58d803b5b93ec9802085a690e',
		'entropy index 1 derived ok';
};

subtest 'should derive a mnemonic according to BIP39 application of BIP85' => sub {
	my $bip85 = Bitcoin::Crypto::BIP85->new(
		key => btc_extprv->from_serialized(
			[
				base58 =>
					'xprv9s21ZrQH143K2LBWUUQRFXhucrQqBpKdRRxNVq2zBqsx8HVqFk2uYo8kmbaLLHRdqtQpUm98uKfu3vca1LqdGhUtyoFnCNkfmXRyPXLjbKb'
			]
		),
	);

	# check words
	is $bip85->derive_mnemonic(words => 12),
		'girl mad pet galaxy egg matter matrix prison refuse sense ordinary nose', '12 words ok';
	is $bip85->derive_mnemonic(words => 18),
		'near account window bike charge season chef number sketch tomorrow excuse sniff circle vital hockey outdoor supply token',
		'18 words ok';
	is $bip85->derive_mnemonic(words => 24),
		'puppy ocean match cereal symbol another shed magic wrap hammer bulb intact gadget divorce twin tonight reason outdoor destroy simple truth cigar social volcano',
		'24 words ok';

	# check index
	is $bip85->derive_mnemonic(words => 12, index => 1),
		'mystery car occur shallow stable order number feature else best trigger curious', '12 words index 1 ok';
};

subtest 'should derive a wif according to HD-Seed WIF application of BIP85' => sub {
	my $bip85 = Bitcoin::Crypto::BIP85->new(
		key => btc_extprv->from_serialized(
			[
				base58 =>
					'xprv9s21ZrQH143K2LBWUUQRFXhucrQqBpKdRRxNVq2zBqsx8HVqFk2uYo8kmbaLLHRdqtQpUm98uKfu3vca1LqdGhUtyoFnCNkfmXRyPXLjbKb'
			]
		),
	);

	is $bip85->derive_prv->to_wif, 'Kzyv4uF39d4Jrw2W7UryTHwZr1zQVNk4dAFyqE6BuMrMh1Za7uhp', 'wif ok';
	is $bip85->derive_prv(index => 1)->to_wif, 'L45nghBsnmqaGj9Vy64FCw9AyJNi6K4LUFP4r41tYHmQLEyXUkYP',
		'wif index 1 ok';
};

subtest 'should derive an extprv according to XPRV application of BIP85' => sub {
	my $bip85 = Bitcoin::Crypto::BIP85->new(
		key => btc_extprv->from_serialized(
			[
				base58 =>
					'xprv9s21ZrQH143K2LBWUUQRFXhucrQqBpKdRRxNVq2zBqsx8HVqFk2uYo8kmbaLLHRdqtQpUm98uKfu3vca1LqdGhUtyoFnCNkfmXRyPXLjbKb'
			]
		),
	);

	is to_format [base58 => $bip85->derive_extprv->to_serialized],
		'xprv9s21ZrQH143K2srSbCSg4m4kLvPMzcWydgmKEnMmoZUurYuBuYG46c6P71UGXMzmriLzCCBvKQWBUv3vPB3m1SATMhp3uEjXHJ42jFg7myX',
		'xprv ok';
	is to_format [base58 => $bip85->derive_extprv(index => 1)->to_serialized],
		'xprv9s21ZrQH143K38mDZkjswdWQv6DWyjWiejciPywBBZsCnZ9Vg3WCWnhkPW3rKsPT6u3MnhDn52huxjBjFES1xCzEtxTSAfQTapE7CXcbQ4b',
		'xprv index 1 ok';
};

subtest 'should derive bytes according to HEX application of BIP85' => sub {
	my $bip85 = Bitcoin::Crypto::BIP85->new(
		key => btc_extprv->from_serialized(
			[
				base58 =>
					'xprv9s21ZrQH143K2LBWUUQRFXhucrQqBpKdRRxNVq2zBqsx8HVqFk2uYo8kmbaLLHRdqtQpUm98uKfu3vca1LqdGhUtyoFnCNkfmXRyPXLjbKb'
			]
		),
	);

	is to_format [hex => $bip85->derive_bytes],
		'492db4698cf3b73a5a24998aa3e9d7fa96275d85724a91e71aa2d645442f878555d078fd1f1f67e368976f04137b1f7a0d19232136ca50c44614af72b5582a5c',
		'bytes ok';
	is to_format [hex => $bip85->derive_bytes(index => 1)],
		'3c7cd8fc51f7381c83c91e838f893405b9fdf14b36c847535173c9ef79723095d3ba70d28a8981129ef3937401e402cd8e7046f17fc9d65d0488107678f213ad',
		'bytes index 1 ok';
	is to_format [hex => $bip85->derive_bytes(bytes => 30)],
		'a8196c02597687a879a029e70cb3926db8f987792b7a19be3c7d6d978c45', 'bytes count 30 ok';
};

done_testing;

