package Map::Tube::Madrid;

$Map::Tube::Madrid::VERSION   = '0.34';
$Map::Tube::Madrid::AUTHORITY = 'cpan:MANWAR';

=head1 NAME

Map::Tube::Madrid - Interface to the Madrid Metro Map.

=head1 VERSION

Version 0.34

=cut

use 5.006;
use Data::Dumper;
use File::ShareDir ':ALL';

use Moo;
use namespace::autoclean;

has json => (is => 'ro', default => sub { return dist_file('Map-Tube-Madrid', 'madrid-map.json') });
with 'Map::Tube';

=head1 DESCRIPTION

Built and released while giving talk on the topic "Introduction to Map::Tube" at
German Perl Workshop 2018.
Source: L<map|https://www.planometromadrid.org/mapas-metro/plano-metro-madrid-2018-01.png>

It currently only covers the following metro lines.

=over 4

=item * L<L1|Map::Tube::Madrid::Line::L1>

=item * L<L2|Map::Tube::Madrid::Line::L2>

=item * L<L3|Map::Tube::Madrid::Line::L3>

=item * L<L4|Map::Tube::Madrid::Line::L4>

=item * L<L5|Map::Tube::Madrid::Line::L5>

=item * L<L11|Map::Tube::Madrid::Line::L11>

=back

=head1 TODO (Metro Lines)

=over 4

=item * L6

=item * L7

=item * L8

=item * L9

=item * L10

=item * L12

=back

=head1 CONSTRUCTOR

The constructor DO NOT expects parameters.This setup the default node definitions.

    use strict; use warnings;
    use Map::Tube::Madrid;

    my $metro = Map::Tube::Madrid->new;

=head1 METHODS

=head2 get_shortest_route($from, $to)

It expects C<$from> and C<$to> station name, required param. It returns an object
of type L<Map::Tube::Route>. On error it throws exception of type L<Map::Tube::Exception>.

    use strict; use warnings;
    use Map::Tube::Madrid;

    my $metro = Map::Tube::Madrid->new;
    my $route = $metro->get_shortest_route('Sol', 'Delicias');

    print "Route: $route\n";

=head2 as_image($line_name)

It expects the plugin  L<Map::Tube::Plugin::Graph> to be  installed. Returns line
image  as  base64  encoded string if C<$line_name> passed in otherwise it returns
base64 encoded string of the entire map.

    use strict; use warnings;
    use MIME::Base64;
    use Map::Tube::Madrid;

    my $metro = Map::Tube::Madrid->new;

    # Entire map image
    my $map = $metro->name;
    open(my $MAP_IMAGE, ">", "$map.png")
        or die "ERROR: Can't open [$map.png]: $!";
    binmode($MAP_IMAGE);
    print $MAP_IMAGE decode_base64($metro->as_image);
    close($MAP_IMAGE);

    # Just a particular line map image
    my $line = 'L3';
    open(my $LINE_IMAGE, ">", "$line.png")
        or die "ERROR: Can't open [$line.png]: $!";
    binmode($LINE_IMAGE);
    print $LINE_IMAGE decode_base64($metro->as_image($line));
    close($LINE_IMAGE);

The L<Madrid Metro Map|https://raw.githubusercontent.com/manwar/Map-Tube-Madrid/master/maps/Madrid-Metro.png>
map as generated by plugin L<Map::Tube::Plugin::Graph>.

=begin html

<a href = "https://raw.githubusercontent.com/manwar/Map-Tube-Madrid/master/maps/Madrid-Metro.png">
<img src    = "https://raw.githubusercontent.com/manwar/Map-Tube-Madrid/master/maps/Madrid-Metro.png"
     alt    = "Madrid Metro Map"
     width  = "500px"
     height = "900px"/>
</a>

=end html

=head1 AUTHOR

Mohammad S Anwar, C<< <mohammad.anwar at yahoo.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-map-tube-madrid at rt.cpan.org>,
or through the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Map-Tube-Madrid>.
I will be notified, and then you'll automatically be notified of progress on your
bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Map::Tube::Madrid

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Map-Tube-Madrid>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Map-Tube-Madrid>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Map-Tube-Madrid>

=item * Search CPAN

L<http://search.cpan.org/dist/Map-Tube-Madrid/>

=back

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2018 Mohammad S Anwar.

This program  is free software; you can  redistribute it and / or modify it under
the  terms  of the the Artistic License  (2.0). You may obtain a copy of the full
license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any  use,  modification, and distribution of the Standard or Modified Versions is
governed by this Artistic License.By using, modifying or distributing the Package,
you accept this license. Do not use, modify, or distribute the Package, if you do
not accept this license.

If your Modified Version has been derived from a Modified Version made by someone
other than you,you are nevertheless required to ensure that your Modified Version
 complies with the requirements of this license.

This  license  does  not grant you the right to use any trademark,  service mark,
tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge patent license
to make,  have made, use,  offer to sell, sell, import and otherwise transfer the
Package with respect to any patent claims licensable by the Copyright Holder that
are  necessarily  infringed  by  the  Package. If you institute patent litigation
(including  a  cross-claim  or  counterclaim) against any party alleging that the
Package constitutes direct or contributory patent infringement,then this Artistic
License to you shall terminate on the date that such litigation is filed.

Disclaimer  of  Warranty:  THE  PACKAGE  IS  PROVIDED BY THE COPYRIGHT HOLDER AND
CONTRIBUTORS  "AS IS'  AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES. THE IMPLIED
WARRANTIES    OF   MERCHANTABILITY,   FITNESS   FOR   A   PARTICULAR  PURPOSE, OR
NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY YOUR LOCAL LAW. UNLESS
REQUIRED BY LAW, NO COPYRIGHT HOLDER OR CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL,  OR CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE
OF THE PACKAGE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1; # End of Map::Tube::Madrid
