#!/usr/bin/perl -w
#
#    DateFormat.t
#
#    Copyright (c) 2000 Brian Stell
#
#    This package is free software and is provided ``as is'' without
#    express or implied warranty. It may be used, redistributed and/or
#    modified under the terms of the Perl Artistic License
#    (see http://www.perl.com/perl/misc/Artistic.html)
#

use strict;
use Test;
BEGIN { plan tests => 9 }
use ExtUtils::testlib;

use utf8;
use ICU qw(U_ZERO_ERROR);
use ICU::DateFormat qw(:EStyle :EField);
use ICU::Locale;

#
# Make sure we always use the same fallback locale
#
my $default_locale = new ICU::Locale('en_US');
ICU::Locale->setDefault($default_locale);

my $tz_pst = ICU::TimeZone->createTimeZone('PST');
ICU::TimeZone->setDefault($tz_pst);

ok(check_enums(), 1);
ok(check_getAmPmStrings(), 1);
ok(check_getLocalPatternChars(), 1);
ok(check_getMonths(), 1);
ok(check_getShortMonths(), 1);
ok(check_getShortWeekdays(), 1);
ok(check_getWeekdays(), 1);
ok(check_DateFormatSymbols__new(), 1);
ok(check_setDebugLevel(), 1);

exit(0);

########################################################################
#
# The subroutines and tests
#
########################################################################

sub check_enums {
    my $result;
    $result = 1;
    return($result);
}

sub check_getAmPmStrings {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my @ampm_strs = $dfs->getAmPmStrings();
    if (scalar(@ampm_strs) != 2) {
        $result = 0;
    }
    if ($ampm_strs[0] ne 'AM') {
        $result = 0;
    }
    if ($ampm_strs[1] ne 'PM') {
        $result = 0;
    }
    return($result);
}

sub check_getLocalPatternChars {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my $localPatternChars = $dfs->getLocalPatternChars();
    if ($localPatternChars ne 'GyMdkHmsSEDFwWahKzYe') {
        $result = 0;
    }
    return($result);
}

sub check_getMonths {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my @months = $dfs->getMonths();
    if (scalar(@months) != 12) {
        $result = 0;
    }
    if ($months[0] ne 'January') {
        $result = 0;
    }
    if ($months[1] ne 'February') {
        $result = 0;
    }
    return($result);
}

sub check_getShortMonths {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my @months = $dfs->getShortMonths();
    if (scalar(@months) != 12) {
        $result = 0;
    }
    if ($months[0] ne 'Jan') {
        $result = 0;
    }
    if ($months[1] ne 'Feb') {
        $result = 0;
    }
    return($result);
}

sub check_getShortWeekdays {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my @weekdays = $dfs->getShortWeekdays();
    #print "weekdays = (".join(')(',@weekdays).")\n";
    if (scalar(@weekdays) != 8) { # don't know why 8 instead of 7
        $result = 0;
    }
    if ($weekdays[0] ne '') {
        $result = 0;
    }
    if ($weekdays[1] ne 'Sun') {
        $result = 0;
    }
    if ($weekdays[2] ne 'Mon') {
        $result = 0;
    }
    return($result);
}

sub check_getWeekdays {
    my $result = 1;

    my $locale = new ICU::Locale('en_US');
    my $dfs = new ICU::DateFormatSymbols($locale);
    my @weekdays = $dfs->getWeekdays();
    #print "weekdays = (".join(')(',@weekdays).")\n";
    if (scalar(@weekdays) != 8) { # don't know why 8 instead of 7
        $result = 0;
    }
    if ($weekdays[0] ne '') {
        $result = 0;
    }
    if ($weekdays[1] ne 'Sunday') {
        $result = 0;
    }
    if ($weekdays[2] ne 'Monday') {
        $result = 0;
    }
    return($result);
}

sub check_DateFormatSymbols__new {
    my $result = 1;

    my $status = U_ZERO_ERROR;
    my $locale = new ICU::Locale('en_US');
    my $dfs1 = new ICU::DateFormatSymbols($locale, \$status);
    if (!$dfs1) {
        $result = 0;
    }
    $status = int(1);
    $dfs1 = new ICU::DateFormatSymbols($locale, \$status);
    if ($dfs1) {
        $result = 0;
    }
    my $dfs2 = new ICU::DateFormatSymbols($locale);
    my $dfs3 = new ICU::DateFormatSymbols(\$status);
    my $dfs4 = new ICU::DateFormatSymbols();
    my $dfs5 = new ICU::DateFormatSymbols($dfs4);

    return($result);
}

sub check_setDebugLevel {
    my $result = 1;
    my $old_debug_level1 = ICU::DateFormatSymbols::setDebugLevel();
    my $old_debug_level2 = ICU::DateFormatSymbols::setDebugLevel();
    if ($old_debug_level1 != $old_debug_level2) {
        $result = 0;
    }
    my $val1 = ICU::DateFormatSymbols::setDebugLevel(3);
    if ($val1 != $old_debug_level2) {
        $result = 0;
    }
    my $val2 = ICU::DateFormatSymbols::setDebugLevel();
    if ($val2 != 3) {
        $result = 0;
    }
    ICU::DateFormatSymbols::setDebugLevel($old_debug_level2);

    return($result);
}

