# (c) Copyright 2009-2015. CodeWeavers, Inc.

import sys

if sys.version_info < (3,):
    import urllib
    from urllib import urlencode # pylint: disable=E0401,E0611
else:
    import urllib.request as urllib # pylint: disable=E0611,E0401
    from urllib.parse import urlencode # pylint: disable=E0401,E0611

import cxutils

USER_AGENT = "Mozilla/5.0 (like Gecko) CrossOver"

_WEEKDAY_NAMES = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"]
_MONTH_NAMES = ["Jan", "Feb", "Mar", "Apr", "May", "Jun",
                "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]

def format_http_date(date_time):
    """Converts a datetime object to an RFC-1123-compliant string for
    use in HTTP headers. Note that this does not rely on strftime() to avoid
    locale issues.
    """
    return "%s, %02d %s %04d %02d:%02d:%02d UTC" % (
        _WEEKDAY_NAMES[date_time.weekday()],
        date_time.day,
        _MONTH_NAMES[date_time.month - 1],
        date_time.year,
        date_time.hour,
        date_time.minute,
        date_time.second
        )

def url_to_basename(url):
    url = url.split('#', 1)[0].split('?', 1)[0]
    components = url.rsplit('/', 4)
    if components[-1] == 'download' and len(components) > 4 and \
            '.' in components[-2]:
        # URLs of the form '.../foo.exe/download' are quite common
        basename = components[-2]
    else:
        basename = components[-1]
    return urllib.url2pathname(basename)

def url_to_sanitized_basename(url):
    return cxutils.sanitize_windows_filename(url_to_basename(url))

class StopDownload(Exception):
    """Stops the download if raised from a notify_headers or notify_progress
    handler.
    """

class UrlError(Exception):
    """Base class for errors raised by UrlGetter."""

    reason = ''

    def __init__(self, reason):
        Exception.__init__(self)
        self.reason = reason

    def __str__(self):
        return str(self.reason)

    def __repr__(self):
        return 'cxurlget.UrlError(%s)' % repr(self.reason)

class HttpError(UrlError):
    """Exception class for error responses from an HTTP server."""

    code = 0

    def __init__(self, code, reason):
        UrlError.__init__(self, reason)
        self.code = code

    def __str__(self):
        return '%s: %s' % (self.code, self.reason)

    def __repr__(self):
        return 'cxurlget.HttpError(%s, %s)' % (repr(self.code), repr(self.reason))

class UrlGetterBase(object):
    bytes_downloaded = 0
    bytes_total = -1
    last_modified = None
    etag = None
    try_count = 0
    finished = False
    url = ''
    basename = ''
    outfile = None
    infile = None
    notify_progress = None
    notify_finished = None
    notify_failed = None
    content_encoding = None
    decoding_event = None
    decoding_exception = None

    blocksize = 4096

    def _do_nothing(self, *args):
        pass

    def __init__(self, url, outfile, notify_progress=_do_nothing, notify_finished=_do_nothing, notify_failed=_do_nothing, notify_headers=_do_nothing, user_agent=USER_AGENT, last_modified=None, etag=None, verb=None, data=None, timeout=None):
        # Strip the anchor; it's of no concern to the web server
        self.url = url.split('#', 1)[0]
        self.user_agent = user_agent
        self.last_modified = last_modified
        self.etag = etag
        self.basename = url_to_sanitized_basename(url)
        if hasattr(outfile, 'write'):
            self.outfile = outfile
        else:
            self.outfile = open(outfile, 'wb')
        self.notify_headers = notify_headers
        self.notify_progress = notify_progress
        self.notify_finished = notify_finished
        self.notify_failed = notify_failed
        self.timeout = timeout
        self.userdata = {}
        if verb is None:
            verb = 'GET' if data is None else 'POST'
        self.verb = verb
        if isinstance(data, dict):
            data = urlencode(data).encode('ascii')
        self.data = data

    def fetch(self):
        raise NotImplementedError()

    def flush(self):
        """Ensures the on-disk file contains an up-to-date view of the download
        in progress. This method must not be called after the download has
        completed."""
        self.outfile.flush()
