#!/usr/bin/perl -w
use strict;

# Copyright (C) 2022-2025 Andrea Monaco
# 
# This file is part of alisp, a lisp implementation.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#


# Test script



use IPC::Open2;



my $total_tests = 0;

my $passed_tests = 0;

my $failed_tests = 0;


my $pid = open2 (my $al_out, my $al_in, './al -q');



for ((1..5))
{
    if (eof ($al_out))
    {
	print "EOF reached from al, it probably crashed\n";

	exit;
    }

    <$al_out>;
}



# reader tests

make_test ("`'`\"\"", "'`\"\"");
make_test ("'(((\n" .
	   ")))", "((NIL))");
make_test ("'(\"\"`\"\")", "(\"\" `\"\")");
make_test ("'(\n" .
	   "(\n" .
	   ";\n" .
	   ")\n" .
	   ")", "(NIL)");
make_test ("\"\"", "\"\"");
make_test ("\"\" \"\" \"\"", "\"\"\n\"\"\n\"\"");
make_test ("` ( ; asd\n" .
           " \"\")", "(\"\")");
make_test ("`';\na", "'A");
make_test ("#|\n" .
	   "|# \"\"", "\"\"");
make_test ("#|\n" .
	   "|# #|\n" .
	   "|# \"\"", "\"\"");
make_test (" #| #|\n" .
	   "|##|\n" .
	   "|#\n" .
	   "  |# \"\"", "\"\"");
make_test ("'\n(\n)", "NIL");
make_test ("'(1 . 2)", "(1 . 2)");
make_test ("'(1 . (2 3))", "(1 2 3)");
make_test ("'(1 . 2\n" .
	   ")", "(1 . 2)");
make_test ("'(1 .\n" .
	   "2)", "(1 . 2)");
make_test ("'(1\n" .
	   ". 2)", "(1 . 2)");
make_test ("'(1 .\n(2)\n)", "(1 2)");
make_test ("'(1 . (\n" .
	   "2))", "(1 2)");
make_test ("'( \"\" #||# )", "(\"\")");
make_test ("'||", "||");
make_test ("'asd\\f|gh|j", "|ASDfghJ|");
make_test ("'\\\\", "\\\\");
make_test ("'\\123", "|123|");
make_test ("'123\\ ", "|123 |");
make_test ("'\\.", "|.|");
make_test ("'\\..", "|..|");
make_test ("'\\\n ", "|\n|");
make_test ("'a\\\n ", "|A\n|");
make_test ("'|aaa\nbb|c", "|aaa\nbbC|");
make_test ("'a\\\n..", "|A\n..|");
make_test ("'|aaa\n:bbb|", "|aaa\n:bbb|");
make_test (":\\asd\\\\f", ":|aSD\\\\F|");
make_test ("'cl:car", "CAR");
make_test ("'cl-user::car", "CAR");
make_test ("(car ''0)", "QUOTE");
make_test ("(car '#'car)", "FUNCTION");
make_test ("'#'car", "#'CAR");
make_test ("`#'(lambda nil ,(+))", "(FUNCTION (LAMBDA NIL 0))");


# now we extensively test backtick notation.  maybe some of these
# cases are not specified by the standard or are undefined behavior,
# so I chose the most consistent behavior to me

make_test ("`,'a", "A");
make_test ("``,a", "`,A");
make_test ("`,`a", "A");
make_test ("`(1 . 2)", "(1 . 2)");
make_test ("``(a ,,(+ 1 2))", "`(A ,3)");
make_test ("``(a ,,(+ 1 2) ,(+ 3 4))", "`(A ,3 ,(+ 3 4))");
make_test ("``(a ,(+ ,1 2))", "`(A ,(+ 1 2))");
make_test ("`(,1 . ,2)", "(1 . 2)");
make_test ("`(,1 ,2 . ,3)", "(1 2 . 3)");
make_test ("`(0 ,1 2 ,3)", "(0 1 2 3)");
#make_test ("`',(car ())", "'NIL");
make_test ("`',(car ())", "(QUOTE NIL)");
make_test ("`(if ``(progn ,,1))", "(IF ``(PROGN ,,1))");
make_test ("'@", "@");
make_test ("`(,@())", "NIL");
make_test ("`(,.())", "NIL");
make_test ("`(,@() ,.())", "NIL");
make_test ("`(1 2 ,@() ,.() 3)", "(1 2 3)");
make_test ("`(,\@nil 1)", "(1)");
make_test ("`(,\@nil ,1)", "(1)");
make_test ("``(1 2 ,,@() ,,.())", "`(1 2)");
make_test ("``(,@())", "`(,\@NIL)");
make_test ("`(\n,@(\n))", "NIL");
make_test ("`(,\@5)", "5");
make_test ("``(,,\@0)", "`,0");
make_test ("`(1 ,\@5)", "(1 . 5)");
make_test ("`(,\@'(3 4))", "(3 4)");
make_test ("`(1 2 ,\@'(3 . 4))", "(1 2 3 . 4)");
make_test ("'(`(\n,@(2)))", "(`(,@(2)))");
make_test ("'(0 . `(\n,@(2)))", "(0 . `(,@(2)))");
make_test ("``(,,@(list 0 1 2))", "`(,0 ,1 ,2)");
make_test ("``(,,.(list 0 1 2))", "`(,0 ,1 ,2)");
make_test ("`(,@``(3 4))", "`(3 4)");
make_test ("`(,1 ,@(cdr '(1 2 3 4)))", "(1 2 3 4)");
make_test ("`(,1 ,@(cdr '(1 2 3 4)) ,5)", "(1 2 3 4 5)");
make_test ("`(,@(cdr '(0 1 2 3 4)) 5)", "(1 2 3 4 5)");
make_test ("`(,@(cdr '(0 1 2 3)) ,4)", "(1 2 3 4)");
make_test ("`(,.(cdr '(0 1 2 3)) ,4)", "(1 2 3 4)");
make_test ("``(,,@(cdr '(0 1 2 3)) ,4)", "`(,1 ,2 ,3 ,4)");
make_test ("``(,,.(cdr '(0 1 2 3)) ,4)", "`(,1 ,2 ,3 ,4)");


# other reader tests

make_test ("#\\a", "#\\a");
make_test ("#\\κ", "#\\κ");
make_test ("#\\,", "#\\,");
make_test ("#\\.", "#\\.");
make_test ("#\\\\", "#\\\\");
make_test ("#\\|", "#\\|");
make_test ("#\\newLINE", "#\\Newline");
make_test ("#\\\n ", "#\\Newline");
make_test ("#\\Newline", "#\\Newline");
make_test ("#.\"\"", "\"\"");
make_test ("#. (+ 1 2)", "3");
make_test ("(defparameter zz 0)", "ZZ");
make_test ("(let nil #.zz)", "0");
make_test ("(defparameter zz 1)", "ZZ");
make_test ("#p\".\"", "#P\".\"");
make_test ("#(a b c)", "#(A B C)");
make_test ("#(1 2 3)", "#(1 2 3)");
make_test ("#(1 2\n" .
	   "3)", "#(1 2 3)");
make_test ("#(\n" .
	   ")", "#()");
make_test ("#(\n" .
	   "\n" .
	   ")", "#()");
make_test ("#(\n" .
	   "0\n" .
	   ")", "#(0)");
make_test ("#(\n" .
	   "(\n" .
	   ")\n" .
	   ")", "#(NIL)");
make_test ("#5(a b c)", "#(A B C C C)");
make_test ("#0()", "#()");
make_test ("#2a((a b c) (d e f))", "#2A((A B C) (D E F))");
make_test ("#3A(((a b c) (d e f)) ((g h i) (j k l)))", "#3A(((A B C) (D E F)) ((G H I) (J K L)))");
make_test ("#1a(0 1 2)", "#(0 1 2)");
make_test ("#2a (#(a b c) \"def\" #*100 (g h i))", "#2A((A B C) (#\\d #\\e #\\f) (1 0 0) (G H I))");
make_test ("#2a #(#(a b c) \"def\" #*100 (g h i))", "#2A((A B C) (#\\d #\\e #\\f) (1 0 0) (G H I))");
make_test ("`#(,(+))", "#(0)");
make_test ("`#(0 ,@(list 1 2 3) 4)", "#(0 1 2 3 4)");
make_test ("`#(0 ,@(list 1 2 3) ,\@nil ,@(list 4) 5)", "#(0 1 2 3 4 5)");
make_test ("'#:a", "#:A");
make_test ("'#:car", "#:CAR");
make_test ("#b01010", "10");
make_test ("#xf4/f", "244/15");
make_test ("#o10", "8");
make_test ("#29rh", "17");
make_test ("(setq *read-base* 16)", "16");
make_test ("(+ a f)", "25");
make_test ("(setq *read-base* 8)", "8");
make_test ("10", "8");
make_test ("10/12", "4/5");
make_test ("(typep '9 'symbol)", "T");
make_test ("(setq *read-base* 12)", "10");
make_test ("(+ #1=1 #2=2 #2#)", "5");
make_test ("(nth 5 '(0 . #1=(1 2 . #1#)))", "1");


# eval tests

make_test ("nil", "NIL");
make_test ("NIL", "NIL");
make_test ("t", "T");
make_test ("(progn)", "NIL");
make_test ("(progn 1 2 3)", "3");
make_test ("(progn (values 1 2 3))", "1\n2\n3");
make_test ("(values (+ 1 0) 2 3)", "1\n2\n3");
make_test ("(values-list '(1 2 3))", "1\n2\n3");
make_test ("(multiple-value-list (values))", "NIL");
make_test ("(multiple-value-list (values 1 2 3))", "(1 2 3)");
make_test ("(multiple-value-call (lambda ()))", "NIL");
make_test ("(multiple-value-call #'+ (floor 5 2) (values) 1)", "4");
make_test ("(multiple-value-call (lambda (x) x) 'i)", "I");
make_test ("(multiple-value-call 'list 1 2 3)", "(1 2 3)");
make_test ("(if nil 1)", "NIL");
make_test ("(common-lisp:if '(1) 2)", "2");
make_test ("(cl:eq 1 1)", "NIL");
make_test ("(eq 'a 'a)", "T");
make_test ("(eql 'a 'a)", "T");
make_test ("(eql 'a 'b)", "NIL");
make_test ("(eql 0 0)", "T");
make_test ("(eql 0 0.0)", "NIL");
make_test ("(eql 2 2/1)", "T");
make_test ("(eql 2.0 2.0)", "T");
make_test ("(eql #\\a #\\a)", "T");
make_test ("(eql #\\A #\\a)", "NIL");
make_test ("(null 1)", "NIL");
make_test ("(not nil)", "T");
make_test ("(cons 1 2)", "(1 . 2)");
make_test ("(list)", "NIL");
make_test ("(list 1 2 3)", "(1 2 3)");
make_test ("(list* 0)", "0");
make_test ("(list* 0 1 2)", "(0 1 . 2)");
make_test ("(append ())", "NIL");
make_test ("(append 0)", "0");
make_test ("(append nil nil 0)", "0");
make_test ("(append '(1 2) '(3 4))", "(1 2 3 4)");
make_test ("(append '(1) (values))", "(1)");
make_test ("(append '(0) '(1) '(2))", "(0 1 2)");
make_test ("(nconc)", "NIL");
make_test ("(nconc 0)", "0");
make_test ("(nconc (list 'a 'b 'c) (list 'd 'e))", "(A B C D E)");
make_test ("(nconc (list 'a 'b 'c) 'd)", "(A B C . D)");
make_test ("(nconc (list 'a 'b 'c) NIL (list 'd 'e) NIL 'f)", "(A B C D E . F)");
make_test ("(nconc nil '(0))", "(0)");
make_test ("(nconc '(0 1 . 2) '(3))", "(0 1 3)");
make_test ("(let ((a 2)) a)", "2");
make_test ("(let ((x 0)) x x x)", "0");
make_test ("(let ((a)) a)", "NIL");
make_test ("(defparameter var20 0)", "VAR20");
make_test ("(let ((var20 0) (j (setq var20 1))))", "NIL");
make_test ("(let () (declare (ignore a b (function c)) (ignorable x (function y))) (declare (ignore)) (+ 1 2))", "3");
make_test ("(let () (declare (inline a b (setf c)) (notinline x (setf y))) (+ 1 2))", "3");
make_test ("(let () (declare (optimize safety speed (compilation-speed 0) (debug 3))) (+ 1 2))", "3");
make_test ("(let ((x 0)) (declare (special x)) x)", "0");
make_test ("(let ((x 0)) (declare (special x)) (let ((x 1)) x))", "1");
make_test ("(let ((x 0)) (declare (special x)) (let ((x 1)) (eval 'x)))", "0");
make_test ("(let ((x 0)) (let ((x 1)) (declare (special x)) (let () x)))", "1");
make_test ("(let ((x 0)) (declare (special x)) (let ((x 1)) (declare (special x))) x)", "0");
make_test ("(let ((var2 0)) (let ((var2 1)) (declare (special var2)) (let ((var2 1))) var2))", "1");
make_test ("(let ((vars 0)) (declare (special vars1 vars)) vars)", "0");
make_test ("(defparameter var3 0)", "VAR3");
make_test ("(defun f (x &optional (var3 var3)))", "F");
make_test ("(f 9)", "NIL");
make_test ("(setq var3 1)", "1");
make_test ("(defun func () x)", "FUNC");
make_test ("(let ((x 10)) (declare (special x)) (func))", "10");
make_test ("(let ((x 12)) (declare (special x)) (let ((x 13)) (write (func)) (write x)))", "1213\n13");
make_test ("(let* ((x 0)) (declare (special x)) x)", "0");
make_test ("(let* ((x 0)) (declare (special x)) (let ((x 1)) x))", "1");
make_test ("(let* ((x 0)) (declare (special x)) (let ((x 1)) (eval 'x)))", "0");
make_test ("(let* ((a)) a)", "NIL");
make_test ("(let ((var10 0)) (declare (special var10)) (let ((var10 1)) (locally (declare (special var10)) var10)))", "0");
make_test ("(let ((var10 0)) (declare (special var10)) (let ((var10 1)) (locally (declare (special var10)) (write var10)) var10))", "0\n1");
make_test ("(defun fun1 () x)", "FUN1");
make_test ("(defun fun2 (x) (declare (ignorable x)) \"docstring\" (declare (special x)) (fun1))", "FUN2");
make_test ("(fun2 10)", "10");
make_test ("(defun fun3 nil \"not a docstring\")", "FUN3");
make_test ("(fun3)", "\"not a docstring\"");
make_test ("(defun fooo () (list x y z))", "FOOO");
make_test ("(progv '(x y z) '(0 1 2) (fooo))", "(0 1 2)");
make_test ("(flet ((a () (write \"\"))) (a))", "\"\"\n\"\"");
make_test ("(let ((i 0)) (flet ((fun () (setq i 1))) (fun)) i)", "1");
make_test ("(flet (((setf foo) (x y z) (list x y z))) (setf (foo 0 1) 2))", "(2 0 1)");
make_test ("(flet ((foo (x y) 10) ((setf foo) (x y) 11)) (foo 0 1))", "10");
make_test ("(labels ((a NIL (write \"\"))) (a))", "\"\"\n\"\"");
make_test ("(labels ((a () (b)) (b () (write \"\"))) (a))", "\"\"\n\"\"");
make_test ("(labels (((setf foo) (x y z) (list x y z))) (setf (foo 0 1) 2))", "(2 0 1)");
make_test ("(let ((i 0)) (labels ((fun () (setq i 1))) (fun)) i)", "1");
make_test ("(defun funfoo (x) x)", "FUNFOO");
make_test ("(labels ((funfoo () 0)) (funfoo))", "0");
make_test ("(macrolet ((w (a) `(write ,a))) (w \"\"))", "\"\"\n\"\"");
make_test ("(eval-when (:compile-toplevel compile) 0)", "NIL");
make_test ("(eval-when (:load-toplevel load) 0)", "NIL");
make_test ("(eval-when (:execute eval load compile) 0)", "0");
make_test ("(defconstant a 8)", "A");
make_test ("(defconstant a 8)", "A");
make_test ("a", "8");
make_test ("(constantp 0)", "T");
make_test ("(constantp #\\a)", "T");
make_test ("(constantp #(0))", "T");
make_test ("(constantp \"\")", "T");
make_test ("(constantp 'a)", "T");
make_test ("(constantp a)", "T");
make_test ("(constantp :a)", "T");
make_test ("(constantp ''a)", "T");
make_test ("(defparameter b 9)", "B");
make_test ("(defvar b (0))", "B");
make_test ("b", "9");
make_test ("(defvar lol 0 \"\")", "LOL");
make_test ("lol", "0");
make_test ("(setf b 10)", "10");
make_test ("(setf b 9)", "9");
make_test ("(setf c 4)", "4");
make_test ("c", "4");
make_test ("(setf)", "NIL");
make_test ("(let ((c 0)) c)", "0");
make_test ("(setf c (+ c 1))", "5");
make_test ("c", "5");
make_test ("(let ((l '(0 1))) (psetf (car l) 2 (cdr l) (car l)) l)", "(2 . 0)");
make_test ("(setq)", "NIL");
make_test ("(setq b 9 c 4 c (+ c 1))", "5");
make_test ("(let ((f 0)) (setq g 1) f)", "0");
make_test ("(psetq)", "NIL");
make_test ("(psetq xx 0)", "NIL");
make_test ("(psetq xx 1 yy xx)", "NIL");
make_test ("xx", "1");
make_test ("yy", "0");
make_test ("(setf tt (list 'a 'b 'd))", "(A B D)");
make_test ("(setf (car tt) 'f)", "F");
make_test ("tt", "(F B D)");
make_test ("(setf (cdr tt) 'g)", "G");
make_test ("tt", "(F . G)");
make_test ("(let ((l '(1 2))) (setf (nth 1 l) 3) l)", "(1 3)");
make_test ("(let ((l '(1 2 3))) (setf (nth 2 l) 4) l)", "(1 2 4)");
make_test ("(defparameter str \"abcdef\")", "STR");
make_test ("(setf (aref str 2) #\\z)", "#\\z");
make_test ("str", "\"abzdef\"");
make_test ("(defparameter arr (make-array 3))", "ARR");
make_test ("(setf (aref arr 2) \"aaa\")", "\"aaa\"");
make_test ("arr", "#(NIL NIL \"aaa\")");
make_test ("(let ((arr (make-array '(1 2 3)))) (setf (aref arr 0 1 2) 10) arr)", "#3A(((NIL NIL NIL) (NIL NIL 10)))");
make_test ("(let ((v #*100010)) (setf (aref v 2) 1) v)", "#*101010");
make_test ("(let ((b 10)) b)", "10");
make_test ("(let ((b 1)) (setf b 2) b)", "2");
make_test ("b", "9");
make_test ("(let ((s \"aaa\")) (list (setf (elt s 1) #\\b) s))", "(#\\b \"aba\")");
make_test ("(let ((s \"abcdefg\")) (setf (elt s 2) #\\ò) s)", "\"abòdefg\"");
make_test ("(let ((s \"abòdefg\")) (setf (elt s 2) #\\c) s)", "\"abcdefg\"");
make_test ("(let ((s \"abc\")) (setf (elt s 0) #\\ò) s)", "\"òbc\"");
make_test ("(let ((s \"òbc\")) (setf (elt s 0) #\\a) s)", "\"abc\"");
make_test ("(let ((s \"abc\")) (setf (elt s 2) #\\ò) s)", "\"abò\"");
make_test ("(let ((s \"abò\")) (setf (elt s 2) #\\c) s)", "\"abc\"");
make_test ("(let ((l (list 'a 'b 'c))) (list (setf (elt l 2) 'd) l))", "(D (A B D))");
make_test ("(let ((arr (make-array 3))) (setf (elt arr 1) 10) arr)", "#(NIL 10 NIL)");
make_test ("(let ((v #*100010)) (setf (elt v 2) 1) v)", "#*101010");
make_test ("(let* ((x 1) (y x)) y)", "1");
make_test ("((lambda (x) (* 2 x)) (*))", "2");
make_test ("(defun f ())", "F");
make_test ("#'f", "#<FUNCTION F>");
make_test ("#'car", "#<FUNCTION BUILTIN CAR>");
make_test ("#'(lambda nil nil)", "#<FUNCTION ?>");
make_test ("#'(setf car)", "#<FUNCTION BUILTIN (SETF CAR)>");
make_test ("(flet ((foo ())) #'foo)", "#<FUNCTION FOO>");
make_test ("(function f)", "#<FUNCTION F>");
make_test ("(flet ((foo ())) (function foo))", "#<FUNCTION FOO>");
make_test ("(function car)", "#<FUNCTION BUILTIN CAR>");
make_test ("(function (lambda nil nil))", "#<FUNCTION ?>");
make_test ("(apply #'+ 1 2 ())", "3");
make_test ("(apply #'+ '(1 2))", "3");
make_test ("(apply #'car '(1 2) ())", "1");
make_test ("(apply 'car '(1 2) ())", "1");
make_test ("(apply (lambda (x) (car x)) '(1 2) ())", "1");
make_test ("(let ((s 0)) (funcall (lambda (s) s) 1))", "1");
make_test ("(funcall #'+ 1 2 3)", "6");
make_test ("(funcall 'car '(1 2 3))", "1");
make_test ("(f)", "NIL");
make_test ("(defun f () \"\")", "F");
make_test ("(f)", "\"\"");
make_test ("(defun aaa () aaa aaa)", "AAA");
make_test ("(defun f (x) \"\")", "F");
make_test ("(f 0)", "\"\"");
make_test ("(defun f () (values 1 2))", "F");
make_test ("(f)", "1\n2");
make_test ("(defun fun (x y) y)", "FUN");
make_test ("(fun 1 2)", "2");
make_test ("(defun fun (b) b)", "FUN");
make_test ("(fun 3)", "3");
make_test ("b", "9");
make_test ("(defun g (x &optional y) y)", "G");
make_test ("(g 1)", "NIL");
make_test ("(g 1 2)", "2");
make_test ("(defun g2 (&optional (x \"a\")) x)", "G2");
make_test ("(g2)", "\"a\"");
make_test ("(g2 4)", "4");
make_test ("(defun g5 (&optional x (y x)) y)", "G5");
make_test ("(g5 0)", "0");
make_test ("(defun fun4 (x &optional (y 0) (z y)) z)", "FUN4");
make_test ("(fun4 20)", "0");
make_test ("(let ((i 0)) (defun fun5 (&optional (v i))))", "FUN5");
make_test ("(fun5)", "NIL");
make_test ("(defun g3 (&optional (x \"a\" provided)) (list x provided))", "G3");
make_test ("(g3)", "(\"a\" NIL)");
make_test ("(g3 7)", "(7 T)");
make_test ("(defun g4 (x &optional y &rest z) z)", "G4");
make_test ("(g4 1 2 3 (+ 3 1) 5)", "(3 4 5)");
make_test ("(defun fun (&rest x) x)", "FUN");
make_test ("(fun)", "NIL");
make_test ("(defun fun (x &rest y) y)", "FUN");
make_test ("(fun 5 6 7 8)", "(6 7 8)");
make_test ("(defun fun2 (&rest r &key x y z) y)", "FUN2");
make_test ("(fun2 :y 0)", "0");
make_test ("(fun2 :y 0 :y 1)", "0");
make_test ("(fun2 :x 0)", "NIL");
make_test ("(fun2 :z 'tt :y (+ 1 2))", "3");
make_test ("(fun2)", "NIL");
make_test ("(defun fun2 (&key ((:k aa) 1 supp)) (list aa supp))", "FUN2");
make_test ("(fun2)", "(1 NIL)");
make_test ("(fun2 :k 1)", "(1 T)");
make_test ("(defun fun3 (x &key (y x)) y)", "FUN3");
make_test ("(fun3 10)", "10");
make_test ("(defun fun2 (&rest r &key x y z) r)", "FUN2");
make_test ("(fun2 :y (+ 1 2) :z 'tt)", "(:Y 3 :Z TT)");
make_test ("(fun2)", "NIL");
make_test ("(defun fun (x &rest y &key z) y)", "FUN");
make_test ("(fun 1 :z 0)", "(:Z 0)");
make_test ("(defun fun (x &optional y &rest z &key k) z)", "FUN");
make_test ("(fun 0 1 :k 2)", "(:K 2)");
make_test ("(defun f (&key &allow-other-keys))", "F");
make_test ("(f :a 0)", "NIL");
make_test ("(defun f (&key x &allow-other-keys) x)", "F");
make_test ("(f :x 0 :y 1)", "0");
make_test ("(f)", "NIL");
make_test ("(defun f2 (&key x))", "F2");
make_test ("(f2 :x 0 :allow-other-keys t)", "NIL");
make_test ("(defun f2 (&key x allow-other-keys) allow-other-keys)", "F2");
make_test ("(f2)", "NIL");
make_test ("(f2 :x 0 :y 0 :allow-other-keys t)", "T");
make_test ("(defun f4 (&key))", "F4");
make_test ("(f4 :a 0 :allow-other-keys t)", "NIL");
make_test ("(defun f (x y &aux (z (+ x y)) w) (list z w))", "F");
make_test ("(f 1 2)", "(3 NIL)");
make_test ("(defun f3 (x &key y &aux (z x)) z)", "F3");
make_test ("(f3 0)", "0");
make_test ("(defun f4 (x &aux y (z x)) z)", "F4");
make_test ("(f4 0)", "0");
make_test ("(let ((x 0)) (defun inc () (setf x (+ x 1))))", "INC");
make_test ("(inc)", "1");
make_test ("(inc)", "2");
make_test ("(inc)", "3");
make_test ("(let ((hh 0)) (defun f nil hh))", "F");
make_test ("(f)", "0");
make_test ("(defparameter hh 1)", "HH");
make_test ("(f)", "0");
make_test ("hh", "1");
make_test ("(defmacro f () (f))", "F");
make_test ("(defmacro f () (f))", "F");
make_test ("(let ((var 1)) (defmacro mac () var))", "MAC");
make_test ("(mac)", "1");
make_test ("(defparameter foo 0)", "FOO");
make_test ("(defmacro mac () foo)", "MAC");
make_test ("(mac)", "0");
make_test ("(let ((foo 1)) (mac))", "0");
make_test ("(defmacro mac (x (y (z))) `(list ,x ,y ,z))", "MAC");
make_test ("(mac 0 (1 (2)))", "(0 1 2)");
make_test ("(defmacro mac (()) 0)", "MAC");
make_test ("(mac nil)", "0");
make_test ("(defmacro mac (()) 0)", "MAC");
make_test ("(defmacro mac (x . z) `(list ,x ,z))", "MAC");
make_test ("(macroexpand-1 '(mac 0 1 2))", "(LIST 0 (1 2))\nT");
make_test ("(defmacro mac (&rest (x y)) `(list ,x ,y))", "MAC");
make_test ("(macroexpand-1 '(mac 0 1))", "(LIST 0 1)\nT");
make_test ("(defmacro mac (&rest nil))", "MAC");
make_test ("(mac)", "NIL");
make_test ("(defmacro mac (&whole wh f g h) `(list ,wh ,f ,g ,h))", "MAC");
make_test ("(macroexpand-1 '(mac x y z))", "(LIST (MAC X Y Z) X Y Z)\nT");
make_test ("(defmacro mac (z (&whole wh f g)) `(list ,wh ,f ,g))", "MAC");
make_test ("(macroexpand-1 '(mac x (y z)))", "(LIST (Y Z) Y Z)\nT");
make_test ("(defmacro mac (&whole wh f g h) `(list ',wh ',f ',g ',h))", "MAC");
make_test ("(mac 0 1 2)", "((MAC 0 1 2) 0 1 2)");
make_test ("(defmacro mac2 (&whole wh &rest args) wh)", "MAC2");
make_test ("(macroexpand-1 '(mac2 0))", "(MAC2 0)\nT");

make_test ("(gensym)", "#:G1");
make_test ("(gensym)", "#:G2");
make_test ("(gensym 5)", "#:G5");
make_test ("(gensym \"A\")", "#:A3");
make_test ("(gensym)", "#:G4");
make_test ("(define-setf-expander foo ())", "FOO");
make_test ("(get-setf-expansion '(foo))", "NIL");
make_test ("(get-setf-expansion '(foo2))", "NIL\nNIL\n(#:G5)\n(FUNCALL (FUNCTION (SETF FOO2)) #:G5)\n(FOO2)");
make_test ("(get-setf-expansion 'a)", "NIL\nNIL\n(#:G6)\n(SETQ A #:G6)\nA");
make_test ("(get-setf-expansion '(w a b))", "(#:G7 #:G8)\n(A B)\n(#:G9)\n(FUNCALL (FUNCTION (SETF W)) #:G9 #:G7 #:G8)\n(W #:G7 #:G8)");

make_test ("(load \"cl.lisp\")", "T");
make_test ("(open-stream-p *standard-input*)", "T");
make_test ("(input-stream-p *standard-input*)", "T");
make_test ("(input-stream-p *standard-output*)", "NIL");
make_test ("(output-stream-p *standard-input*)", "NIL");
make_test ("(output-stream-p *standard-output*)", "T");
make_test ("(interactive-stream-p *standard-input*)", "NIL");
make_test ("(upper-case-p #\\A)", "T");
make_test ("(upper-case-p #\\a)", "NIL");
make_test ("(upper-case-p #\\2)", "NIL");
make_test ("(lower-case-p #\\A)", "NIL");
make_test ("(lower-case-p #\\a)", "T");
make_test ("(both-case-p #\\a)", "T");
make_test ("(both-case-p #\\1)", "NIL");
make_test ("(identity \"hi\")", "\"hi\"");
make_test ("(funcall (constantly nil))", "NIL");
make_test ("(funcall (constantly 20) 21 22 23)", "20");
make_test ("(1+ 5)", "6");
make_test ("(1- .5)", "-0.5");
make_test ("(byte 20 100)", "#<BYTE-SPECIFIER size 20 position 100>");
make_test ("(byte-size (byte 20 100))", "20");
make_test ("(byte-position (byte 20 100))", "100");

make_test ("(defmacro mm (x y) `(list ,x ,y))", "MM");
make_test ("(macroexpand-1 '(mm a b))", "(LIST A B)\nT");
make_test ("(macroexpand-1 '(+ 0))", "(+ 0)\nNIL");
make_test ("(defmacro test2 nil (+ 1 2))", "TEST2");
make_test ("(defmacro test nil '(test2))", "TEST");
make_test ("(macroexpand '(test2))", "3\nT");
make_test ("(macroexpand '(test))", "3\nT");
make_test ("(macroexpand '(+ 1 2))", "(+ 1 2)\nNIL");
make_test ("(let ((mf (macro-function 'mm))) (funcall mf '(mm c d) nil))", "(LIST C D)");
make_test ("(macro-function 'foo)", "NIL");
make_test ("(macro-function 'car)", "NIL");
make_test ("(setf (macro-function 'foo) (lambda (form env) `(incf ,(cadr form))))", "#<FUNCTION ?>");
make_test ("(let ((mf (macro-function 'foo))) (funcall mf '(foo var) nil))", "(INCF VAR)");
make_test ("(macroexpand-1 '(foo var))", "(INCF VAR)\nT");
make_test ("(let ((i 0)) (foo i) i)", "1");
make_test ("(defmacro mac (&whole wh) wh)", "MAC");
make_test ("(funcall (macro-function 'mac) nil nil)", "NIL");

make_test ("(defparameter x 0)", "X");
make_test ("(defun fun nil (let ((x 1)) x))", "FUN");
make_test ("(defmacro mac nil (fun))", "MAC");
make_test ("(mac)", "1");

make_test ("(typep '(1 . 2) 'cons)", "T");
make_test ("(typep () 'atom)", "T");
make_test ("(typep (cons 1 2) 'atom)", "NIL");
make_test ("(typep 0 'number)", "T");
make_test ("(typep 0.1 'real)", "T");
make_test ("(typep 0 'rational)", "T");
make_test ("(typep 0.1 'rational)", "NIL");
make_test ("(typep 0 'integer)", "T");
make_test ("(typep 2/1 'integer)", "T");
make_test ("(typep (+ 1/2 1/2) 'integer)", "T");
make_test ("(typep 0 '(integer 0 0))", "T");
make_test ("(typep 0 '(integer 1 2))", "NIL");
make_test ("(typep 0 '(integer 1))", "NIL");
make_test ("(typep 0 '(integer 1 *))", "NIL");
make_test ("(typep 0 'bignum)", "T");
make_test ("(typep 0 'fixnum)", "NIL");
make_test ("(typep 0 'bit)", "T");
make_test ("(typep 1 'bit)", "T");
make_test ("(typep 2 'bit)", "NIL");
make_test ("(typep 0 'signed-byte)", "T");
make_test ("(typep 3 '(signed-byte 3))", "T");
make_test ("(typep 4 '(signed-byte 3))", "NIL");
make_test ("(typep -4 '(signed-byte 3))", "T");
make_test ("(typep -5 '(signed-byte 3))", "NIL");
make_test ("(typep 10 '(unsigned-byte))", "T");
make_test ("(typep -10 '(unsigned-byte))", "NIL");
make_test ("(typep 7 '(unsigned-byte 3))", "T");
make_test ("(typep 8 '(unsigned-byte 3))", "NIL");
make_test ("(typep -1 '(mod 2))", "NIL");
make_test ("(typep 1 '(mod 2))", "T");
make_test ("(typep 2 '(mod 2))", "NIL");
make_test ("(typep 2/1 'ratio)", "NIL");
make_test ("(typep 1/2 'ratio)", "T");
make_test ("(typep 0.1 'float)", "T");
make_test ("(typep 0.1 'single-float)", "T");
make_test ("(typep 0.1 'double-float)", "T");
make_test ("(typep 0.1 'short-float)", "T");
make_test ("(typep 0.1 'long-float)", "T");
make_test ("(typep 1.0 'complex)", "NIL");
make_test ("(typep (complex 1 1) 'complex)", "T");
make_test ("(typep (make-random-state) 'random-state)", "T");
make_test ("(typep #\\a 'character)", "T");
make_test ("(typep #\\ò 'character)", "T");
make_test ("(typep #\\ò 'base-char)", "T");
make_test ("(typep #\\ò 'extended-char)", "NIL");
make_test ("(typep #\\ò 'standard-char)", "NIL");
make_test ("(typep #\\! 'standard-char)", "T");
make_test ("(typep \"abc\" 'string)", "T");
make_test ("(typep \"abc\" '(string))", "T");
make_test ("(typep 0 'string)", "NIL");
make_test ("(typep \"abc\" 'simple-string)", "T");
make_test ("(typep 0 'bit-vector)", "NIL");
make_test ("(typep #*010 'bit-vector)", "T");
make_test ("(typep #*010 'simple-bit-vector)", "T");
make_test ("(typep (make-array '(1 2)) 'vector)", "NIL");
make_test ("(typep \"abc\" 'vector)", "T");
make_test ("(typep \"abc\" 'simple-vector)", "NIL");
make_test ("(typep #(0 1 2) 'simple-vector)", "T");
make_test ("(typep (make-array 3 :fill-pointer t) 'simple-vector)", "NIL");
make_test ("(typep \"abc\" 'array)", "T");
make_test ("(typep \"abc\" 'simple-array)", "T");
make_test ("(typep (make-array 3 :fill-pointer t) 'simple-array)", "NIL");
make_test ("(typep (make-hash-table) 'hash-table)", "T");
make_test ("(typep '(1 2) 'list)", "T");
make_test ("(typep #(1 2) 'sequence)", "T");
make_test ("(typep \"\" t)", "T");
make_test ("(typep 0 'nil)", "NIL");
make_test ("(typep nil nil)", "NIL");
make_test ("(typep nil 'null)", "T");
make_test ("(typep 'aaa 'symbol)", "T");
make_test ("(typep :aaa 'keyword)", "T");
make_test ("(typep 'aaa 'keyword)", "NIL");
make_test ("(typep nil 'boolean)", "T");
make_test ("(typep #p\"\" 'pathname)", "T");
make_test ("(typep 0 'logical-pathname)", "NIL");
make_test ("(typep #p\"\" 'logical-pathname)", "NIL");
make_test ("(typep (open #p\"README\") 'stream)", "T");
make_test ("(typep (open #p\"README\") 'file-stream)", "T");
make_test ("(typep (make-string-input-stream \"hello\") 'string-stream)", "T");
make_test ("(typep 0 '(or integer string))", "T");
make_test ("(typep 0 '(and integer string))", "NIL");
make_test ("(typep 0 '(not string))", "T");
make_test ("(type-of 'a)", "SYMBOL");
make_test ("(type-of \"aaa\")", "STRING");
make_test ("(type-of #\\a)", "CHARACTER");
make_test ("(type-of '(1 . 2))", "CONS");
make_test ("(type-of 0)", "INTEGER");
make_test ("(type-of 1/2)", "RATIO");
make_test ("(type-of 1.0)", "FLOAT");
make_test ("(type-of (complex 0 1))", "COMPLEX");
make_test ("(type-of #'car)", "FUNCTION");
make_test ("(type-of *package*)", "PACKAGE");
make_test ("(type-of #(0 1 2))", "ARRAY");
make_test ("(type-of (make-hash-table))", "HASH-TABLE");
make_test ("(type-of #p\"aaa\")", "PATHNAME");
make_test ("(type-of *standard-output*)", "STREAM");
make_test ("(type-of '`0)", "AL-BACKQUOTE");
make_test ("(al-next (al-next (al-next '`,\@0)))", "0");
make_test ("(subtypep 'integer nil)", "NIL\nT");
make_test ("(subtypep 'integer t)", "T\nT");
make_test ("(subtypep nil 'integer)", "T\nT");
make_test ("(subtypep t 'integer)", "NIL\nT");
make_test ("(subtypep 'integer 'number)", "T\nT");
make_test ("(subtypep 'integer 'cons)", "NIL\nT");
make_test ("(subtypep 'short-float 'long-float)", "T\nT");
make_test ("(subtypep 'long-float 'short-float)", "T\nT");
make_test ("(subtypep 'long-float 'single-float)", "T\nT");
make_test ("(subtypep 'float 'number)", "T\nT");
make_test ("(subtypep 'float 'real)", "T\nT");
make_test ("(subtypep 'character 'float)", "NIL\nT");
make_test ("(subtypep 'character 'integer)", "NIL\nT");
make_test ("(subtypep 'character 'bit)", "NIL\nT");
make_test ("(subtypep 'integer '(or string number))", "T\nT");
make_test ("(subtypep 'integer '(and number float))", "NIL\nT");
make_test ("(subtypep '(and string vector) 'array)", "T\nT");
make_test ("(subtypep '(or rational float) 'number)", "T\nT");
make_test ("(subtypep '(and symbol list) 'null)", "NIL\nNIL");
make_test ("(deftype newinteger nil 'integer)", "NEWINTEGER");
make_test ("(subtypep 'newinteger 'real)", "T\nT");
make_test ("(coerce 4 'number)", "4");
make_test ("(coerce \"abc\" 'list)", "(#\\a #\\b #\\c)");
make_test ("(coerce nil 'vector)", "#()");
make_test ("(coerce '(0 1 2) 'vector)", "#(0 1 2)");
make_test ("(coerce \"a\" 'character)", "#\\a");
make_test ("(coerce 3 'complex)", "3");
make_test ("(coerce 3.0 'complex)", "#C(3.0 0.0)");
make_test ("(coerce 1/2 'float)", "0.5");
make_test ("(coerce 'car 'function)", "#<FUNCTION BUILTIN CAR>");
make_test ("(deftype not-integer () `(not integer))", "NOT-INTEGER");
make_test ("(typep 0 'not-integer)", "NIL");
make_test ("(typep \"\" 'not-integer)", "T");
make_test ("(deftype this-object (obj) `(eql ,obj))", "THIS-OBJECT");
make_test ("(typep 0 '(this-object 0))", "T");
make_test ("(typep 0 '(this-object 1))", "NIL");
make_test ("(deftype foo (&optional s) (write s) 'integer)", "FOO");
make_test ("(typep 0 'foo)", "*\nT");
make_test ("(typep \"a\" '(eql \"a\"))", "NIL");
make_test ("(typep 1 '(member 1 2 3))", "T");
make_test ("(typep 4 '(member 1 2 3))", "NIL");
make_test ("(typep 4 '(member))", "NIL");
make_test ("(deftype odd-integer nil '(and integer (satisfies oddp)))", "ODD-INTEGER");
make_test ("(typep 0 'odd-integer)", "NIL");
make_test ("(typep 1 'odd-integer)", "T");
make_test ("(let ((i 0)) (check-type i integer))", "NIL");
make_test ("(assert (= 1 1))", "NIL");
make_test ("(defun (setf foo2) (a1 a2) (list a1 a2))", "(SETF FOO2)");
make_test ("#'(setf foo2)", "#<FUNCTION (SETF FOO2)>");
make_test ("(fboundp '(setf foo2))", "T");
make_test ("(fboundp '(setf foo3))", "NIL");
make_test ("(let ((x 0) (y 1)) (setf (foo2 x) y))", "(1 0)");
make_test ("(defsetf 2dlist (l x y) (newval) `(setf (elt (elt ,l ,x) ,y) ,newval))", "2DLIST");
make_test ("(let ((ls '((1 2) (3 4))) (x_ 1) (y_ 0)) (list (setf (2dlist ls x_ y_) 10) ls))", "(10 ((1 2) (10 4)))");
make_test ("(define-modify-macro appendf (&rest args) append)", "APPENDF");
make_test ("(let ((l '(0 1 2))) (appendf l '(3) '(4)) l)", "(0 1 2 3 4)");
make_test ("(define-symbol-macro symmac *read-base*)", "SYMMAC");
make_test ("symmac", "10");
make_test ("(macroexpand 'symmac)", "*READ-BASE*\nT");
make_test ("(symbol-macrolet ((smac1 (1+ *read-base*)) (smac2 (+ 2 *read-base*))) smac1)", "11");
make_test ("(let ((cns (cons 0 1))) (symbol-macrolet ((carc (car cns))) (setf carc 10)) cns)", "(10 . 1)");

make_test ("(make-string 3)", "\"\0\0\0\"");
make_test ("(make-string 2 :element-type 'character)", "\"\0\0\"");
make_test ("(make-string 4 :initial-element #\\ò)", "\"òòòò\"");
make_test ("(intern \"hi\")", "|hi|\nNIL");
make_test ("(intern \"hi\")", "|hi|\n:INTERNAL");
make_test ("(intern \"hi\" 'keyword)", ":|hi|\nNIL");
make_test ("(intern \"hi\" 'keyword)", ":|hi|\n:EXTERNAL");
make_test ("(find-symbol \"CAR\")", "CAR\n:INHERITED");
make_test ("(find-symbol \"CAR\" 'cl)", "CAR\n:EXTERNAL");
make_test ("(find-symbol \"car\")", "NIL\nNIL");
make_test ("(find-symbol \"INTERNAL\" 'keyword)", ":INTERNAL\n:EXTERNAL");
make_test ("(unintern :hi)", "NIL");
make_test ("(unintern '|hi|)", "T");
make_test ("(intern \"hi\")", "|hi|\nNIL");
make_test ("(unintern :|hi| 'keyword)", "T");
make_test ("(intern \"hi\" 'keyword)", ":|hi|\nNIL");
make_test ("(make-symbol \"aaa\")", "#:|aaa|");
make_test ("(make-symbol \"\")", "#:||");
make_test ("(symbol-package (copy-symbol 'abc))", "NIL");
make_test ("(fboundp (copy-symbol 'car))", "NIL");
make_test ("(fboundp (copy-symbol 'car t))", "T");
make_test ("(boundp 'b)", "T");
make_test ("(boundp 'g3)", "NIL");
make_test ("(let ((var 0)) (boundp 'var))", "NIL");
make_test ("(let ((var 0)) (declare (special var)) (boundp 'var))", "T");
make_test ("(symbol-value 'b)", "9");
make_test ("(let ((b 10)) (symbol-value 'b))", "10");
make_test ("(let ((lll 10)) (declare (special lll)) (symbol-value 'lll))", "10");
make_test ("(setf (symbol-value 'foo) 0)", "0");
make_test ("foo", "0");
make_test ("(let ((b 11)) (set 'b 12) b)", "12");
make_test ("(fboundp 'b)", "NIL");
make_test ("(fboundp 'g3)", "T");
make_test ("(symbol-function 'fun)", "#<FUNCTION FUN>");
make_test ("(symbol-function 'defparameter)", "#<MACRO BUILTIN DEFPARAMETER>");
make_test ("(symbol-function 'if)", "#<SPECIAL OPERATOR IF>");
make_test ("(flet ((foo nil)) (symbol-function 'car))", "#<FUNCTION BUILTIN CAR>");
make_test ("(defun globfun nil 0)", "GLOBFUN");
make_test ("(setf (symbol-function 'globfun) (lambda nil 1))", "#<FUNCTION ?>");
make_test ("(globfun)", "1");
make_test ("(fdefinition 'fun)", "#<FUNCTION FUN>");
make_test ("(fdefinition 'car)", "#<FUNCTION BUILTIN CAR>");
make_test ("(fdefinition '(setf car))", "#<FUNCTION BUILTIN (SETF CAR)>");
make_test ("(defun (setf globfun) (x) 10)", "(SETF GLOBFUN)");
make_test ("(setf (fdefinition '(setf globfun)) (lambda (x) 11))", "#<FUNCTION ?>");
make_test ("(setf (globfun) 100)", "11");
make_test ("(funcall (complement #'numberp) \"\")", "T");
make_test ("(symbol-name 'aaa)", "\"AAA\"");
make_test ("(symbol-name :bbb)", "\"BBB\"");
make_test ("(symbol-package 'if)", "#<PACKAGE \"COMMON-LISP\">");
make_test ("(symbol-package 'caar)", "#<PACKAGE \"COMMON-LISP\">");
make_test ("(symbol-package 'ggg)", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(symbol-package :eee)", "#<PACKAGE \"KEYWORD\">");
make_test ("(symbol-package '#:aaa)", "NIL");
make_test ("(symbol-package (make-symbol \"aaa\"))", "NIL");
make_test ("(symbol-plist 's)", "NIL");
make_test ("(setf (symbol-plist 's) (list :a 0 :b 1))", "(:A 0 :B 1)");
make_test ("(symbol-plist 's)", "(:A 0 :B 1)");
make_test ("(get 's :b)", "1");
make_test ("(get-properties (symbol-plist 's) '(:b :a))", ":A\n0\n(:A 0 :B 1)");
make_test ("(get-properties (symbol-plist 's) '(:b))", ":B\n1\n(:B 1)");
make_test ("(get-properties (symbol-plist 's) '(:c))", "NIL\nNIL\nNIL");
make_test ("(setf (symbol-plist 'foo) '(:a 0 :b 1 :c 2 :b 3))", "(:A 0 :B 1 :C 2 :B 3)");
make_test ("(remprop 'foo :b)", "T");
make_test ("(symbol-plist 'foo)", "(:A 0 :C 2 :B 3)");
make_test ("(remprop 'foo :d)", "NIL");
make_test ("(getf (symbol-plist 'foo) :c)", "2");
make_test ("(getf (symbol-plist 'foo) :d)", "NIL");
make_test ("(getf (symbol-plist 'foo) :d 10)", "10");
make_test ("(setf (getf (symbol-plist 'foo) :c) 11)", "11");
make_test ("(setf (getf (symbol-plist 'foo) :d) 12)", "12");
make_test ("(symbol-plist 'foo)", "(:A 0 :C 11 :B 3 :D 12)");
make_test ("(special-operator-p 'if)", "T");
make_test ("(special-operator-p 'car)", "NIL");
make_test ("(special-operator-p 'aaa)", "NIL");
make_test ("(let ((varb 0)) (declare (special varb)) (makunbound 'varb) (boundp 'varb))", "NIL");
make_test ("(defparameter ha 0)", "HA");
make_test ("(boundp 'ha)", "T");
make_test ("(makunbound 'ha)", "HA");
make_test ("(boundp 'ha)", "NIL");
make_test ("(defun what ())", "WHAT");
make_test ("(fboundp 'what)", "T");
make_test ("(fmakunbound 'what)", "WHAT");
make_test ("(fboundp 'what)", "NIL");
make_test ("(equal 'a 'a)", "T");
make_test ("(equal #\\a #\\a)", "T");
make_test ("(equal #\\a #\\A)", "NIL");
make_test ("(equal 0 0)", "T");
make_test ("(equal 0 0.0)", "NIL");
make_test ("(equal (cons 1 2) 1)", "NIL");
make_test ("(equal (cons 'a 'b) (cons 'a 'b))", "T");
make_test ("(equal '(1 2 3) '(1 2 3))", "T");
make_test ("(equal '(1 2 . 4) '(1 2 . 3))", "NIL");
make_test ("(equal \"abc\" \"abc\")", "T");
make_test ("(equal \"Abc\" \"abc\")", "NIL");
make_test ("(equal \"abc \" \"abc\")", "NIL");
make_test ("(equal (make-array '(1 2 3)) (make-array '(1 2 3)))", "NIL");
make_test ("(equalp 0 0)", "T");
make_test ("(equalp 0 0.0)", "T");
make_test ("(equalp #\\a #\\A)", "T");
make_test ("(equalp '(1 2 3) '(1 2 3))", "T");
make_test ("(equalp '(1 2 . 4) '(1 2 . 3))", "NIL");
make_test ("(equalp \"Abc\" \"abc\")", "T");
make_test ("(equalp \"abc \" \"abc\")", "NIL");
make_test ("(equalp #(1 2 3) #(1 2 3))", "T");
make_test ("(equalp #(1 2 3) #(1 2 3 0))", "NIL");
make_test ("(equalp (make-array '(1 2 3)) (make-array '(1 2 3)))", "T");
make_test ("(equalp (make-array '(1 2 3)) (make-array '(1 2 4)))", "NIL");
make_test ("(string \"ccc\")", "\"ccc\"");
make_test ("(string 'ddd)", "\"DDD\"");
make_test ("(string #\\F)", "\"F\"");
make_test ("(string= \"a\" \"a \")", "NIL");
make_test ("(string= \"aaa\" \"aaa\")", "T");
make_test ("(string= \"aaa\" \"aab\")", "NIL");
make_test ("(string= \"ABC\" 'abc)", "T");
make_test ("(string= \"ABC\" 'abcd)", "NIL");
make_test ("(string= \"a\" #\\a)", "T");
make_test ("(string/= \"aaa\" \"aaa\")", "NIL");
make_test ("(string/= \"aaa\" \"aab\")", "2");
make_test ("(string/= \"aaa\" \"aaab\")", "3");
make_test ("(string/= \"abcdef\" \"bcdefg\" :start1 2 :end1 4 :start2 1 :end2 3)", "NIL");
make_test ("(string/= \"abcgef\" \"bcdefg\" :start1 2 :end1 4 :start2 1 :end2 3)", "3");
make_test ("(string< \"0123\" \"0234\" :start2 1)", "0");
make_test ("(string< \"0123\" \"123\" :start1 1)", "NIL");
make_test ("(string< \"0123\" \"1234\" :start1 1)", "4");
make_test ("(string< \"3456\" \"012345\" :start1 1 :start2 1 :end2 5)", "NIL");
make_test ("(string< \"1234\" \"123\")", "NIL");
make_test ("(string<= \"123\" \"1234\")", "3");
make_test ("(string<= \"123\" \"124\")", "2");
make_test ("(string<= \"123\" \"123\")", "3");
make_test ("(string<= \"124\" \"123\")", "NIL");
make_test ("(string> \"012345\" \"3456\" :start1 1 :end1 5 :start2 1)", "NIL");
make_test ("(string> \"123\" \"122\")", "2");
make_test ("(string> \"123 \" \"123\")", "3");
make_test ("(string>= \"123 \" \"123\")", "3");
make_test ("(string>= \"123 \" \"123  \")", "NIL");
make_test ("(string-equal \"abc\" \"abC\")", "T");
make_test ("(string-equal \"abc\" \"abc \")", "NIL");
make_test ("(string-not-equal \"abc\" \"aBd\")", "2");
make_test ("(string-lessp \"abc\" \"aBd\")", "2");
make_test ("(string-not-greaterp \"abc\" \"aBc\")", "3");
make_test ("(string-greaterp \"abc\" \"aBd\")", "NIL");
make_test ("(string-not-lessp \"abc\" \"aBd\")", "NIL");
make_test ("(char= #\\a)", "T");
make_test ("(char= #\\a #\\b)", "NIL");
make_test ("(char= #\\a #\\a #\\a)", "T");
make_test ("(char= #\\a #\\a #\\b)", "NIL");
make_test ("(char= #\\a #\\A #\\a)", "NIL");
make_test ("(char/= #\\a #\\A #\\a)", "NIL");
make_test ("(char/= #\\a #\\A #\\b)", "T");
make_test ("(char< #\\0)", "T");
make_test ("(char< #\\0 #\\1 #\\2 #\\3)", "T");
make_test ("(char< #\\0 #\\1 #\\1 #\\3)", "NIL");
make_test ("(char<= #\\0 #\\1 #\\1 #\\3)", "T");
make_test ("(char<= #\\0 #\\1 #\\0 #\\3)", "NIL");
make_test ("(char> #\\3 #\\2 #\\1 #\\0)", "T");
make_test ("(char> #\\3 #\\1 #\\1 #\\0)", "NIL");
make_test ("(char>= #\\3 #\\1 #\\1 #\\0)", "T");
make_test ("(char>= #\\3 #\\0 #\\1 #\\0)", "NIL");
make_test ("(char-equal #\\a)", "T");
make_test ("(char-equal #\\a #\\a)", "T");
make_test ("(char-equal #\\a #\\A)", "T");
make_test ("(char-equal #\\a #\\A #\\a)", "T");
make_test ("(char-equal #\\a #\\A #\\b)", "NIL");
make_test ("(char-not-equal #\\a #\\b #\\B)", "NIL");
make_test ("(char-not-equal #\\a #\\b #\\C)", "T");
make_test ("(char-lessp #\\a #\\b #\\B)", "NIL");
make_test ("(char-lessp #\\a #\\b #\\c)", "T");
make_test ("(char-not-greaterp #\\a #\\b #\\B)", "T");
make_test ("(char-not-greaterp #\\a #\\b #\\a)", "NIL");
make_test ("(char-greaterp #\\B #\\b #\\a)", "NIL");
make_test ("(char-greaterp #\\c #\\b #\\a)", "T");
make_test ("(char-not-lessp #\\B #\\b #\\a)", "T");
make_test ("(char-not-lessp #\\a #\\b #\\a)", "NIL");
make_test ("(char-upcase #\\a)", "#\\A");
make_test ("(char-upcase #\\A)", "#\\A");
make_test ("(char-upcase #\\3)", "#\\3");
make_test ("(char-downcase #\\B)", "#\\b");
make_test ("(char-downcase #\\b)", "#\\b");
make_test ("(char-downcase #\\@)", "#\\@");
make_test ("(alpha-char-p #\\a)", "T");
make_test ("(alpha-char-p #\\3)", "NIL");
make_test ("(alphanumericp #\\a)", "T");
make_test ("(alphanumericp #\\3)", "T");
make_test ("(alphanumericp #\\#)", "NIL");
make_test ("(code-char (char-code #\\a))", "#\\a");
make_test ("(code-char (char-code #\\ò))", "#\\ò");
make_test ("(digit-char 2)", "#\\2");
make_test ("(digit-char 10)", "NIL");
make_test ("(digit-char 10 16)", "#\\A");
make_test ("(digit-char 16 16)", "NIL");
make_test ("(digit-char-p #\\2)", "2");
make_test ("(digit-char-p #\\A)", "NIL");
make_test ("(digit-char-p #\\a)", "NIL");
make_test ("(digit-char-p #\\A 16)", "10");
make_test ("(digit-char-p #\\a 16)", "10");
make_test ("(digit-char-p #\\g 16)", "NIL");
make_test ("(string-upcase \"Hello\")", "\"HELLO\"");
make_test ("(string-downcase \"Hello\")", "\"hello\"");
make_test ("(string-capitalize \"this is a Good day\")", "\"This Is A Good Day\"");
make_test ("(string-left-trim '(#\\a #\\b) \"aabbabahello\")", "\"hello\"");
make_test ("(string-left-trim '(#\\a #\\b) \"aabbab\")", "\"\"");
make_test ("(string-left-trim #(#\\N) nil)", "\"IL\"");
make_test ("(string-right-trim '(#\\a #\\b) \"helloaabbab\")", "\"hello\"");
make_test ("(string-right-trim '(#\\a #\\b) \"aabbab\")", "\"\"");
make_test ("(string-right-trim #(#\\l #\\L) nil)", "\"NI\"");
make_test ("(string-right-trim #(#\\l #\\L) #\\l)", "\"\"");
make_test ("(string-trim '(#\\a #\\b) \"baahelloaabbab\")", "\"hello\"");
make_test ("(string-trim \"l\" #\\l)", "\"\"");
make_test ("(car '(1 2))", "1");
make_test ("(cdr '(1 2))", "(2)");
make_test ("(car ())", "NIL");
make_test ("(cdr ())", "NIL");
make_test ("(cdar '((0 . 1)))", "1");
make_test ("(setf tt (list 'a 'b 'd))", "(A B D)");
make_test ("(rplaca tt 'f)", "(F B D)");
make_test ("tt", "(F B D)");
make_test ("(rplacd tt 'g)", "(F . G)");
make_test ("tt", "(F . G)");
make_test ("(nth 0 '(0))", "0");
make_test ("(nth 1 '(0))", "NIL");
make_test ("(nth 3 nil)", "NIL");
make_test ("(nthcdr 1 ())", "NIL");
make_test ("(nthcdr 2 '(a b c))", "(C)");
make_test ("(nthcdr 1 '(0 . 1))", "1");
make_test ("(nth-value 0 (values 10 9))", "10");
make_test ("(nth-value 1 (values 10 9))", "9");
make_test ("(nth-value 2 (values 10 9))", "NIL");
make_test ("(nth-value 3 (values 10 9))", "NIL");
make_test ("(elt '(0 3) 1)", "3");
make_test ("(elt \"abc\" 0)", "#\\a");
make_test ("(elt \"abcòe\" 3)", "#\\ò");
make_test ("(elt \"abcòe\" 4)", "#\\e");
make_test ("(elt \"abcòe\" 2)", "#\\c");
make_test ("(elt #(3 2 1) 1)", "2");
make_test ("(elt #*01001 0)", "0");
make_test ("(elt #*01001 4)", "1");
make_test ("(aref \"abc\" 2)", "#\\c");
make_test ("(aref #(1 2 3) 1)", "2");
make_test ("(aref #*01001 4)", "1");
make_test ("(aref #*01001 0)", "0");
make_test ("(aref (make-array nil))", "NIL");
make_test ("(aref (make-array '(1 2 3)) 0 1 2)", "NIL");
make_test ("(row-major-aref #(0 1 2) 1)", "1");
make_test ("(row-major-aref (make-array '(3 2 1)) 5)", "NIL");
make_test ("(char \"àbcdef\" 2)", "#\\c");
make_test ("(schar \"àbcdef\" 2)", "#\\c");
make_test ("(bit #*0100 1)", "1");
make_test ("(sbit #*0100 1)", "1");
make_test ("(svref #(0 1 2) 1)", "1");
make_test ("(let ((v (make-array 5 :fill-pointer 3 :initial-contents '(5 6 7 8 9)))) (list (vector-pop v) (fill-pointer v)))", "(7 2)");
make_test ("(let ((v (make-array 5 :fill-pointer 5 :initial-contents '(5 6 7 8 9)))) (list (vector-push 10 v) v))", "(NIL #(5 6 7 8 9))");
make_test ("(let ((v (make-array 5 :fill-pointer 4 :initial-contents '(5 6 7 8 9)))) (list (vector-push 10 v) v))", "(4 #(5 6 7 8 10))");
make_test ("(let ((v (make-array 5 :fill-pointer 5 :initial-contents '(5 6 7 8 9)))) (list (vector-push-extend 10 v 2) v))", "(5 #(5 6 7 8 9 10))");
make_test ("(let ((v (make-array 5 :fill-pointer 4 :initial-contents '(5 6 7 8 9)))) (list (vector-push-extend 10 v) v))", "(4 #(5 6 7 8 10))");
make_test ("(copy-list '(0 1 2))", "(0 1 2)");
make_test ("(elt (copy-list '(0 1)) 1)", "1");
make_test ("(copy-list '(0 1 . 2))", "(0 1 . 2)");
make_test ("(copy-list nil)", "NIL");
make_test ("(copy-seq nil)", "NIL");
make_test ("(copy-seq \"abc\")", "\"abc\"");
make_test ("(copy-seq '(0 1 2))", "(0 1 2)");
make_test ("(copy-seq #(0 1 2))", "#(0 1 2)");
make_test ("(subseq nil 0)", "NIL");
make_test ("(subseq \"hello\" 1)", "\"ello\"");
make_test ("(subseq \"hello\" 1 3)", "\"el\"");
make_test ("(subseq \"hello\" 1 1)", "\"\"");
make_test ("(subseq #(0 1 2) 1 3)", "#(1 2)");
make_test ("(subseq #(0 1 2) 1 1)", "#()");
make_test ("(subseq '(0 1 2) 1 nil)", "(1 2)");
make_test ("(subseq '(0 1 2) 1 1)", "NIL");
make_test ("(setf (subseq '(0 1 2 3) (+)) '(4 5 6))", "(4 5 6)");
make_test ("(let ((x '(0 1 2 3))) (setf (subseq x (+)) '(4 5 6)) x)", "(4 5 6 3)");
make_test ("(let ((x '(0 1))) (setf (subseq x (+)) '(4 5 6)) x)", "(4 5)");
make_test ("(let ((x '(0 1 2 3))) (setf (subseq x 1 2) #(4 5 6)) x)", "(0 4 2 3)");
make_test ("(let ((x #(0 1 2 3))) (setf (subseq x 1 2) '(4 5 6)) x)", "#(0 4 2 3)");
make_test ("(let ((l '(0 1 2))) (setf (subseq l 0 0) '(3)) l)", "(0 1 2)");
make_test ("(setf (subseq nil 0) '(1))", "(1)");
make_test ("(length ())", "0");
make_test ("(length \"aaa\")", "3");
make_test ("(length #(1 2))", "2");
make_test ("(length '(0 1 2))", "3");
make_test ("(list-length '(0 1 2 3))", "4");
make_test ("(make-array 4)", "#(NIL NIL NIL NIL)");
make_test ("(make-array '(2 3))", "#2A((NIL NIL NIL) (NIL NIL NIL))");
make_test ("(let ((*print-array* nil)) (write (make-array '(1 2 3))))", "#<ARRAY, RANK 3>\n#3A(((NIL NIL NIL) (NIL NIL NIL)))");
make_test ("(make-array '(1 2 3))", "#3A(((NIL NIL NIL) (NIL NIL NIL)))");
make_test ("(make-array nil)", "#0ANIL");
make_test ("(make-array '(2 3 2) :initial-contents '(((a b) (c d) (e f)) ((g h) (i j) (k l))))", "#3A(((A B) (C D) (E F)) ((G H) (I J) (K L)))");
make_test ("(make-array '(2 3 2) :initial-contents #(((a b) #(c d) \"ef\") ((g h) #(i j) #*10)))", "#3A(((A B) (C D) (#\\e #\\f)) ((G H) (I J) (1 0)))");
make_test ("(make-array 3 :initial-contents '(a b c))", "#(A B C)");
make_test ("(make-array 3 :element-type 'character)", "\"\0\0\0\"");
make_test ("(make-array 3 :element-type 'character :initial-contents \"abc\")", "\"abc\"");
make_test ("(make-array 3 :element-type 'character :initial-contents #(#\\a #\\b #\\c))", "\"abc\"");
make_test ("(make-array 3 :element-type 'character :initial-contents '(#\\a #\\b #\\c))", "\"abc\"");
make_test ("(make-array 3 :fill-pointer 2 :element-type 'character :initial-contents \"abc\")", "\"ab\"");
make_test ("(defparameter arr (make-array 5 :fill-pointer 3))", "ARR");
make_test ("arr", "#(NIL NIL NIL)");
make_test ("(fill-pointer arr)", "3");
make_test ("(setf (fill-pointer arr) 4)", "4");
make_test ("arr", "#(NIL NIL NIL NIL)");
make_test ("(fill-pointer arr)", "4");
make_test ("(array-has-fill-pointer-p arr)", "T");
make_test ("(let ((arr (make-array 5 :fill-pointer nil))) (array-has-fill-pointer-p arr))", "NIL");
make_test ("(let ((arr (make-array 7 :fill-pointer t))) (fill-pointer arr))", "7");
make_test ("(length arr)", "4");
make_test ("(vector)", "#()");
make_test ("(vector 'a 'b \"\")", "#(A B \"\")");
make_test ("#*", "#*");
make_test ("#0*", "#*");
make_test ("#*011001", "#*011001");
make_test ("#4*010", "#*0100");
make_test ("#4*011", "#*0111");
make_test ("(array-has-fill-pointer-p \"aaa\")", "NIL");
make_test ("(array-has-fill-pointer-p #*010)", "NIL");
make_test ("(length #*011001)", "6");
make_test ("(array-rank #())", "1");
make_test ("(array-rank (make-array nil))", "0");
make_test ("(array-rank (make-array '(1 2)))", "2");
make_test ("(array-rank #*01)", "1");
make_test ("(array-dimensions \"aaa\")", "(3)");
make_test ("(array-dimensions #(1 2 3 4))", "(4)");
make_test ("(array-dimensions #*01)", "(2)");
make_test ("(array-dimension \"aaa\" 0)", "3");
make_test ("(array-total-size \"aaa\")", "3");
make_test ("(array-total-size #*01)", "2");
make_test ("(array-in-bounds-p (make-array nil))", "T");
make_test ("(array-in-bounds-p \"aaa\" 3)", "NIL");
make_test ("(array-in-bounds-p (make-array '(1 2 3)) 0 1 2)", "T");
make_test ("(array-in-bounds-p (make-array '(1 2 3)) 1 1 2)", "NIL");
make_test ("(array-in-bounds-p #*01 2)", "NIL");
make_test ("(array-row-major-index \"aaa\" 1)", "1");
make_test ("(array-row-major-index #(0 1 2) 0)", "0");
make_test ("(array-row-major-index (make-array nil))", "0");
make_test ("(array-row-major-index (make-array '(1 2 3)) 0 0 0)", "0");
make_test ("(array-row-major-index (make-array '(2 3 4)) 0 0 1)", "1");
make_test ("(array-row-major-index (make-array '(2 3 4)) 0 2 0)", "8");
make_test ("(array-row-major-index (make-array '(2 3 4)) 1 0 0)", "12");
make_test ("(array-row-major-index #*010 1)", "1");
make_test ("(array-element-type \"abc\")", "CHARACTER");
make_test ("(array-element-type #*01)", "BIT");
make_test ("(let ((arr (make-array 5))) (adjust-array arr 3) arr)", "#(NIL NIL NIL)");
make_test ("(let ((arr (make-array 2))) (adjust-array arr 3) arr)", "#(NIL NIL NIL)");
make_test ("(let ((str \"abcdef\")) (adjust-array str 4) str)", "\"abcd\"");
make_test ("(let ((str \"abcdef\")) (adjust-array str 8) (length str))", "8");
make_test ("(adjust-array #(0 1 2 4 5 3) 5)", "#(0 1 2 4 5)");
make_test ("(adjust-array #2a((a b c) (d e f)) '(3 4))", "#2A((A B C NIL) (D E F NIL) (NIL NIL NIL NIL))");
make_test ("(adjust-array #2a((a b c) (d e f)) '(2 2))", "#2A((A B) (D E))");
make_test ("(adjust-array #2a((a b c d) (e f g h) (i j k l)) '(2 3))", "#2A((A B C) (E F G))");
make_test ("(adjust-array #3a(((a b) (c d) (e f)) ((g h) (i j) (k l))) '(3 4 3))", "#3A(((A B NIL) (C D NIL) (E F NIL) (NIL NIL NIL)) ((G H NIL) (I J NIL) (K L NIL) (NIL NIL NIL)) ((NIL NIL NIL) (NIL NIL NIL) (NIL NIL NIL) (NIL NIL NIL)))");
make_test ("(defparameter tbl (make-hash-table))", "TBL");
make_test ("tbl", "#<HASH-TABLE EQL 0/1024>");
make_test ("(hash-table-size tbl)", "1024");
make_test ("(hash-table-count tbl)", "0");
make_test ("(hash-table-test tbl)", "EQL");
make_test ("(gethash 10 tbl)", "NIL\nNIL");
make_test ("(setf (gethash 10 tbl) 11)", "11");
make_test ("(gethash 10 tbl)", "11\nT");
make_test ("(hash-table-size tbl)", "1024");
make_test ("(hash-table-count tbl)", "1");
make_test ("(setf (gethash #\\a tbl) 12)", "12");
make_test ("(maphash (lambda (k v) (write k) (write v)) tbl)", "1011#\\a12\nNIL");
make_test ("(hash-table-count tbl)", "2");
make_test ("(with-hash-table-iterator (nextval tbl) (do nil (nil) (multiple-value-bind (pred k v) (nextval) (if pred (write v) (return)))))", "1211\nNIL");
make_test ("(defparameter tbl (make-hash-table :test 'eq))", "TBL");
make_test ("tbl", "#<HASH-TABLE EQ 0/1024>");
make_test ("(gethash 'k tbl)", "NIL\nNIL");
make_test ("(setf (gethash 'k tbl) 10)", "10");
make_test ("(hash-table-size tbl)", "1024");
make_test ("(hash-table-count tbl)", "1");
make_test ("(gethash 'k tbl)", "10\nT");
make_test ("(remhash 'k tbl)", "T");
make_test ("(gethash 'k tbl)", "NIL\nNIL");
make_test ("(hash-table-count tbl)", "0");
make_test ("(defparameter tbl (make-hash-table :test #'equal))", "TBL");
make_test ("(setf (gethash \"hello\" tbl) 12)", "12");
make_test ("(gethash \"hello\" tbl)", "12\nT");
make_test ("(setf (gethash \"helloo\" tbl) 13)", "13");
make_test ("(gethash \"helloo\" tbl)", "13\nT");
make_test ("(hash-table-count tbl)", "2");
make_test ("(remhash \"hello\" tbl)", "T");
make_test ("(gethash \"hello\" tbl)", "NIL\nNIL");
make_test ("(hash-table-count tbl)", "1");
make_test ("(clrhash tbl)", "#<HASH-TABLE EQUAL 0/1024>");
make_test ("(hash-table-count tbl)", "0");
make_test ("(setf (gethash '(\"a\") tbl) 10)", "10");
make_test ("(gethash '(\"a\") tbl)", "10\nT");
make_test ("(defparameter tbl (make-hash-table :test 'equalp))", "TBL");
make_test ("(setf (gethash \"ab\" tbl) 10)", "10");
make_test ("(gethash \"Ab\" tbl)", "10\nT");
make_test ("(setf (gethash #(0 1 2) tbl) 11)", "11");
make_test ("(gethash #(0 1 2) tbl)", "11\nT");
make_test ("(last '(1 2 3))", "(3)");
make_test ("(last '(1 2 3) 0)", "NIL");
make_test ("(last '(1 2 3) 2)", "(2 3)");
make_test ("(pathname \"foo\")", "#P\"foo\"");
make_test ("(pathname (open #p\"README\"))", "#P\"README\"");
make_test ("(pathname #p\"foo\")", "#P\"foo\"");
make_test ("(logical-pathname \"foo\")", "#P\"foo\"");
make_test ("(translate-logical-pathname #p\".\")", "#P\".\"");
make_test ("(translate-logical-pathname \".\")", "#P\".\"");
make_test ("(make-pathname :name \"foo\")", "#P\"foo\"");
make_test ("(make-pathname :directory \"foo\" :name \"bar\")", "#P\"/foo/bar\"");
make_test ("(make-pathname :device \"foo\" :directory \"bar\")", "#P\"/bar/\"");
make_test ("(make-pathname :name nil)", "#P\"\"");
make_test ("(make-pathname :name :wild)", "#P\"*\"");
make_test ("(make-pathname :directory \"foo\" :name :wild)", "#P\"/foo/*\"");
make_test ("(make-pathname :directory :wild)", "#P\"/**/\"");
make_test ("(make-pathname :directory '(:relative :wild))", "#P\"*/\"");
make_test ("(make-pathname :directory '(:absolute :wild))", "#P\"/*/\"");
make_test ("(make-pathname :directory '(:absolute \"foo\" :wild \"bar\"))", "#P\"/foo/*/bar/\"");
make_test ("(make-pathname :directory '(:relative \"foo\" :wild \"bar\"))", "#P\"foo/*/bar/\"");
make_test ("(make-pathname :directory '(:relative \"foo\" :wild \"bar\") :name \"baz\")", "#P\"foo/*/bar/baz\"");
make_test ("(make-pathname :directory '(:relative \"foo\" :wild-inferiors \"bar\") :name \"blah\")", "#P\"foo/**/bar/blah\"");
make_test ("(make-pathname :name \"foo\" :defaults #p\"/home/\")", "#P\"/home/foo\"");
make_test ("(make-pathname :directory \"home\" :defaults #p\"foo\")", "#P\"/home/foo\"");
make_test ("(make-pathname :name \"foo\" :directory nil :defaults \"/home/\")", "#P\"foo\"");
make_test ("(make-pathname :name \"foo\" :type \"bar\")", "#P\"foo.bar\"");
make_test ("(make-pathname :name :wild :type :wild)", "#P\"*\"");
make_test ("(make-pathname :directory \"foo\" :name \"bar\" :type \"baz\")", "#P\"/foo/bar.baz\"");
make_test ("(make-pathname :directory (al-pathname-directory \"/home/foo.bar\") :name (pathname-name \"/home/foo.bar\") :type (pathname-type \"/home/foo.bar\"))", "#P\"/home/foo.bar\"");
make_test ("(namestring \"foo/bar\")", "\"foo/bar\"");
make_test ("(namestring \"/foo/bar\")", "\"/foo/bar\"");
make_test ("(al-pathname-directory #p\"foo/bar\")", "\"foo/\"");
make_test ("(pathname-directory #p\"foo/bar\")", "(:RELATIVE \"foo\")");
make_test ("(al-pathname-directory #p\"/foo/bar\")", "\"/foo/\"");
make_test ("(pathname-directory #p\"/foo/bar\")", "(:ABSOLUTE \"foo\")");
make_test ("(al-pathname-directory #p\"/\")", "\"/\"");
make_test ("(pathname-directory #p\"/\")", "(:ABSOLUTE)");
make_test ("(al-pathname-directory #p\"/foo/\")", "\"/foo/\"");
make_test ("(al-pathname-directory #p\"foo\")", "NIL");
make_test ("(al-pathname-directory \"abc\")", "NIL");
make_test ("(pathname-directory \"abc\")", "NIL");
make_test ("(pathname-name #p\"foo\")", "\"foo\"");
make_test ("(pathname-name (make-pathname :name \"foo\"))", "\"foo\"");
make_test ("(pathname-name #p\"/foo/\")", "NIL");
make_test ("(pathname-name \"abc\")", "\"abc\"");
make_test ("(pathname-name (make-pathname :name \"foo\" :type nil))", "\"foo\"");
make_test ("(pathname-name \"..\")", "\".\"");
make_test ("(pathname-name \"foo/.\")", "\".\"");
make_test ("(pathname-name \"foo.\")", "\"foo\"");
make_test ("(pathname-type \"foo.\")", "\"\"");
make_test ("(pathname-name \".foo\")", "\"\"");
make_test ("(pathname-type \".foo\")", "\"foo\"");
make_test ("(pathname-name \".\")", "\".\"");
make_test ("(pathname-type \".\")", "NIL");
make_test ("(pathname-type (make-pathname :name \"foo\"))", "NIL");
make_test ("(pathname-type #p\"foo.bar\")", "\"bar\"");
make_test ("(al-pathname-directory \"\")", "NIL");
make_test ("(pathname-name \"\")", "NIL");
make_test ("(pathname-type \"\")", "NIL");
make_test ("(make-pathname :directory \"/foo/\" :name \"bar\")", "#P\"/foo/bar\"");
make_test ("(make-pathname :directory (al-pathname-directory #p\"/foo\") :name (pathname-name #p\"/foo\"))", "#P\"/foo\"");
make_test ("(make-pathname :name \"foo\" :type \"\")", "#P\"foo.\"");
make_test ("(wild-pathname-p (make-pathname :name :wild))", "T");
make_test ("(wild-pathname-p (make-pathname :name :wild) :name)", "T");
make_test ("(wild-pathname-p (make-pathname :directory :wild) :name)", "NIL");
make_test ("(wild-pathname-p (make-pathname :directory :wild) :directory)", "T");
make_test ("(merge-pathnames \"foo/\" #p\"bar\")", "#P\"foo/bar\"");
make_test ("(merge-pathnames #p\"foo\" \"bar/\")", "#P\"bar/foo\"");
make_test ("(merge-pathnames \"/foo/\" #p\"bar\")", "#P\"/foo/bar\"");
make_test ("(merge-pathnames \".\" \"..\")", "#P\"..\"");
make_test ("(merge-pathnames \"/\" \"foo.bar\")", "#P\"/foo.bar\"");
make_test ("(truename #p\"README\")", "#P\"README\"");
make_test ("(pathnamep (user-homedir-pathname))", "T");
make_test ("(pathname-match-p \"foo\" \"*\")", "T");
make_test ("(pathname-match-p #p\"foo\" \"foo\")", "T");
make_test ("(pathname-match-p \"foo\" \"fooo\")", "NIL");
make_test ("(pathname-match-p \"/home/foo\" #p\"/home/foo\")", "T");
make_test ("(pathname-match-p \"/home/foo/\" \"/home/\")", "NIL");
make_test ("(pathname-match-p \"/home/foo\" \"/home/*\")", "T");
make_test ("(pathname-match-p \"/home/foo\" \"/home/fooo\")", "NIL");
make_test ("(pathname-match-p #P\"/foo/bar/baz/abc.def\" \"/foo/**/*\")", "T");
make_test ("(pathname-match-p #p\"/home/foo/bar/baz/abc/def\" #p\"/home/**/baz/abc/def\")", "T");
make_test ("(pathname-match-p #p\"/home/foo.bar\" \"/home/foo.*\")", "T");
make_test ("(pathname-match-p #p\"/home/foo.bar\" \"/home/*.bar\")", "T");
make_test ("(translate-pathname \"foo\" \"*\" \"bar\")", "#P\"bar\"");
make_test ("(translate-pathname \"foo\" \"*\" \"*.bar\")", "#P\"foo.bar\"");
make_test ("(translate-pathname \"foo.bar\" \"*\" \"baz.*\")", "#P\"baz.bar\"");
make_test ("(translate-pathname \"foo/bar/\" \"*\" \"*/baz\")", "#P\"foo/bar/baz\"");
make_test ("(translate-pathname \"/foo/bar/\" \"*\" \"*/baz\")", "#P\"/foo/bar/baz\"");
make_test ("(translate-pathname \"/bar/baz/file\" \"**/\" \"/foo/**/\")", "#P\"/foo/bar/baz/file\"");
make_test ("(probe-file \"README\")", "#P\"README\"");
make_test ("(probe-file \"foo\")", "NIL");
make_test ("(with-open-file (s \"README\" :direction :input) (read s) (write (read s)) (write (file-position s)) (file-position s 0) (write (read s)))", "(C)13COPYRIGHT\nCOPYRIGHT");
make_test ("(with-open-file (s \"README\" :direction :input) (read s) (read s))", "(C)");
make_test ("(with-open-stream (s (make-string-input-stream \"0\")) (read s))", "0");
make_test ("(read-char (make-string-input-stream \"abcd\"))", "#\\a");
make_test ("(let ((str (make-string-input-stream \"abcd\"))) (write (read-char str)) (write (read-char str)))", "#\\a#\\b\n#\\b");
make_test ("(read-char (make-string-input-stream \"\") nil 0)", "0");
make_test ("(with-open-stream (s (open \"README\" :direction :input)) (write (read-char s)) (write (read-char s)))", "#\\C#\\o\n#\\o");
make_test ("(read-line)\nabc", "\"abc\"\nNIL", 1);
make_test ("(read-line (make-string-input-stream \"hello world\"))", "\"hello world\"\nT");
make_test ("(read-line (make-string-input-stream \"hello world\n" .
	   "\"))", "\"hello world\"\nNIL");
make_test ("(read)\n\"hello\"\"world\" 12 `(1 2 3) #\\c aa  ", "\"hello\"", 1);
make_test ("(read)", "\"world\"");
make_test ("(read)", "12");
make_test ("(read)", "`(1 2 3)");
make_test ("(read)", "#\\c");
make_test ("(read)", "AA");
make_test ("(read-line)", "\" \"\nNIL");
make_test ("(defparameter inp (make-string-input-stream \"\\\"hello\\\"\\\"world\\\" #+foo 0 12 `(1 2 3) #\\\\c aa  \"))", "INP");
make_test ("(read inp)", "\"hello\"");
make_test ("(read inp)", "\"world\"");
make_test ("(read inp)", "12");
make_test ("(read inp)", "`(1 2 3)");
make_test ("(read inp)", "#\\c");
make_test ("(read inp)", "AA");
make_test ("(read-line inp)", "\" \"\nT");
make_test ("(read (make-string-input-stream \"hello\"))", "HELLO");
make_test ("(read (make-string-input-stream \"123\"))", "123");
make_test ("(read (make-string-input-stream \"\") nil)", "NIL");
make_test ("(read (make-string-input-stream \"\") nil 10)", "10");
make_test ("(read *standard-input*)\n(1 2 . 3)", "(1 2 . 3)", 1);
make_test ("(read)\n#+foo #\\space 0", "0", 1);
make_test ("(read-preserving-whitespace)\n\"hello\"\"world\" 12 `(1 2 3) #\\c aa  ", "\"hello\"", 1);
make_test ("(read-preserving-whitespace)", "\"world\"");
make_test ("(read-preserving-whitespace)", "12");
make_test ("(read-preserving-whitespace)", "`(1 2 3)");
make_test ("(read-preserving-whitespace)", "#\\c");
make_test ("(read-preserving-whitespace)", "AA");
make_test ("(read-preserving-whitespace (make-string-input-stream \"\") nil)", "NIL");
make_test ("(read-preserving-whitespace (make-string-input-stream \"\") nil 10)" , "10");
make_test ("(read-line)", "\"  \"\nNIL");
make_test ("(defparameter inp2 (make-string-input-stream \"\\\"hello\\\"\\\"world\\\" #+foo 0 12 `(1 2 3) #\\\\c aa  \"))", "INP2");
make_test ("(read-preserving-whitespace inp2)", "\"hello\"");
make_test ("(read-preserving-whitespace inp2)", "\"world\"");
make_test ("(read-preserving-whitespace inp2)", "12");
make_test ("(read-preserving-whitespace inp2)", "`(1 2 3)");
make_test ("(read-preserving-whitespace inp2)", "#\\c");
make_test ("(read-preserving-whitespace inp2)", "AA");
make_test ("(read-line inp2)", "\"  \"\nT");
make_test ("(read-preserving-whitespace (make-string-input-stream \"hello\"))", "HELLO");
make_test ("(read-preserving-whitespace (make-string-input-stream \"123\"))", "123");
make_test ("(let ((s (make-string-input-stream \"abc def\"))) (read s) (al-string-input-stream-string s))", "\"def\"");
make_test ("(read-from-string \" abc\")", "ABC\n4");
make_test ("(read-from-string \" 1  \")", "1\n2");
make_test ("(read-from-string \":hi\")", ":HI\n3");
make_test ("(read-from-string \"#+foo nil #+foo nil 0\")", "0\n21");
make_test ("(let ((*package* *package*)) (read-from-string \"#:foo\"))", "#:FOO\n5");
make_test ("(parse-integer \"2\")", "2\n1");
make_test ("(parse-integer \"  25 \")", "25\n5");
make_test ("(parse-integer \"  -4\")", "-4\n4");
make_test ("(let ((seq (make-array 8))) (write (read-sequence seq (make-string-input-stream \"0 1 2 3 4 5\") :start 5)) seq)", "8\n#(NIL NIL NIL NIL NIL #\\0 #\\Space #\\1)");
make_test ("(eval '(write \"\"))", "\"\"\n\"\"");
make_test ("(setq var 10)", "10");
make_test ("(let ((var 12)) (eval 'var))", "12");
make_test ("(defun testfun () (loop for i to 5 do (write i)))", "TESTFUN");
make_test ("(testfun)", "012345\nNIL");
make_test ("(typep #'testfun 'compiled-function)", "NIL");
make_test ("(typep #'car 'compiled-function)", "T");
make_test ("(compile 'testfun)", "TESTFUN\nNIL\nNIL");
make_test ("(testfun)", "012345\nNIL");
make_test ("(typep #'testfun 'compiled-function)", "T");
make_test ("(compile 'testfun (lambda nil (loop for i to 4 do (write i))))", "TESTFUN\nNIL\nNIL");
make_test ("(testfun)", "01234\nNIL");
make_test ("(funcall (compile nil (lambda nil (loop for i to 3 do (write i)))))", "0123\nNIL");
make_test ("(defmacro testmac nil (loop for i to 5 do (write i)))", "TESTMAC");
make_test ("(testmac)", "012345\nNIL");
make_test ("(compile 'testmac)", "TESTMAC\nNIL\nNIL");
make_test ("(testmac)", "012345\nNIL");
make_test ("(defmacro testmac2 nil '(loop for i to 5 do (write i)))", "TESTMAC2");
make_test ("(defun testfun2 nil (testmac2))", "TESTFUN2");
make_test ("(compile 'testfun2)", "TESTFUN2\nNIL\nNIL");
make_test ("(testfun2)", "012345\nNIL");
make_test ("(defun testfun3 nil)", "TESTFUN3");
make_test ("(compile 'testfun3)", "TESTFUN3\nNIL\nNIL");
make_test ("(defun testfun4 () 0 1 2)", "TESTFUN4");
make_test ("(compile 'testfun4)", "TESTFUN4\nNIL\nNIL");
make_test ("(testfun4)", "2");
make_test ("(defun testfun5 nil (1+ (loop for i from 0 return i)))", "TESTFUN5");
make_test ("(compile 'testfun5)", "TESTFUN5\nNIL\nNIL");
make_test ("(testfun5)", "1");
make_test ("(defun testfun6 nil (if 0 (cond (nil))))", "TESTFUN6");
make_test ("(compile 'testfun6)", "TESTFUN6\nNIL\nNIL");
make_test ("(testfun6)", "NIL");
make_test ("(with-compilation-unit nil (write 0) (write 1))", "01\n1");
make_test ("(compile-file-pathname \"foo.lisp\")", "#P\"foo.alc\"");
make_test ("(compile-file-pathname \"foo/bar.lisp\")", "#P\"foo/bar.alc\"");
make_test ("(write \"\")", "\"\"\n\"\"");
make_test ("(write 'hello :stream *error-output*)", "HELLO\nHELLO");
make_test ("(defun f nil (write 0) (write 1))", "F");
make_test ("(f)", "01\n1");
make_test ("(defun f nil (write 0) (values))", "F");
make_test ("(f)", "0");
make_test ("(let ((*standard-output* *standard-output*) (var (write 0))))", "0\nNIL");
make_test ("(write-string \"aaa\\n\")", "aaan\n\"aaan\"");
make_test ("(write-string \"\n" .
	   "\")", "\n\"\n\"");
make_test ("(write-char #\\a)", "a\n#\\a");
make_test ("(write-char #\\newline)", "\n#\\Newline");
make_test ("(write-byte (char-code #\\a) *standard-output*)", "a\n97");
make_test ("(fresh-line)", "NIL");
make_test ("(terpri)", "\nNIL");
make_test ("(write-line \"aaa\")", "aaa\n\"aaa\"");
make_test ("(write-sequence \"hello world\" *standard-output* :start 1 :end 5)", "ello\n\"hello world\"");
make_test ("(let ((*print-base* 8)) (write 8))", "10\n8");
make_test ("(let ((*print-base* 16)) (write 32))", "20\n32");
make_test ("(let ((*print-base* 16)) (write 32/19))", "20/13\n32/19");
make_test ("(let ((*print-radix* t)) (write 15))", "15.\n15");
make_test ("(let ((*print-radix* t)) (write 1/2))", "#10r1/2\n1/2");
make_test ("(let ((*print-base* 8) (*print-radix* t)) (write 1/2))", "#o1/2\n1/2");
make_test ("(let ((*print-base* 8) (*print-radix* t)) (write 10))", "#o12\n10");
make_test ("(let ((*print-base* 16) (*print-radix* t)) (write 10))", "#xa\n10");
make_test ("(let ((*print-gensym* nil)) (write (make-symbol \"FOOO\")) (write '#:foo))", "FOOOFOO\n#:FOO");
make_test ("(prin1 '|Aa,a|)", "|Aa,a|\n|Aa,a|");
make_test ("(prin1 #\\a)", "#\\a\n#\\a");
make_test ("(prin1 \"aaa\" *standard-output*)", "\"aaa\"\n\"aaa\"");
make_test ("(princ '|Aa,a|)", "Aa,a\n|Aa,a|");
make_test ("(princ :hi)", "HI\n:HI");
make_test ("(princ #\\a)", "a\n#\\a");
make_test ("(princ \"aaa\" *standard-output*)", "aaa\n\"aaa\"");
make_test ("(write-to-string :aaa)", "\":AAA\"");
make_test ("(prin1-to-string :aaa)", "\":AAA\"");
make_test ("(princ-to-string :aaa)", "\"AAA\"");
make_test ("(print \"aaa\" *standard-output*)", "\n\"aaa\" \n\"aaa\"");
make_test ("(pprint 0)", "\n0");
make_test ("(pprint '(0 1 2))", "\n(0 1 2)");
make_test ("(pprint '(0 1 . 2))", "\n(0 1 . 2)");
make_test ("(pprint '(0 (1 2) 3))", "\n(0\n  (1 2) 3)");
make_test ("(pprint '(0 (1)))", "\n(0\n  (1))");
make_test ("(pprint '(0 1 (2 3) . 4))", "\n(0 1\n  (2 3) . 4)");
make_test ("#'print-object", "#<STANDARD-GENERIC-FUNCTION PRINT-OBJECT>");
make_test ("(print-object \"abc\" *standard-output*)", "\"abc\"\n\"abc\"");
make_test ("(defmethod print-object ((object string) stream) (write 'thisisastring) (call-next-method))", "#<STANDARD-METHOD PRINT-OBJECT (STRING T)>");
make_test ("(print-object \"abc\" *standard-output*)", "THISISASTRING\"abc\"\n\"abc\"");
make_test ("(remove-method #'print-object (find-method #'print-object nil '(string t)))", "#<STANDARD-GENERIC-FUNCTION PRINT-OBJECT>");
make_test ("(print-object \"abc\" *standard-output*)", "\"abc\"\n\"abc\"");
make_test ("(let ((c 0)) (loop (if (= c 2) (return 0)) (incf c)))", "0");
make_test ("(loop with var1 and var2 return (list var1 var2))", "(NIL NIL)");
make_test ("(loop with i = (+ 0) with j = 1 do (incf i) (decf j) (if (= i 5) (return (list i j))))", "(5 -4)");
make_test ("(loop with i = (+ 0) with j = (+ i 1) do (incf i) (decf j) (if (= i 5) (return (list i j))))", "(5 -4)");
make_test ("(let ((i 0)) (loop with i = 1 and j = (+ i 1) do (incf i) (decf j) (if (= i 5) (return (list i j)))))", "(5 -3)");
make_test ("(loop with i = 1 return i)", "1");
make_test ("(loop with i = 0 and j = 5 do (incf i) (incf j) if (> i 1) if (> j 7) do (write 'done) (return (list i j)))", "DONE\n(3 8)");
make_test ("(loop with i = 0 do (incf i) unless (> i 5) do (write i) else do (return nil))", "12345\nNIL");
make_test ("(loop named foo for i from 0 if (= i 5) return i)", "5");
make_test ("(loop :when nil :do 't :else :when nil :do 't :else :return 't)", "T");
make_test ("(loop for i in '(0 1 2) do (write i) finally (write i))", "0122\nNIL");
make_test ("(loop for x of-type fixnum in '(0 1) do (write x))", "01\nNIL");
make_test ("(loop for i in '(0 1 2 3 4) if (numberp i) if (oddp i) collect i into l end finally (return l))", "(1 3)");
make_test ("(loop for i to 5 if (oddp i) if i collect it)", "(1 3 5)");
make_test ("(loop for i to 5 if (= i 3) return it)", "T");
make_test ("(loop for i in nil for j = (+ \"\"))", "NIL");
make_test ("(loop for i t = 0 do (write i) (return nil))", "0\nNIL");
make_test ("(let ((i 0)) (loop when (= i 0) do (write 'stop) :and do (return nil)))", "STOP\nNIL");
make_test ("(loop initially (write 'hello) for i from 1 to 5 finally (write 'world) do (write i))", "HELLO12345WORLD\nNIL");
make_test ("(loop for i in nil initially (write 'hello) finally (write 'world))", "HELLOWORLD\nNIL");
make_test ("(loop for i to 2 do (write i) initially (write i))", "0012\nNIL");
make_test ("(loop for i from 1 to 5 do (write i))", "12345\nNIL");
make_test ("(loop for i of-type fixnum to 5 do (write i))", "012345\nNIL");
make_test ("(loop for i from 1 below 5 do (write i))", "1234\nNIL");
make_test ("(let ((j 4)) (loop for i below j))", "NIL");
make_test ("(loop for i from 10 downto 5 by 2 do (write i))", "1086\nNIL");
make_test ("(loop for i upfrom 5 to 10 do (write i))", "5678910\nNIL");
make_test ("(loop for i from 5 upto 11 for j downfrom 20 above 15 do (write i) (write j))", "520619718817916\nNIL");
make_test ("(loop for i downfrom 5 to 1 by 2 do (write i))", "531\nNIL");
make_test ("(loop named foo for i from 0 do (write i) (if (> i 4) (return-from foo 0)))", "012345\n0");
make_test ("(loop as i from 0 do (if (> i 5) (return 10) (write i)) finally (write 'finished))", "012345\n10");
make_test ("(loop for i in '(0 1 2) do (write i))", "012\nNIL");
make_test ("(loop for i in '(0 1 2) for j = i do (write j))", "012\nNIL");
make_test ("(loop for i in '(0) with c = '(0 . 1) with (car . cdr) = c do (write cdr))", "1\nNIL");
make_test ("(loop for (i j) in '((0) (1) (2)) do (write i))", "012\nNIL");
make_test ("(loop for (i . j) in '((0 . 1) (1 . 2)) do (write j))", "12\nNIL");
make_test ("(loop for (nil . i) in '((0 . 1) (2 . 3)) do (write i))", "13\nNIL");
make_test ("(loop for (i . j) in nil finally (write 'done))", "DONE\nNIL");
make_test ("(loop for (i . j) in nil do (write j) finally (write j))", "NIL\nNIL");
make_test ("(loop for i in '(0 1 2 3) by #'cddr do (write i))", "02\nNIL");
make_test ("(loop for i from 10 downto 2 collect i)", "(10 9 8 7 6 5 4 3 2)");
make_test ("(loop :for i :in '(0 1 2) :when (= i 2) :collect i finally (write 'done))", "DONE\n(2)");
make_test ("(loop :for i :in '(0 1 2) :when (= i 2) :collect i :into out :and :collect i)", "(2)");
make_test ("(loop for (x . y) in '((0 . 1) (2 . 3)) collect x collect y)", "(0 1 2 3)");
make_test ("(loop for i in '((0 1) (2 3)) append i into l finally (return l))", "(0 1 2 3)");
make_test ("(loop for i on nil do (write i))", "NIL");
make_test ("(loop for i on '(0 1 2) do (write i) finally (write i))", "(0 1 2)(1 2)(2)NIL\nNIL");
make_test ("(loop for i on '(0 1 2 3) do (write i))", "(0 1 2 3)(1 2 3)(2 3)(3)\nNIL");
make_test ("(loop for i on '(0 1 2 3 4) by #'cddr do (write i))", "(0 1 2 3 4)(2 3 4)(4)\nNIL");
make_test ("(loop for i = 0 then (incf i) do (write i) if (= i 4) return 10)", "01234\n10");
make_test ("(let ((j 0)) (loop as i = (incf j) do (write i) if (= i 5) return 11))", "12345\n11");
make_test ("(loop for c across \"abc\" do (write c))", "#\\a#\\b#\\c\nNIL");
make_test ("(loop repeat (+ 3) do (write 'hi) finally (write 'done))", "HIHIHIDONE\nNIL");
make_test ("(loop for i from 0 do (write i) always (< i 4) finally (write 'done))", "01234\nNIL");
make_test ("(loop for i from 0 to 3 do (write i) always (< i 4) finally (write 'done))", "0123DONE\nT");
make_test ("(loop for i from 0 do (write i) never (> i 3) finally (write 'done))", "01234\nNIL");
make_test ("(loop for i from 0 to 3 do (write i) never (> i 3) finally (write 'done))", "0123DONE\nT");
make_test ("(loop with i = 0 do (write i) while (< i 5) do (incf i) finally (write 'done))", "012345DONE\nNIL");
make_test ("(loop with i = 0 do (write i) until (>= i 5) do (incf i) finally (write 'done))", "012345DONE\nNIL");
make_test ("(loop for x = 0 then nil while x)", "NIL");
make_test ("(loop for (x y . z) = '(0 1 . 2) then nil while x)", "NIL");
make_test ("(loop for i to 5 summing i)", "15");
make_test ("(loop for i to 5 sum i into ret finally (return ret))", "15");
make_test ("(loop for (x . y) in '((0 . 1) (2 . 3)) sum x sum y)", "6");
make_test ("(defparameter tbl (make-hash-table))", "TBL");
make_test ("(setf (gethash 1 tbl) 5)", "5");
make_test ("(setf (gethash 2 tbl) 6)", "6");
make_test ("(setf (gethash 3 tbl) 7)", "7");
make_test ("(loop for k being each hash-key in tbl do (write k))", "321\nNIL");
make_test ("(loop for k being each hash-key in (make-hash-table) do (write k))", "NIL");
make_test ("(loop for v being the hash-values in tbl using (hash-key k) do (write v) (write k))", "736251\nNIL");
make_test ("(format t \"aaa\")", "aaa\nNIL");
make_test ("(format t \"aaa~~\")", "aaa~\nNIL");
make_test ("(format t \"aaa~%~%\")", "aaa\n\nNIL");
make_test ("(format t \"aa~&~&bb\")", "aa\nbb\nNIL");
make_test ("(format t \"the number is ~a and the list is ~s\" 10 '(1 2 3))", "the number is 10 and the list is (1 2 3)\nNIL");
make_test ("(format nil \"the number is ~a and the list is ~s\" 10 '(1 2 3))", "\"the number is 10 and the list is (1 2 3)\"");
make_test ("(format nil \"~d~O~x ~8r\" 10 11 12 14)", "\"1013c 16\"");
make_test ("(format nil \"~{~a, ~d  ~} ~a\" '(a 0 b 1 c 2) 10)", "\"A, 0  B, 1  C, 2   10\"");
make_test ("(format nil \"~\@{~a.~a ~}\" 0 1 2 3)", "\"0.1 2.3 \"");
make_test ("(format nil \"~{~d~^.~} ~a\" '(1 2 3) 10)", "\"1.2.3 10\"");
make_test ("(format nil \"~a ~^ ~a\" 0)", "\"0 \"");
make_test ("(format nil \"~(~a ~a~a ~) ~a\" nil :hi 0 :hello)", "\"nil hi0  HELLO\"");
make_test ("(let ((s (make-string-output-stream))) (list (format s \"the number is ~a and the list is ~s\" 10 '(1 2 3)) (get-output-stream-string s)))", "(NIL \"the number is 10 and the list is (1 2 3)\")");
make_test ("(let ((s (make-string-output-stream \"abc\"))) (write 'def :stream s) (get-output-stream-string s))", "\"abcDEF\"");
make_test ("(let ((s (make-string-output-stream (make-array 6 :element-type 'character :initial-contents \"abcdef\" :fill-pointer 3)))) (write 'def :stream s) (write 'ghi :stream s) (get-output-stream-string s))", "\"abcDEFGHI\"");
make_test ("(open \"README\" :foo 'what :allow-other-keys t :allow-other-keys nil)", "#<STREAM FILE INPUT>");
make_test ("(with-open-file (s \"writetest2\" :direction :io) (write 'hello :stream s) (file-position s 0) (read s))", "HELLO");
make_test ("(with-open-file (s \"writetest3\" :direction :io) (list (input-stream-p s) (output-stream-p s)))", "(T T)");
make_test ("(with-open-file (s \"writetest3\" :direction :output :if-exists :overwrite))", "NIL");
make_test ("(defparameter str (open \"writetest\" :direction :output :direction :input))", "STR");
make_test ("(open \"writetest\" :direction :output :if-exists nil)", "NIL");
make_test ("(open \"writetest5\" :direction :output :if-does-not-exist :create)", "#<STREAM FILE OUTPUT>");
make_test ("(open \"bohboh\" :direction :output :if-exists :overwrite :if-does-not-exist nil)", "NIL");
make_test ("(delete-file \"writetest\")", "T");
make_test ("(delete-file \"writetest2\")", "T");
make_test ("(rename-file \"writetest3\" #p\"writetest4\")", "#P\"writetest4\"\n#P\"writetest3\"\n#P\"writetest4\"");
make_test ("(delete-file #p\"writetest4\")", "T");
make_test ("(delete-file #p\"writetest5\")", "T");
make_test ("(write \"hello\" :stream str)", "\"hello\"");
make_test ("(write-char #\\newline str)", "#\\Newline");
make_test ("(file-length str)", "8");
make_test ("(write #\\a :stream str :stream *standard-input*)", "#\\a");
make_test ("(write 1 :stream str)", "1");
make_test ("(write 1/2 :stream str)", "1/2");
make_test ("(write 0.1 :stream str)", "0.0999999999999999916733");
make_test ("(write (complex 1 1) :stream str)", "#C(1 1)");
make_test ("(write '(1 2 3) :stream str)", "(1 2 3)");
make_test ("(write #(1 2 3) :stream str)", "#(1 2 3)");
make_test ("(close str)", "T");
make_test ("(make-string-input-stream \"aaa\")", "#<STREAM STRING INPUT>");
make_test ("(read-line (make-string-input-stream \"abcdef\" 1 3))", "\"bc\"\nT");
make_test ("(read-line (make-string-input-stream \"abcdef\" 1))", "\"bcdef\"\nT");
make_test ("(read-line (make-string-input-stream \"abcdef\" 1 2))", "\"b\"\nT");
make_test ("(read-line (make-string-input-stream \"abcdef\" 1 6))", "\"bcdef\"\nT");
make_test ("(read-line (make-string-input-stream \"abcdef\" 1 nil))", "\"bcdef\"\nT");
make_test ("(read-line (make-string-input-stream \"\") nil 0)", "0\nT");
make_test ("(make-string-input-stream \"abc\" 1 1)", "#<STREAM STRING INPUT>");
make_test ("(defparameter str (make-string-output-stream))", "STR");
make_test ("(write 'aaa :stream str)", "AAA");
make_test ("(write \"bbb\" :stream str)", "\"bbb\"");
make_test ("(write 'ccc :stream str)", "CCC");
make_test ("(write #\\c :stream str)", "#\\c");
make_test ("(write 100 :stream str)", "100");
make_test ("(write 1.2 :stream str)", "1.19999999999999995559");
make_test ("(get-output-stream-string str)", "\"AAA\\\"bbb\\\"CCC#\\\\c1001.19999999999999995559\"");
make_test ("(get-output-stream-string str)", "\"\"");
make_test ("(let ((str (make-synonym-stream '*standard-output*))) (write 'hello :stream str) str)", "HELLO\n#<STREAM ?>");
make_test ("(defparameter mystream (make-string-input-stream \"hi\"))", "MYSTREAM");
make_test ("(let ((str (make-synonym-stream 'mystream))) (read str))", "HI");
make_test ("(let ((str (make-synonym-stream 'mystream))) (synonym-stream-symbol str))", "MYSTREAM");
make_test ("(typep (make-synonym-stream '*standard-output*) 'synonym-stream)", "T");
make_test ("(typep *standard-output* 'synonym-stream)", "NIL");
make_test ("(defparameter brstr (make-broadcast-stream *standard-output* (make-string-output-stream)))", "BRSTR");
make_test ("(write 'hi :stream brstr)", "HI\nHI");
make_test ("(write 0 :stream brstr)", "0\n0");
make_test ("(broadcast-stream-streams brstr)", "(#<STREAM FILE OUTPUT> #<STREAM STRING OUTPUT>)");
make_test ("(get-output-stream-string (cadr (broadcast-stream-streams brstr)))", "\"HI0\"");
make_test ("(with-input-from-string (s \"abc def\" :start 1) (read s))", "BC");
make_test ("(let (ind) (with-input-from-string (s \"abc def\" :start 1 :index ind) (read s)) ind)", "4");
make_test ("(with-output-to-string (s) (format s \"hello\") (write 0 :stream s))", "\"hello0\"");
make_test ("(with-output-to-string (s (make-array 6 :element-type 'character :fill-pointer 3 :initial-contents \"abcdef\")) (format s \"hello\") (write 0 :stream s) (get-output-stream-string s))", "\"abchello0\"");
make_test ("(gentemp)", "T1");
make_test ("(gentemp \"S\")", "S2");
make_test ("'s3", "S3");
make_test ("(gentemp \"S\")", "S4");
make_test ("(second '(0))", "NIL");
make_test ("(fifth '(0 1 2 3 4))", "4");
make_test ("(make-list 3)", "(NIL NIL NIL)");
make_test ("(make-list 4 :initial-element 0)", "(0 0 0 0)");
make_test ("(copy-alist '((1 . 10) (2 . \"a\") (3 . 'a)))", "((1 . 10) (2 . \"a\") (3 QUOTE A))");
make_test ("(copy-tree '(1 (2 (3 4 (5)) 6) 7))", "(1 (2 (3 4 (5)) 6) 7)");
make_test ("(tree-equal '(1 (2 (3))) '(1 (2)))", "NIL");
make_test ("(tree-equal '(1 (2 (3))) '(1 (2 (3))) :test 'equal)", "T");
make_test ("(sublis '((1 . -1) (2 . -2)) '(1 2 (3 4 2 (1))))", "(-1 -2 (3 4 -2 (-1)))");
make_test ("(nsublis '((1 . -1) (2 . -2)) '(1 2 (3 4 2 (1))))", "(-1 -2 (3 4 -2 (-1)))");
make_test ("(endp '(1 . 2))", "NIL");
make_test ("(endp ())", "T");
make_test ("(butlast '(0 1 2))", "(0 1)");
make_test ("(butlast '(0 1 2) 3)", "NIL");
make_test ("(butlast '(0 1 2 . 3) 2)", "(0)");
make_test ("(nbutlast '(0 1 2))", "(0 1)");
make_test ("(nbutlast '(0 1 2) 3)", "NIL");
make_test ("(nbutlast '(0 1 2 . 3) 2)", "(0)");
make_test ("(acons 1 \"\" '((2 . \"\")))", "((1 . \"\") (2 . \"\"))");
make_test ("(pairlis '(1 2 3) '(\"a\" \"b\" \"c\") '((4 . \"d\")))", "((3 . \"c\") (2 . \"b\") (1 . \"a\") (4 . \"d\"))");
make_test ("(when t \"\" \"\" \"\")", "\"\"");
make_test ("(when nil \"\" \"\" \"\")", "NIL");
make_test ("(unless nil \"\" \"\" \"\")", "\"\"");
make_test ("(the integer 0)", "0");
make_test ("(the integer (values 0 1))", "0\n1");
make_test ("(prog1 2 (write \"hello\"))", "\"hello\"\n2");
make_test ("(prog2 2 3 (write \"hello\"))", "\"hello\"\n3");
make_test ("(multiple-value-prog1 (values 0 1) 2)", "0\n1");
make_test ("(destructuring-bind ((x)) '((0)) x)", "0");
make_test ("(let ((z 0)) (destructuring-bind (i) (list z) (setq z 0)))", "0");
make_test ("(let ((z 0)) (destructuring-bind (i) (list z) (setq z (+ i 1))))", "1");
make_test ("(destructuring-bind (x (y (z))) '(0 (1 (2))) (list x y z))", "(0 1 2)");
make_test ("(destructuring-bind (x (y z) w) '(0 ((1 2) 3) 4) (list x y z w))", "(0 (1 2) 3 4)");
make_test ("(destructuring-bind (x . y) '(0 1 2 3) (list x y))", "(0 (1 2 3))");
make_test ("(al-loopy-destructuring-bind (x y) '(0 1 2) y)", "1");
make_test ("(al-loopy-destructuring-bind (x y z) '(0 1) z)", "NIL");
make_test ("(al-loopy-destructuring-bind (nil) '(0))", "NIL");
make_test ("(al-loopy-destructuring-bind (x nil) '(0) x)", "0");
make_test ("(al-loopy-destructuring-bind (x (y)) '(0 (1)) (list x y))", "(0 1)");
make_test ("(al-loopy-destructuring-bind (x . y) '(0 . 1) (list x y))", "(0 1)");
make_test ("(al-loopy-destructuring-bind (x . y) '(nil) y)", "NIL");
make_test ("(al-loopy-destructuring-bind ((x y)) '(nil) x)", "NIL");
make_test ("(al-loopy-destructuring-bind (x (y z)) '(nil) y)", "NIL");
make_test ("(let (i) (al-loopy-setq i 0) i)", "0");
make_test ("(let ((i 0)) (al-loopy-setq i (+ i 1)) i)", "1");
make_test ("(let (x y) (al-loopy-setq (x y) '(0 1 2)) y)", "1");
make_test ("(let (x y z) (al-loopy-setq (x y z) '(0 1)) z)", "NIL");
make_test ("(al-loopy-setq (nil) '(0))", "NIL");
make_test ("(let (x) (al-loopy-setq (x nil) '(0)) x)", "0");
make_test ("(let (x y) (al-loopy-setq (x (y)) '(0 (1))) (list x y))", "(0 1)");
make_test ("(let (x y) (al-loopy-setq (x . y) '(0 . 1)) (list x y))", "(0 1)");
make_test ("(let (x y) (al-loopy-setq (x . y) '(nil)) y)", "NIL");
make_test ("(let (x y) (al-loopy-setq ((x y)) '(nil)) x)", "NIL");
make_test ("(let (x y z) (al-loopy-setq (x (y z)) '(nil)) y)", "NIL");
make_test ("'make-ship", "MAKE-SHIP");
make_test ("(defstruct ship x y)", "SHIP");
make_test ("(defstruct ship \"A ship.\" x y)", "SHIP");
make_test ("#'make-ship", "#<FUNCTION MAKE-SHIP>");
make_test ("#'ship-x", "#<FUNCTION SHIP-X>");
make_test ("(defparameter s1 (make-ship))", "S1");
make_test ("(defparameter s1 (make-ship))", "S1");
make_test ("s1", "#<STRUCTURE OF CLASS SHIP>");
make_test ("(class-of s1)", "#<STRUCTURE CLASS SHIP>");
make_test ("(class-name (class-of s1))", "SHIP");
make_test ("(type-of s1)", "SHIP");
make_test ("(typep s1 'ship)", "T");
make_test ("(typep 0 'ship)", "NIL");
make_test ("(ship-p s1)", "T");
make_test ("(ship-p 0)", "NIL");
make_test ("(ship-x s1)", "NIL");
make_test ("(ship-y s1)", "NIL");
make_test ("(setf (ship-x s1) 0)", "0");
make_test ("(setf (ship-y s1) 1)", "1");
make_test ("(ship-x s1)", "0");
make_test ("(ship-y s1)", "1");
make_test ("(setf (ship-x s1) 2 (ship-y s1) 3)", "3");
make_test ("(ship-x (copy-ship s1))", "2");
make_test ("(ship-y (copy-ship s1))", "3");
make_test ("(defparameter s1 (make-ship :y 0 :y 1 :x 2))", "S1");
make_test ("(ship-x s1)", "2");
make_test ("(ship-y s1)", "0");
make_test ("(ship-x #s(ship :x a :y b))", "A");
make_test ("(ship-y #s(ship :x a :y b))", "B");
make_test ("(equalp #s(ship :x \"a\" :y \"b\") 0)", "NIL");
make_test ("(equalp #s(ship :x \"a\" :y \"b\") #s(ship :x \"a\" :y \"c\"))", "NIL");
make_test ("(equalp #s(ship :x \"a\" :y \"b\") #s(ship :x \"a\" :y \"B\"))", "T");
make_test ("(defstruct (ship2 (:predicate is-ship) (:copier copy-the-ship)) x y)", "SHIP2");
make_test ("(defparameter s2 (make-ship2))", "S2");
make_test ("(is-ship s2)", "T");
make_test ("(ship2-x (copy-the-ship s2))", "NIL");
make_test ("(defclass starship () (x y speed))", "#<STANDARD-CLASS STARSHIP>");
make_test ("(defclass starship () (x y speed))", "#<STANDARD-CLASS STARSHIP>");
make_test ("(find-class 'starship)", "#<STANDARD-CLASS STARSHIP>");
make_test ("(typep (find-class 'starship) 'standard-class)", "T");
make_test ("(typep (make-instance 'starship) (find-class 'starship))", "T");
make_test ("(typep 0 (find-class 'starship))", "NIL");
make_test ("(find-class 'blah nil)", "NIL");
make_test ("(defparameter inst (make-instance 'starship))", "INST");
make_test ("inst", "#<STARSHIP OBJECT ...>");
make_test ("(class-of inst)", "#<STANDARD-CLASS STARSHIP>");
make_test ("(class-name (class-of inst))", "STARSHIP");
make_test ("(type-of inst)", "STARSHIP");
make_test ("(typep inst 'starship)", "T");
make_test ("(typep inst 'standard-object)", "T");
make_test ("(typep 0 'starship)", "NIL");
make_test ("(slot-exists-p inst 'x)", "T");
make_test ("(slot-exists-p inst 'z)", "NIL");
make_test ("(slot-boundp inst 'x)", "NIL");
make_test ("(setf (slot-value inst 'x) 0)", "0");
make_test ("(setf (slot-value inst 'y) 2.5)", "2.5");
make_test ("(slot-value inst 'x)", "0");
make_test ("(slot-value inst 'y)", "2.5");
make_test ("(slot-boundp inst 'x)", "T");
make_test ("(slot-makunbound inst 'x)", "#<STARSHIP OBJECT ...>");
make_test ("(slot-boundp inst 'x)", "NIL");
make_test ("(defclass animal nil (leg-number))", "#<STANDARD-CLASS ANIMAL>");
make_test ("(defclass dog (animal) (hair-length))", "#<STANDARD-CLASS DOG>");
make_test ("(subtypep (find-class 'dog) (find-class 'animal))", "T\nT");
make_test ("(subtypep (find-class 'dog) (find-class 'dog))", "T\nT");
make_test ("(subtypep (find-class 'animal) (find-class 'dog))", "NIL\nT");
make_test ("(subtypep (find-class 'starship) (find-class 'dog))", "NIL\nT");
make_test ("(defparameter d (make-instance 'dog))", "D");
make_test ("(setf (slot-value d 'leg-number) 4)", "4");
make_test ("(slot-value d 'leg-number)", "4");
make_test ("(setf (slot-value d 'hair-length) 'short)", "SHORT");
make_test ("(slot-value d 'hair-length)", "SHORT");
make_test ("(setq d 0)", "0");
make_test ("(defclass car-vehicle () ((speed :initform (+)) (gear :initform (+ 1))))", "#<STANDARD-CLASS CAR-VEHICLE>");
make_test ("(defparameter cv (make-instance 'car-vehicle))", "CV");
make_test ("(slot-boundp cv 'speed)", "T");
make_test ("(slot-value cv 'speed)", "0");
make_test ("(slot-value cv 'gear)", "1");
make_test ("(defclass martian () ((legs :initarg :legsnum :initform (+)) (arms :initarg arms :initform 4)))", "#<STANDARD-CLASS MARTIAN>");
make_test ("(defparameter inst (make-instance 'martian 'arms 2))", "INST");
make_test ("(slot-value inst 'arms)", "2");
make_test ("(slot-value inst 'legs)", "0");
make_test ("(defparameter inst (make-instance 'martian :legsnum 5))", "INST");
make_test ("(slot-value inst 'legs)", "5");
make_test ("(slot-value inst 'arms)", "4");
make_test ("(defparameter inst2 (allocate-instance (find-class 'martian)))", "INST2");
make_test ("(slot-exists-p inst2 'legs)", "T");
make_test ("(slot-exists-p inst2 'arms)", "T");
make_test ("(slot-boundp inst2 'legs)", "NIL");
make_test ("(slot-boundp inst2 'arms)", "NIL");
make_test ("(initialize-instance inst2)", "#<MARTIAN OBJECT ...>");
make_test ("(slot-boundp inst2 'legs)", "T");
make_test ("(slot-boundp inst2 'arms)", "T");
make_test ("(defparameter inst3 (make-instance 'martian :legsnum 1 :legsnum 2))", "INST3");
make_test ("(slot-value inst3 'legs)", "1");
make_test ("(defclass cl nil ((sl :initarg :sl1 :initarg :sl2)))", "#<STANDARD-CLASS CL>");
make_test ("(slot-value (make-instance 'cl :sl1 10) 'sl)", "10");
make_test ("(slot-value (make-instance 'cl :sl2 11) 'sl)", "11");
make_test ("#'make-instance", "#<STANDARD-GENERIC-FUNCTION MAKE-INSTANCE>");
make_test ("#'allocate-instance", "#<STANDARD-GENERIC-FUNCTION ALLOCATE-INSTANCE>");
make_test ("#'initialize-instance", "#<STANDARD-GENERIC-FUNCTION INITIALIZE-INSTANCE>");
make_test ("(defclass martian () ((legs :initform (+) :documentation \"Number of legs\")))", "#<STANDARD-CLASS MARTIAN>");
make_test ("(make-instance 'martian)", "#<MARTIAN OBJECT ...>");
make_test ("(defclass cl nil ((sl1 :reader sl1-cl :writer wrsl1-cl) (sl2 :reader sl2-cl :accessor accsl2-cl)))", "#<STANDARD-CLASS CL>");
make_test ("#'sl1-cl", "#<STANDARD-GENERIC-FUNCTION SL1-CL>");
make_test ("#'wrsl1-cl", "#<STANDARD-GENERIC-FUNCTION WRSL1-CL>");
make_test ("#'accsl2-cl", "#<STANDARD-GENERIC-FUNCTION ACCSL2-CL>");
make_test ("#'(setf accsl2-cl)", "#<STANDARD-GENERIC-FUNCTION (SETF ACCSL2-CL)>");
make_test ("(defparameter inst (make-instance 'cl))", "INST");
make_test ("(wrsl1-cl 10 inst)", "10");
make_test ("(sl1-cl inst)", "10");
make_test ("(setf (accsl2-cl inst) 20)", "20");
make_test ("(accsl2-cl inst)", "20");
make_test ("(sl2-cl inst)", "20");
make_test ("(defclass cl nil ((sl :writer (setf sl-cl))))", "#<STANDARD-CLASS CL>");
make_test ("(defparameter inst (make-instance 'cl))", "INST");
make_test ("(setf (sl-cl inst) 30)", "30");
make_test ("(slot-value inst 'sl)", "30");
make_test ("(defclass cl nil ((sl :reader rcl1 :writer wcl1 :reader rcl2 :writer wcl2)))", "#<STANDARD-CLASS CL>");
make_test ("(defparameter inst (make-instance 'cl))", "INST");
make_test ("(wcl1 40 inst)", "40");
make_test ("(rcl1 inst)", "40");
make_test ("(wcl2 50 inst)", "50");
make_test ("(rcl2 inst)", "50");
make_test ("(defclass cl2 nil ((f1 :type string)))", "#<STANDARD-CLASS CL2>");
make_test ("(make-instance 'cl2)", "#<CL2 OBJECT ...>");
make_test ("(defclass cl3 nil ((sl :initarg :slcl3 :initform (+) :allocation :class)))", "#<STANDARD-CLASS CL3>");
make_test ("(defparameter inst (make-instance 'cl3 :slcl3 10))", "INST");
make_test ("(defparameter inst2 (make-instance 'cl3 :slcl3 11))", "INST2");
make_test ("(slot-value inst 'sl)", "11");
make_test ("(slot-value inst2 'sl)", "11");
make_test ("(defparameter inst3 (make-instance 'cl3))", "INST3");
make_test ("(slot-value inst 'sl)", "11");
make_test ("(slot-value inst2 'sl)", "11");
make_test ("(slot-value inst3 'sl)", "11");
make_test ("(defparameter inst4 (make-instance 'cl3 :slcl3 20 :slcl3 21))", "INST4");
make_test ("(slot-value inst4 'sl)", "20");
make_test ("(defclass cl4 nil ((sl :initarg :sl)))", "#<STANDARD-CLASS CL4>");
make_test ("(defclass cl5 nil (sl))", "#<STANDARD-CLASS CL5>");
make_test ("(defclass cl6 (cl4 cl5) nil)", "#<STANDARD-CLASS CL6>");
make_test ("(slot-value (make-instance 'cl6 :sl 10) 'sl)", "10");
make_test ("(al-class-precedence-list (find-class 'cl6))", "(#<STANDARD-CLASS CL6> #<STANDARD-CLASS CL4> #<STANDARD-CLASS CL5> #<STANDARD-CLASS STANDARD-OBJECT>)");
make_test ("(defclass cl7 (cl8) (sl1))", "#<STANDARD-CLASS CL7>");
make_test ("(defclass cl8 nil (sl2))", "#<STANDARD-CLASS CL8>");
make_test ("(slot-exists-p (make-instance 'cl7) 'sl1)", "T");
make_test ("(slot-exists-p (make-instance 'cl7) 'sl2)", "T");
make_test ("(defclass c9 nil nil)", "#<STANDARD-CLASS C9>");
make_test ("(defclass c10 (c9) nil)", "#<STANDARD-CLASS C10>");
make_test ("(defclass c11 (c9) nil)", "#<STANDARD-CLASS C11>");
make_test ("(defclass c12 (c10 c11) nil)", "#<STANDARD-CLASS C12>");
make_test ("(make-instance 'c12)", "#<C12 OBJECT ...>");
make_test ("(defclass c13 nil nil)", "#<STANDARD-CLASS C13>");
make_test ("(defclass c14 (c13) nil)", "#<STANDARD-CLASS C14>");
make_test ("(defclass c15 (c13) nil)", "#<STANDARD-CLASS C15>");
make_test ("(defclass c16 (c14 c15) nil)", "#<STANDARD-CLASS C16>");
make_test ("(defclass c17 nil nil)", "#<STANDARD-CLASS C17>");
make_test ("(defclass c18 (c16 c17) nil)", "#<STANDARD-CLASS C18>");
make_test ("(make-instance 'c18)", "#<C18 OBJECT ...>");
make_test ("(defclass c19 nil (x y z))", "#<STANDARD-CLASS C19>");
make_test ("(defparameter inst (make-instance 'c19))", "INST");
make_test ("(with-slots (y (z1 z)) inst (setf y 10) (write y) (setf z1 11) (write z1))", "1011\n11");
make_test ("(slot-value inst 'z)", "11");
make_test ("(defclass c20 nil ((x :accessor c20-x) (y :accessor c20-y) z))", "#<STANDARD-CLASS C20>");
make_test ("(defparameter inst (make-instance 'c20))", "INST");
make_test ("(with-accessors ((x1 c20-x) (y1 c20-y)) inst (setf x1 20) (write x1))", "20\n20");
make_test ("(c20-x inst)", "20");
make_test ("(defclass c21 nil ((f :reader fr :writer fw :accessor fa :allocation :class)))", "#<STANDARD-CLASS C21>");
make_test ("(defparameter inst (make-instance 'c21))", "INST");
make_test ("(setf (fa inst) 10)", "10");
make_test ("(fr inst)", "10");
make_test ("(fw 11 inst)", "11");
make_test ("(fa inst)", "11");
make_test ("(defclass c22 nil (x))", "#<STANDARD-CLASS C22>");
make_test ("(defclass c23 (c22) ((x :initform 0)))", "#<STANDARD-CLASS C23>");
make_test ("(defparameter inst (make-instance 'c23))", "INST");
make_test ("(slot-value inst 'x)", "0");
make_test ("(defclass c24 nil (w x y))", "#<STANDARD-CLASS C24>");
make_test ("(defparameter inst (make-instance 'c24))", "INST");
make_test ("(setf (slot-value inst 'y) 0)", "0");
make_test ("(defclass c25 nil (x (y :initform 1) (z :initform 2)))", "#<STANDARD-CLASS C25>");
make_test ("(change-class inst (find-class 'c25))", "#<C25 OBJECT ...>");
make_test ("(slot-boundp inst 'x)", "NIL");
make_test ("(slot-value inst 'y)", "0");
make_test ("(slot-value inst 'z)", "2");
make_test ("(class-of inst)", "#<STANDARD-CLASS C25>");
make_test ("(defclass c26 nil (x y))", "#<STANDARD-CLASS C26>");
make_test ("(defparameter inst (make-instance 'c26))", "INST");
make_test ("(setf (slot-value inst 'x) 10)", "10");
make_test ("(defclass c27 nil ((x :initarg :x) (y :initarg :y)))", "#<STANDARD-CLASS C27>");
make_test ("(change-class inst 'c27 :x 20 :y 21)", "#<C27 OBJECT ...>");
make_test ("(slot-value inst 'x)", "20");
make_test ("(slot-value inst 'y)", "21");
make_test ("(defclass c28 nil ((x :initarg :x :initform 0) (y :initform 1)))", "#<STANDARD-CLASS C28>");
make_test ("(defparameter inst (make-instance 'c28))", "INST");
make_test ("(setf (slot-value inst 'y) 10)", "10");
make_test ("(reinitialize-instance inst :x 1)", "#<C28 OBJECT ...>");
make_test ("(slot-value inst 'x)", "1");
make_test ("(slot-value inst 'y)", "10");
make_test ("(defclass c29  () (x))", "#<STANDARD-CLASS C29>");
make_test ("(defclass c30 (c29) nil)", "#<STANDARD-CLASS C30>");
make_test ("(defclass c31 () (y))", "#<STANDARD-CLASS C31>");
make_test ("(defparameter inst (make-instance 'c30))", "INST");
make_test ("(change-class inst 'c31)", "#<C31 OBJECT ...>");
make_test ("(defclass c32 nil ((sl :initarg :sl)))", "#<STANDARD-CLASS C32>");
make_test ("(defclass c33 (c32) (sl))", "#<STANDARD-CLASS C33>");
make_test ("(slot-value (make-instance 'c33 :sl 0) 'sl)", "0");
make_test ("(defgeneric genfun (x y z))", "#<STANDARD-GENERIC-FUNCTION GENFUN>");
make_test ("(typep #'genfun 'generic-function)", "T");
make_test ("(typep #'genfun 'standard-generic-function)", "T");
make_test ("(typep #'car 'generic-function)", "NIL");
make_test ("(typep #'car 'standard-generic-function)", "NIL");
make_test ("(defmethod genfun ((x string) y z) 10)", "#<STANDARD-METHOD GENFUN (STRING T T)>");
make_test ("(defmethod genfun ((x integer) y z) 11)", "#<STANDARD-METHOD GENFUN (INTEGER T T)>");
make_test ("(genfun \"\" nil nil)", "10");
make_test ("(genfun 0 nil nil)", "11");
make_test ("(defgeneric genfun (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN>");
make_test ("(defmethod genfun2 ((x string) y z) 12)", "#<STANDARD-METHOD GENFUN2 (STRING T T)>");
make_test ("(defmethod genfun2 ((x integer) y z) 13)", "#<STANDARD-METHOD GENFUN2 (INTEGER T T)>");
make_test ("(defgeneric genfun3 (x y z))", "#<STANDARD-GENERIC-FUNCTION GENFUN3>");
make_test ("(defmethod genfun3 ((x string) (y number) z) (write 'first))", "#<STANDARD-METHOD GENFUN3 (STRING NUMBER T)>");
make_test ("(defmethod genfun3 ((x string) (y integer) z) (write 'second))", "#<STANDARD-METHOD GENFUN3 (STRING INTEGER T)>");
make_test ("(defmethod genfun3 ((x string) (y real) z) (write 'third))", "#<STANDARD-METHOD GENFUN3 (STRING REAL T)>");
make_test ("(genfun3 \"\" 0 nil)", "SECOND\nSECOND");
make_test ("(defmethod genfun3 ((x string) (y integer) (z symbol)) (write 'fourth))", "#<STANDARD-METHOD GENFUN3 (STRING INTEGER SYMBOL)>");
make_test ("(genfun3 \"\" 0 nil)", "FOURTH\nFOURTH");
make_test ("(al-dump-methods #'genfun3)", "(#<STANDARD-METHOD GENFUN3 (STRING NUMBER T)> #<STANDARD-METHOD GENFUN3 (STRING INTEGER T)> #<STANDARD-METHOD GENFUN3 (STRING REAL T)> #<STANDARD-METHOD GENFUN3 (STRING INTEGER SYMBOL)>)");
make_test ("(defgeneric genfun4 (x y z))", "#<STANDARD-GENERIC-FUNCTION GENFUN4>");
make_test ("(defmethod genfun4 ((x string) (y number) z) (if (next-method-p) (write 'okay)) (list x y z))", "#<STANDARD-METHOD GENFUN4 (STRING NUMBER T)>");
make_test ("(defmethod genfun4 ((x string) (y real) z) (call-next-method \"hi\" 30 z))", "#<STANDARD-METHOD GENFUN4 (STRING REAL T)>");
make_test ("(defmethod genfun4 ((x string) (y integer) z) (setq z 10) (if (next-method-p) (write 'okay)) (call-next-method))", "#<STANDARD-METHOD GENFUN4 (STRING INTEGER T)>");
make_test ("(genfun4 \"what\" 0 0)", "OKAY\n(\"hi\" 30 0)");
make_test ("(defgeneric genfun5 (x y))", "#<STANDARD-GENERIC-FUNCTION GENFUN5>");
make_test ("(ensure-generic-function 'genfun5 :lambda-list '(j k))", "#<STANDARD-GENERIC-FUNCTION GENFUN5>");
make_test ("(defmethod genfun5 ((x string) (y (eql (+)))) (write 'first))", "#<STANDARD-METHOD GENFUN5 (STRING (EQL 0))>");
make_test ("(defmethod genfun5 ((x string) (y integer)) (write 'second))", "#<STANDARD-METHOD GENFUN5 (STRING INTEGER)>");
make_test ("(genfun5 \"\" 0)", "FIRST\nFIRST");
make_test ("(genfun5 \"\" 1)", "SECOND\nSECOND");
make_test ("(ensure-generic-function 'genfun6 :lambda-list '(x))", "#<STANDARD-GENERIC-FUNCTION GENFUN6>");
make_test ("(ensure-generic-function 'genfun6 :lambda-list '(x))", "#<STANDARD-GENERIC-FUNCTION GENFUN6>");
make_test ("(defgeneric genfun7 (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN7>");
make_test ("(defmethod genfun7 :around ((x integer)) (write 'first) (call-next-method))", "#<STANDARD-METHOD GENFUN7 :AROUND (INTEGER)>");
make_test ("(defmethod genfun7 :around ((x real)) (write 'second) (call-next-method))", "#<STANDARD-METHOD GENFUN7 :AROUND (REAL)>");
make_test ("(defmethod genfun7 :before ((x integer)) (write 'third))", "#<STANDARD-METHOD GENFUN7 :BEFORE (INTEGER)>");
make_test ("(defmethod genfun7 :before ((x real)) (write 'fourth))", "#<STANDARD-METHOD GENFUN7 :BEFORE (REAL)>");
make_test ("(defmethod genfun7 ((x real)) (write 'fifth))", "#<STANDARD-METHOD GENFUN7 (REAL)>");
make_test ("(defmethod genfun7 :after ((x integer)) (write 'sixth))", "#<STANDARD-METHOD GENFUN7 :AFTER (INTEGER)>");
make_test ("(defmethod genfun7 :after ((x real)) (write 'seventh))", "#<STANDARD-METHOD GENFUN7 :AFTER (REAL)>");
make_test ("(genfun7 0)", "FIRSTSECONDTHIRDFOURTHFIFTHSEVENTHSIXTH\nFIFTH");
make_test ("(defgeneric genfun8 (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN8>");
make_test ("(defmethod genfun8 :after ((x integer)) (write 'first))", "#<STANDARD-METHOD GENFUN8 :AFTER (INTEGER)>");
make_test ("(defmethod genfun8 :before ((x integer)) (write 'second))", "#<STANDARD-METHOD GENFUN8 :BEFORE (INTEGER)>");
make_test ("(defmethod genfun8 ((x integer)) (write 'third))", "#<STANDARD-METHOD GENFUN8 (INTEGER)>");
make_test ("(genfun8 0)", "SECONDTHIRDFIRST\nTHIRD");
make_test ("(find-method #'genfun8 nil '(integer))", "#<STANDARD-METHOD GENFUN8 (INTEGER)>");
make_test ("(find-method #'genfun8 '(:after) '(integer))", "#<STANDARD-METHOD GENFUN8 :AFTER (INTEGER)>");
make_test ("(function-lambda-expression (find-method #'genfun8 '(:after) '(integer)))", "((WRITE 'FIRST))");
make_test ("(find-method #'genfun8 '(:before) '(integer))", "#<STANDARD-METHOD GENFUN8 :BEFORE (INTEGER)>");
make_test ("(find-method #'genfun8 '(:around) '(integer))", "NIL");
make_test ("(find-method #'genfun8 '(:before) '(number))", "NIL");
make_test ("(defparameter meth (find-method #'genfun8 '(:before) '(integer)))", "METH");
make_test ("(remove-method #'genfun8 meth)", "#<STANDARD-GENERIC-FUNCTION GENFUN8>");
make_test ("(find-method #'genfun8 '(:before) '(integer))", "NIL");
make_test ("(add-method #'genfun8 meth)", "#<STANDARD-GENERIC-FUNCTION GENFUN8>");
make_test ("(find-method #'genfun8 '(:before) '(integer))", "#<STANDARD-METHOD GENFUN8 :BEFORE (INTEGER)>");
make_test ("(defgeneric genfun9 (x &key))", "#<STANDARD-GENERIC-FUNCTION GENFUN9>");
make_test ("(defmethod genfun9 (x &key y))", "#<STANDARD-METHOD GENFUN9 (T)>");
make_test ("(defgeneric (setf genfun10) (x y z))", "#<STANDARD-GENERIC-FUNCTION (SETF GENFUN10)>");
make_test ("(defmethod (setf genfun10) (x y z) (list x y z))", "#<STANDARD-METHOD (SETF GENFUN10) (T T T)>");
make_test ("(setf (genfun10 0 1) 2)", "(2 0 1)");
make_test ("(setf (genfun10 'x 'y) 'z)", "(Z X Y)");
make_test ("(defgeneric genfun11 (x y z))", "#<STANDARD-GENERIC-FUNCTION GENFUN11>");
make_test ("(defmethod genfun11 (x y z) (list x y z))", "#<STANDARD-METHOD GENFUN11 (T T T)>");
make_test ("(funcall #'genfun11 'x 'y 'z)", "(X Y Z)");
make_test ("(defgeneric genfun12 (x &key))", "#<STANDARD-GENERIC-FUNCTION GENFUN12>");
make_test ("(defmethod genfun12 ((x number) &key &allow-other-keys))", "#<STANDARD-METHOD GENFUN12 (NUMBER)>");
make_test ("(defmethod genfun12 ((x real) &key) (write 'here))", "#<STANDARD-METHOD GENFUN12 (REAL)>");
make_test ("(genfun12 0 :other 1)", "HERE\nHERE");
make_test ("(defgeneric genfun13 (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN13>");
make_test ("(defmethod genfun13 (x) (declare (special x)) (f))", "#<STANDARD-METHOD GENFUN13 (T)>");
make_test ("(defun f nil x)", "F");
make_test ("(genfun13 0)", "0");
make_test ("(defgeneric genfun14 (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN14>");
make_test ("(defmethod genfun14 ((x integer)) (genfun15 x) (call-next-method))", "#<STANDARD-METHOD GENFUN14 (INTEGER)>");
make_test ("(defmethod genfun14 ((x real)) 0)", "#<STANDARD-METHOD GENFUN14 (REAL)>");
make_test ("(defgeneric genfun15 (x))", "#<STANDARD-GENERIC-FUNCTION GENFUN15>");
make_test ("(defmethod genfun15 (x) 10)", "#<STANDARD-METHOD GENFUN15 (T)>");
make_test ("(genfun14 5)", "0");
make_test ("(defun fle (b &optional c &rest d &key e &allow-other-keys &aux f) 0 1 2)", "FLE");
make_test ("(function-lambda-expression #'fle)", "(LAMBDA (B &OPTIONAL C &REST D &KEY E &ALLOW-OTHER-KEYS &AUX F) 0 1 2)\nNIL\nFLE");
make_test ("(function-lambda-expression (let ((i 0)) (lambda (x) 0 1)))", "(LAMBDA (X) 0 1)\nT\nNIL");
make_test ("(define-condition new-math-error (arithmetic-error) (a b))", "NEW-MATH-ERROR");
make_test ("(define-condition new-math-error (arithmetic-error) (a b))", "NEW-MATH-ERROR");
make_test ("(make-condition 'new-math-error)", "#<NEW-MATH-ERROR OBJECT ...>");
make_test ("(class-of (make-condition 'new-math-error))", "#<STANDARD-CLASS NEW-MATH-ERROR>");
make_test ("(class-name (class-of (make-condition 'new-math-error)))", "NEW-MATH-ERROR");
make_test ("(type-of (make-condition 'new-math-error))", "NEW-MATH-ERROR");
make_test ("(typep (make-condition 'new-math-error) 'new-math-error)", "T");
make_test ("(typep (make-condition 'new-math-error) 'arithmetic-error)", "T");
make_test ("(typep (make-condition 'new-math-error) 'file-error)", "NIL");
make_test ("(typep 0 'new-math-error)", "NIL");
make_test ("(define-condition strange-condition nil nil)", "STRANGE-CONDITION");
make_test ("(typep (make-condition 'strange-condition) 'condition)", "T");
make_test ("(simple-condition-format-control (make-condition 'simple-condition))", "NIL");
make_test ("(simple-condition-format-control (make-condition 'simple-error))", "NIL");
make_test ("(simple-condition-format-control (make-condition 'simple-condition :format-control \"aaa\"))", "\"aaa\"");
make_test ("(arithmetic-error-operation (make-condition 'arithmetic-error))", "NIL");
make_test ("(define-compiler-macro foo (x) x)", "FOO");
make_test ("(funcall (compiler-macro-function 'foo) '(foo 0) nil)", "0");
make_test ("(compiler-macro-function 'bar)", "NIL");
make_test ("(define-compiler-macro foo (x) 10)", "FOO");
make_test ("(funcall (compiler-macro-function 'foo) '(foo 0) nil)", "10");
make_test ("(define-compiler-macro (setf foo) (x y) (list y x))", "(SETF FOO)");
make_test ("(funcall (compiler-macro-function '(setf foo)) '(setf (foo 10) 11) nil)", "(11 (FOO 10))");
make_test ("(declaim (optimize speed (safety 0) (debug 3)))", "T");
make_test ("(declaim (ignorable a) (ignore b c) (inline e) (notinline d f))", "T");
make_test ("(declaim (type list a) (ftype (function nil nil) b))", "T");
make_test ("(proclaim '(optimize speed (safety 0) (debug 3)))", "T");
make_test ("(proclaim '(ignorable a))", "T");
make_test ("(proclaim '(type list a))", "T");
make_test ("(tagbody\n" .
	   "  (write \"1\")\n" .
	   "  (go jmp)\n" .
	   "  (write \"2\")\n" .
	   "  jmp\n" .
	   "  (write \"3\"))", "\"1\"\"3\"\nNIL");
make_test ("(tagbody 1 (go 3) 2 3 (go 4) 4 (write \"\"))", "\"\"\nNIL");
make_test ("(tagbody 1 2 (tagbody 3 (go 4)) 4 (write \"\"))", "\"\"\nNIL");
make_test ("(tagbody (write 1) (block nil (write 2) (go c)) c (write 4))", "124\nNIL");
make_test ("(tagbody (go c) c)", "NIL");
make_test ("(block nil (tagbody (defun f nil (go a)) (go b) a (return 0) b (tagbody a (f))))", "0");
make_test ("(block test (write 1) (write 2) (return-from test 10) (write 3))", "12\n10");
make_test ("(block test (write 1) (block test2 (write 2) (return-from test 10) (write 3)))", "12\n10");
make_test ("(block test (write 1) (block test2 (write 2) (return-from test (values 10 11)) (write 3)))", "12\n10\n11");
make_test ("(block nil (return (values 11 12)))", "11\n12");
make_test ("(block b (do nil (nil) (return-from b (values 0 1 2))))", "0\n1\n2");
make_test ("(block nil (defun f nil (return 10)) (block nil (f)) 11)", "10");
make_test ("(block a (defun a nil) (block b (defun b nil) (block c (defun c nil))))", "C");
make_test ("(block a (block b (block c (block d (defun f nil (return-from b 10)))) (block e (block b (f))) 11))", "10");
make_test ("(defun test () (write 1) (write 2) (return-from test 10) (write 3))", "TEST");
make_test ("(test)", "12\n10");
make_test ("(defun (setf test2) (x y) (return-from test2 10))", "(SETF TEST2)");
make_test ("(setf (test2 0) 1)", "10");
make_test ("(defun f () (throw 'label (values 0 1 2)))", "F");
make_test ("(catch 'label (f))", "0\n1\n2");
make_test ("(block nil (handler-bind ((file-error (lambda (e) (write 0))) (division-by-zero (lambda (e) (return 1)))) (/ 1 0)))", "1");
make_test ("(block nil (handler-bind ((division-by-zero (lambda (e) (write 0))) (arithmetic-error (lambda (e) (write 1))) (error (lambda (e) (return 2)))) (/ 1 0)))", "01\n2");
make_test ("(block nil (handler-bind ((division-by-zero (lambda (e) (return 0)))) (handler-bind ((arithmetic-error (lambda (e) (write 1)))) (/ 1 0))))", "1\n0");
make_test ("(block nil (handler-bind ((type-error (lambda (e) (write 0) (return 1)))) (+ \"\")))", "0\n1");
make_test ("(restart-bind ((rest (lambda () (write 10)))) (invoke-restart 'rest) (write 11))", "1011\n11");
make_test ("(restart-bind ((foo (lambda (b) (write b)))) (invoke-restart 'foo 'c))", "C\nC");
make_test ("(restart-bind ((foo (lambda nil))) (restart-bind nil) (invoke-restart 'foo))", "NIL");
make_test ("(restart-bind ((foo (lambda nil 0)) (foo (lambda nil 1))) (invoke-restart 'foo))", "0");
make_test ("(restart-bind ((rest1 (lambda nil 10)) (rest2 (lambda nil 20)) (rest2 (lambda nil 30))) (invoke-restart 'rest2))", "20");
make_test ("(restart-bind ((foo (lambda nil (write-line \"hi\") (abort)))) (invoke-restart 'foo))", "hi");
make_test ("(continue)", "NIL");
make_test ("(compute-restarts)", "NIL");
make_test ("(restart-bind ((rest1 (lambda nil 10)) (rest2 (lambda (x) 20))) (write (compute-restarts)) 0)", "((REST1 . #<FUNCTION ?>) (REST2 . #<FUNCTION ?>))\n0");
make_test ("(restart-bind ((rest1 (lambda nil 10)) (rest2 (lambda (x) 20))) (find-restart 'rest1))", "(REST1 . #<FUNCTION ?>)");
make_test ("(restart-case (values 1 2) (rest1 (x y) 0) (rest2 (x) 10))", "1\n2");
make_test ("(restart-case (invoke-restart 'rest2 0 1) (rest1 (x) 10) (rest2 (x y) (values x y)) (rest2 (x) 10))", "0\n1");
make_test ("(restart-case (invoke-restart 'rest1 0 1) (rest1 (x y)) (rest1 (x y) 10))", "NIL");
make_test ("(restart-bind ((rest1 (lambda (x) 20))) (restart-case (invoke-restart 'rest2 0 1) (rest2 (x y) (invoke-restart 'rest1 40)) (rest1 (x y) 30)))", "20");
make_test ("(with-simple-restart (continue \"\") (values 1 2))", "1\n2");
make_test ("(with-simple-restart (continue \"\") 1 2 (invoke-restart 'continue))", "NIL\nT");
make_test ("(ignore-errors (error \"what\") (write 1))", "NIL\n#<SIMPLE-ERROR OBJECT ...>");
make_test ("(block test (unwind-protect 0 (write 'whatever)))", "WHATEVER\n0");
make_test ("(block test (unwind-protect (return-from test 0) (write 'whatever)))", "WHATEVER\n0");
make_test ("(block test (unwind-protect (return-from test (values 0 1)) (write 'whatever)))", "WHATEVER\n0\n1");
make_test ("(signal \"blah\")", "NIL");
make_test ("(signal 'type-error)", "NIL");
make_test ("(signal 'error)", "NIL");
make_test ("(signal (make-condition 'error))", "NIL");
make_test ("(warn \"something bad may happen\")", "emitted SIMPLE-WARNING: \"something bad may happen\"\nNIL");
make_test ("(handler-bind ((warning (lambda (e) (write 'warningignored) (abort)))) (warn \"something bad may happen\"))", "WARNINGIGNORED");
make_test ("(handler-bind ((simple-condition (lambda (e) (write 'oh)))) (signal \"blah\"))", "OH\nNIL");
make_test ("(handler-bind ((condition (lambda (e) (write 'ah))) (simple-condition (lambda (e) (write 'oh)))) (signal \"blah\"))", "AHOH\nNIL");
make_test ("(handler-bind ((condition (lambda (e) (write (signal \"w\"))))) (signal \"blah\"))", "NIL\nNIL");
make_test ("(block nil (handler-bind ((condition (lambda (e) (return 0)))) (signal \"blah\")))", "0");
make_test ("(handler-bind ((condition (lambda (e) (write (simple-condition-format-control e))))) (signal \"blah\"))", "\"blah\"\nNIL");
make_test ("(handler-bind ((arithmetic-error (lambda (e) (write 'oh)))) (signal 'arithmetic-error))", "OH\nNIL");
make_test ("(handler-bind (((or file-error division-by-zero) (lambda (e) (write 0)))) (signal 'division-by-zero))", "0\nNIL");
make_test ("(handler-case (/ 1 0) (arithmetic-error (e)))", "NIL");
make_test ("(handler-case (values 1 2) (arithmetic-error (e)))", "1\n2");
make_test ("(handler-case (/ 1 0) (division-by-zero nil (write 'first)) (arithmetic-error (e) (write 'second)) (:no-error nil 'noerror))", "FIRST\nFIRST");
make_test ("(handler-case (/ 1 0) (file-error nil (write 'first)) (arithmetic-error (e) (write 'second)) (:no-error nil 'noerror))", "SECOND\nSECOND");
make_test ("(handler-case (/ 1 1) (file-error nil (write 'first)) (arithmetic-error (e) (write 'second)) (:no-error (n) n))", "1");
make_test ("(handler-bind ((error (lambda (e) (write 0) (throw 'bl 1)))) (catch 'bl (error \"hi\")))", "0\n1");
make_test ("(handler-bind ((error (lambda (e) (write 0) (throw 'bl 1)))) (catch 'bl (cerror \"hi\" \"hi\")))", "0\n1");
make_test ("(handler-bind ((unbound-variable (lambda (e) (write 'undefined) (abort)))) undef)", "UNDEFINED");
make_test ("(handler-bind ((undefined-function (lambda (e) (write 'undefined) (abort)))) (foooo))", "UNDEFINED");
make_test ("(handler-bind ((undefined-function (lambda (e) (write 'undefined) (abort)))) (function foooo))", "UNDEFINED");
make_test ("(handler-bind ((undefined-function (lambda (e) (write 'undefined) (abort)))) #'(setf foooo))", "UNDEFINED");
make_test ("(handler-bind ((type-error (lambda (e) (write 'wrongtype) (abort)))) (car 0))", "WRONGTYPE");
make_test ("(handler-bind ((type-error (lambda (e) (write 'wrongtype) (abort)))) (pathname 0))", "WRONGTYPE");
make_test ("(handler-bind ((al-unknown-keyword-argument (lambda (e) (write 'invalidkeyword) (abort)))) (count 0 nil :foo 0))", "INVALIDKEYWORD");
make_test ("(handler-bind ((al-odd-number-of-arguments-in-keyword-part-of-form (lambda (e) (write 'oddargs) (abort)))) (write 0 :stream))", "ODDARGS");
make_test ("(handler-bind ((program-error (lambda (e) (write 'wrongnumargs) (abort)))) (car))", "WRONGNUMARGS");
make_test ("(handler-bind ((error (lambda (e) (write 'error) (abort)))) (find-class 'foo))", "ERROR");
make_test ("(cddr '(0 1 2))", "(2)");
make_test ("(cddddr '(0 1 2 3 4))", "(4)");
make_test ("(cadddr '(0 1 2 3 4))", "3");
make_test ("(caar '((4) 1 2 3))", "4");
make_test ("(let ((l '((0 1) 2))) (setf (caar l) 3))", "3");
make_test ("(let ((l '((0 1) 2))) (setf (caar l) 3) l)", "((3 1) 2)");
make_test ("(let ((l '(((0 1 2))))) (setf (cadaar l) 3) l)", "(((0 3 2)))");
make_test ("(let ((s \"abc\")) (setf (char s 1) #\\d) s)", "\"adc\"");
make_test ("(let ((s \"abc\")) (setf (schar s 1) #\\d) s)", "\"adc\"");
make_test ("(let ((s #*0100)) (setf (bit s 1) 0) s)", "#*0000");
make_test ("(let ((s #*0100)) (setf (sbit s 1) 0) s)", "#*0000");
make_test ("(let ((v #(0 1 2))) (setf (svref v 1) 3) v)", "#(0 3 2)");
make_test ("(incf c)", "6");
make_test ("(incf c 3)", "9");
make_test ("(let ((i 0)) (incf i))", "1");
make_test ("(let ((arr (make-array 2)) (x 0)) (setf (aref arr (incf x)) (incf x)) arr)", "#(NIL 2)");
make_test ("(let ((i 0)) (incf (car (cons (incf i) 0)) 1))", "2");
make_test ("(decf c)", "8");
make_test ("(decf c c)", "0");
make_test ("(and)", "T");
make_test ("(and t)", "T");
make_test ("(and (+ 1 2))", "3");
make_test ("(and t t t)", "T");
make_test ("(and t nil t)", "NIL");
make_test ("(and (= 1 1) t)", "T");
make_test ("(and t (= 1 2) t)", "NIL");
make_test ("(and 1 2 (+ 1 2))", "3");
make_test ("(and 0 (values 1 2))", "1\n2");
make_test ("(or)", "NIL");
make_test ("(or (+ 1 2))", "3");
make_test ("(or nil nil nil)", "NIL");
make_test ("(or (= 1 2) t)", "T");
make_test ("(or t nil t)", "T");
make_test ("(or nil nil (+ 1 2))", "3");
make_test ("(or nil nil (+ 1 2) nil)", "3");
make_test ("(or (values 1 2))", "1\n2");
make_test ("(or (values 1 2) nil)", "1");
make_test ("(concatenate 'string)", "\"\"");
make_test ("(concatenate 'string \"\")", "\"\"");
make_test ("(concatenate 'string \"aa\" \"bb\")", "\"aabb\"");
make_test ("(concatenate 'vector #(0 1 2) #(3) #(4 5))", "#(0 1 2 3 4 5)");
make_test ("(concatenate 'list '(0 1 2) nil '(3 4 5))", "(0 1 2 3 4 5)");
make_test ("(concatenate 'list nil)", "NIL");
make_test ("(do (x (y 0) (z 0 (1+ z))) ((>= z 10) (values x y z)) (write z))", "0123456789\nNIL\n0\n10");
make_test ("(do ((x 0 (1+ x)) (y 0 (1+ x))) ((>= y 10)) (write y))", "0123456789\nNIL");
make_test ("(do ((dd 0 (1+ dd))) (nil) (if (> dd 5) (return 10) (write dd)))", "012345\n10");
make_test ("(do ((i 0 (1+ i))) ((= i 3)) 1 (write i) (go 2) (write 3) 2 (write 4))", "041424\nNIL");
make_test ("(do nil ((return t)))", "T");
make_test ("(do* (x (y 0) (z 0 (1+ z))) ((>= z 10) (values x y z)) (write z))", "0123456789\nNIL\n0\n10");
make_test ("(do* ((x 0 (1+ x)) (y x (1+ x))) ((>= y 10)) (write y))", "023456789\nNIL");
make_test ("(do* ((dd 0 (1+ dd))) (nil) (if (> dd 5) (return 10) (write dd)))", "012345\n10");
make_test ("(do* ((i 0 (1+ i))) ((= i 3)) 1 (write i) (go 2) (write 3) 2 (write 4))", "041424\nNIL");
make_test ("(dotimes (i 0 i))", "0");
make_test ("(dotimes (tmp 10 tmp))", "10");
make_test ("(dotimes (j 1) (+ 1) 1)", "NIL");
make_test ("(dotimes (i 10) (write i) (if (= i 5) (return (values 11 12))))", "012345\n11\n12");
make_test ("(dotimes (i 2) 1 (write i) (go 2) (write 3) 2 (write 4))", "0414\nNIL");
make_test ("(defun dotimestest nil (write i))", "DOTIMESTEST");
make_test ("(dotimes (i 5) (declare (special i)) (dotimestest))", "01234\nNIL");
make_test ("(dolist (i nil) (write i))", "NIL");
make_test ("(dolist (i nil (values 0 1)))", "0\n1");
make_test ("(dolist (i '(0 1 2 3)) (write i))", "0123\nNIL");
make_test ("(dolist (i '(0 1 2 3)) (if (= i 2) (return (values 12 13))) (write i))", "01\n12\n13");
make_test ("(dolist (c '(0 1 2)) 1 (write c) (go 2) (write 3) 2 (write 4))", "041424\nNIL");
make_test ("(mapcar #'car '((1 a) (2 b) (3 c)))", "(1 2 3)");
make_test ("(mapcar #'abs '(3 -4 2 -5 -6))", "(3 4 2 5 6)");
make_test ("(mapcar 'abs '(3 -4 2 -5 -6))", "(3 4 2 5 6)");
make_test ("(mapcar #'cons '(a b c) '(1 2 3))", "((A . 1) (B . 2) (C . 3))");
make_test ("(let ((s 0)) (write (mapcar (lambda (s) s) '(1 2 3))) s)", "(1 2 3)\n0");
make_test ("(defun funcdes (x) 0)", "FUNCDES");
make_test ("(flet ((funcdes (x) 1)) (mapcar 'funcdes '(nil nil nil)))", "(0 0 0)");
make_test ("(mapc #'+ '(0 1 2) '(10 11) '(20 21 22 23))", "(0 1 2)");
make_test ("(mapcan #'list '(0 1 2 3 4) '(6 7 8 9))", "(0 6 1 7 2 8 3 9)");
make_test ("(maplist #'write '(0 1 2))", "(0 1 2)(1 2)(2)\n((0 1 2) (1 2) (2))");
make_test ("(maplist #'append '(0 1 2) '(5 6) '(7 8 9 10))", "((0 1 2 5 6 7 8 9 10) (1 2 6 8 9 10))");
make_test ("(maplist #'append '(0 1 2) nil '(7 8 9 10))", "NIL");
make_test ("(mapl #'append '(0 1 2) '(5 6) '(7 8 9 10))", "(0 1 2)");
make_test ("(mapcon #'list '(0 1 2 3 4))", "((0 1 2 3 4) (1 2 3 4) (2 3 4) (3 4) (4))");
make_test ("(map 'string #'char-upcase \"abcDefGh\")", "\"ABCDEFGH\"");
make_test ("(map 'vector #'+ #(1 2 3 4) '(10 11 12) #(20 21 22))", "#(31 34 37)");
make_test ("(map 'list #'+ #(1 2 3 4) '(10 11 12) #(20 21 22))", "(31 34 37)");
make_test ("(map nil #'+ '(1 2 3 4) '(10 11 12) '(20 21 22))", "NIL");
make_test ("(map 'vector #'+ #(1 2 3 4) nil #(20 21 22))", "#()");
make_test ("(map 'bit-vector '+ #*1010 '(0 1 0 1))", "#*1111");
make_test ("(map 'vector #'1+ (make-array 4 :initial-contents '(0 1 2 3) :fill-pointer 2))", "#(1 2)");
make_test ("(map-into \"xxx\" #'char-upcase \"aBcD\")", "\"ABC\"");
make_test ("(map-into (make-array 3) #'max '(0 1) #(1 0 2))", "#(1 1 NIL)");
make_test ("(reduce #'* '(2 3 5 7))", "210");
make_test ("(reduce #'* #(2 3 5 7) :start 1 :end 1)", "1");
make_test ("(reduce (lambda (f s) (write f)) '(2 3 5 7) :from-end t)", "532\n2");
make_test ("(reduce #'* '(2 3 5 7) :from-end t :initial-value 0)", "0");
make_test ("(reduce #'+ '(\"\"))", "\"\"");
make_test ("(reduce #'+ nil)", "0");
make_test ("(reduce #'+ nil :initial-value 1)", "1");
make_test ("(reduce 'list (list 'x) :initial-value 'z :from-end t)", "(X Z)");
make_test ("(merge 'list '(0 1 3 7 8 5) #(0 4 2 10) #'<)", "(0 0 1 3 4 2 7 8 5 10)");
make_test ("(merge 'vector '((0 . 1) (3 . 1)) #((0 . 2) (1 . 2)) #'< :key #'car)", "#((0 . 1) (0 . 2) (1 . 2) (3 . 1))");
make_test ("(remove 3 '(1 2 3 4 3 2 1))", "(1 2 4 2 1)");
make_test ("(remove #\\e \"abcdefg\")", "\"abcdfg\"");
make_test ("(remove-if #'numberp #(1 2 3))", "#()");
make_test ("(remove-if 'numberp #(1 2 3))", "#()");
make_test ("(remove-if #'consp '(x y))", "(X Y)");
make_test ("(remove-if #'not '(nil))", "NIL");
make_test ("(remove-if #'oddp (make-array 4 :initial-contents '(0 1 2 3) :fill-pointer 2))", "#(0)");
make_test ("(remove-if-not #'listp #(()))", "#(NIL)");
make_test ("(remove-duplicates \"abcaadeabfc\")", "\"deabfc\"");
make_test ("(remove-duplicates '(0 1 2 3 4 3 3 5 6 8 6))", "(0 1 2 4 3 5 8 6)");
make_test ("(remove-duplicates '(0 1 1 2 3) :key nil)", "(0 1 2 3)");
make_test ("(remove-duplicates nil)", "NIL");
make_test ("(remove-duplicates #(0 1 2 4 5 3 4 6 9 3) :start 1 :end 7 :test #'=)", "#(0 1 2 5 3 4 6 9 3)");
make_test ("(delete-duplicates #(0 1 2 4 5 3 4 6 9 3) :from-end t :test #'=)", "#(0 1 2 4 5 3 6 9)");
make_test ("(substitute #\\a #\\b \"abcabc\" :count 1 :from-end t)", "\"abcaac\"");
make_test ("(substitute-if 0 #'evenp #(0 1 2 3 4 5 6))", "#(0 1 0 3 0 5 0)");
make_test ("(nsubstitute-if 0 #'evenp '(0 1 2 3 4 5 6) :start 2 :end 5 :count 1)", "(0 1 0 3 4 5 6)");
make_test ("(substitute-if 0 #'evenp '(0 1 2 3 4 5 6) :start 2 :end 5 :count 1 :from-end t)", "(0 1 2 3 0 5 6)");
make_test ("(substitute-if-not 0 #'oddp '(0 1 2 3 4 5 6) :start 2 :count 1 :from-end t)", "(0 1 2 3 4 5 0)");
make_test ("(subst '(1) '(2) '(1 (2) (3 1 (4 (2)) 1)) :test 'equal)", "(1 (1) (3 1 (4 (1)) 1))");
make_test ("(nsubst 0 1 '(1 2 (3 1 (4 1) 1)))", "(0 2 (3 0 (4 0) 0))");
make_test ("(nsubst-if 0 #'consp '(1 2 (3 4)) :key #'identity)", "0");
make_test ("(reverse '(0 1 2 3))", "(3 2 1 0)");
make_test ("(reverse \"hello\")", "\"olleh\"");
make_test ("(reverse #(0 1 2 3))", "#(3 2 1 0)");
make_test ("(reverse (make-array 4 :initial-contents '(0 1 2 3) :fill-pointer 2))", "#(1 0)");
make_test ("(nreverse '(0 1 2 3))", "(3 2 1 0)");
make_test ("(nreverse \"hello\")", "\"olleh\"");
make_test ("(nreverse #(0 1 2 3))", "#(3 2 1 0)");
make_test ("(revappend '(2 1 0) '(3 4))", "(0 1 2 3 4)");
make_test ("(nreconc '(2 1 0) '(3 4))", "(0 1 2 3 4)");
make_test ("(adjoin 0 '(0 1 2))", "(0 1 2)");
make_test ("(adjoin 0 '(1 2))", "(0 1 2)");
make_test ("(fill \"abcdef\" #\\g)", "\"gggggg\"");
make_test ("(fill '(0 1 2 3 4 5) 0 :start 1 :end 4)", "(0 0 0 0 4 5)");
make_test ("(replace \"abcdefg\" \"xyz\" :start1 2 :start2 1)", "\"abyzefg\"");
make_test ("(let ((s \"abcdefg\")) (replace s s :start1 1 :start2 2))", "\"acdefgg\"");
make_test ("(defparameter ll NIL)", "LL");
make_test ("(push (1+ 0) ll)", "(1)");
make_test ("ll", "(1)");
make_test ("(let ((l '(1 2 3 4))) (pushnew 3 l :test #'=))", "(1 2 3 4)");
make_test ("(let ((l '(1 2 3 4))) (pushnew 0 l :test #'= :key 'identity))", "(0 1 2 3 4)");
make_test ("(let ((l '((1 2 3)))) (pushnew 0 (car l)))", "(0 1 2 3)");
make_test ("(pop ll)", "1");
make_test ("ll", "NIL");
make_test ("(set-difference '(0 1 1 2) nil)", "(0 1 1 2)");
make_test ("(nset-difference '(0 1 1 2) '(3 1 5) :test #'=)", "(0 2)");
make_test ("(union '(1 2 3) '(3 4 5) :test #'=)", "(1 2 3 4 5)");
make_test ("(nunion '((1 . 2) (2 . 3)) '((2 . 5) (3 . 6)) :key #'car)", "((1 . 2) (2 . 5) (3 . 6))");
make_test ("(intersection '(1 2 3) '(3 4 5) :test #'=)", "(3)");
make_test ("(nintersection '((1 . 2) (2 . 3)) '((2 . 5) (3 . 6)) :key #'car)", "((2 . 3))");
make_test ("(set-exclusive-or '(1 2 3 4 5) '(8 7 6 5 4 1) :key #'identity :test #'=)", "(2 3 8 7 6)");
make_test ("(nset-exclusive-or '(1 2 3 4 5) '(8 7 6 5 4 1) :key #'identity :test #'=)", "(2 3 8 7 6)");
make_test ("(subsetp '(1 4 nil 5) '(3 2 1 nil 4 5) :key #'identity :test #'equal)", "T");
make_test ("(subsetp '(1 4 nil 5) '(3 2 1 4 5) :key #'identity :test #'equal)", "NIL");
make_test ("(mismatch '(0 1 2 3 4 5 6) #(1 2 3 4) :start1 2 :end1 4 :start2 1 :end2 3 :test #'= :key #'identity)", "NIL");
make_test ("(mismatch '(0 1 2 10 4 5 6) #(1 2 3 4) :start1 2 :end1 4 :start2 1 :end2 3 :test #'= :key #'identity)", "3");
make_test ("(mismatch '(0 1 2 3 4) #(1 2) :start1 2 :end1 4 :start2 1 :test #'=)", "3");
make_test ("(mismatch \"fcde\" \"abcde\" :from-end t)", "1");
make_test ("(mismatch \"abcde\" \"abcde\" :start1 1 :from-end t)", "1");
make_test ("(mismatch \"abcde\" \"bcde\" :start1 1 :from-end t)", "NIL");
make_test ("(mismatch \"abcde\" \"bcde\" :from-end t)", "1");
make_test ("(mismatch \"xywabcde\" \"xyzabcde\" :start1 2 :from-end t)", "3");
make_test ("(search \"hello\" \"hey hello\" :test 'char-equal)", "4");
make_test ("(search \"hello\" \"hey hello hey hello\" :test 'char-equal)", "4");
make_test ("(search \"hello\" \"hey hello hey hello\" :test 'char-equal :from-end t)", "14");
make_test ("(search \"hello\" \"hey hello hey hello\" :test 'char-equal :from-end t :start2 10)", "14");
make_test ("(search \"hello\" \"hey hello hey hello\" :start2 8 :end2 10)", "NIL");
make_test ("(sort '(0 4 2 7 1 8 9 6 3 6) #'<)", "(0 1 2 3 4 6 6 7 8 9)");
make_test ("(stable-sort #((0 . 1) (4 . 2) (7 . 0) (1 . 0) (6 . 2) (3 . 9) (6 . 1)) #'< :key #'car)", "#((0 . 1) (1 . 0) (3 . 9) (4 . 2) (6 . 2) (6 . 1) (7 . 0))");
make_test ("(cond (t (+ 1 2)))", "3");
make_test ("(cond (nil 2) (t 3) (nil 4))", "3");
make_test ("(cond ((= 2 (+ 1 2)) (+ 0 10)) ((= 3 (+ 1 2)) (+ 12) (+ 0 11)))", "11");
make_test ("(cond ((+)) ((+)))", "0");
make_test ("(cond (nil 0) (t 0 (values 1 2)))", "1\n2");
make_test ("(cond ((values 0 1)))", "0");
make_test ("(case (+ 1 2) (3 (+ 1 5)))", "6");
make_test ("(case (+ 1 2) (4 (+ 1 5)))", "NIL");
make_test ("(case (+ 1 2) (2 10) ((4 3) (+ 1 5)))", "6");
make_test ("(case (+ 1 2) (2 10) ((4 3) (+ 1 5)) (otherwise 12))", "6");
make_test ("(case (+ 1 2) (2 10) ((4 5) (+ 1 5)) (otherwise 11 12))", "12");
make_test ("(ccase (+) (1 (+ 2)) (0 (+ 5)))", "5");
#make_test ("(ccase (+) (1 (+ 2)) (2 (+ 5)))", "NIL");
make_test ("(ecase (+) (1 (+ 2)) (0 (+ 5)))", "5");
#make_test ("(ecase (+) (1 (+ 2)) (2 (+ 5)))", "NIL");
make_test ("(typecase (+) (integer (+ 10) (+ 11)) (number 12 13))", "11");
make_test ("(typecase (+) (number 12 13) (integer 15))", "13");
make_test ("(typecase (+) (string 15) (otherwise (+ 20)))", "20");
make_test ("(let (y) (labels ((f (xx) (typecase xx (cons (f (car xx)))))) (f '(\"\"))))", "NIL");
make_test ("(let (y) (labels ((f (x) (typecase x (string x) (cons (f (car x)))))) (f '(\"\"))))", "\"\"");
make_test ("(ctypecase (+) (number 12 13) (integer 15))", "13");
#make_test ("(ctypecase (+) (string 15))", "NIL");
make_test ("(etypecase (+) (number 12 13) (integer 15))", "13");
#make_test ("(etypecase (+) (string 15))", "NIL");
make_test ("(multiple-value-bind (x y) (floor 2 1) (list x y))", "(2 0)");
make_test ("(multiple-value-bind (x) (floor 2 1) (list x))", "(2)");
make_test ("(multiple-value-bind (x y z) (floor 2 1) (list x y z))", "(2 0 NIL)");
make_test ("(multiple-value-bind (x y) (values 'i 'j) (list x y))", "(I J)");
make_test ("(let (x y z) (multiple-value-setq (x y z) (values 0 1 2)))", "0");
make_test ("(let (x y z) (multiple-value-setq (x y z) (values 0 1)) (list x y z))", "(0 1 NIL)");
make_test ("(let (x y z) (multiple-value-setq (x y z) (values 0 1 2 3)) (list x y z))", "(0 1 2)");
make_test ("(let (x y z) (setf (values x y z) (values 0 1)))", "0\n1\nNIL");
make_test ("(let (x y z) (setf (values x y z) (values 0 1)) (list x y z))", "(0 1 NIL)");
make_test ("(let ((x '(0 1)) y z) (setf (values (car x) y z) (values 2 3)) (list x y z))", "((2 1) 3 NIL)");
make_test ("(let ((x 0) (y '(1 2)) (z 3)) (shiftf x (car y) z 4) (list x y z))", "(1 (3 2) 4)");
make_test ("(let ((x 0) (y '(1 2)) (z 3)) (shiftf x (car y) z 4))", "0");
make_test ("(let ((x 0) (y '(1 2)) (z 3)) (rotatef x (car y) z) (list x y z))", "(1 (3 2) 0)");
make_test ("(prog ((x 0) y) a (write 0) (go c) b (write 1) c (return x))", "0\n0");
make_test ("(prog* ((x 0) (y x)) a (write 0) (go c) b (write 1) c (return y))", "0\n0");
make_test ("(every #'evenp '(0 2 4))", "T");
make_test ("(every #'< '(0 1 2) #(1 1))", "NIL");
make_test ("(some #'< '(0 1 2) '(1 1))", "T");
make_test ("(notany #'oddp '(1 3 4))", "NIL");
make_test ("(notevery #'oddp '(1 3 4))", "T");
make_test ("(member 3 '(1 2 3 4))", "(3 4)");
make_test ("(member 5 '(1 2 3))",  "NIL");
make_test ("(member 2 '((0 . 5) (1 . 6) (2 . 7) (3 . 8)) :key #'car :test #'=)", "((2 . 7) (3 . 8))");
make_test ("(member-if #'evenp '(1 2 3 4))", "(2 3 4)");
make_test ("(member-if #'listp '(nil))", "(NIL)");
make_test ("(member-if #'evenp '((1 . 5) (3 . 6) (2 . 7) (4 . 8)) :key #'car)", "((2 . 7) (4 . 8))");
make_test ("(member-if-not #'evenp '(0 2 4))", "NIL");
make_test ("(find #\\3 \"012\")", "NIL");
make_test ("(find #\\3 \"0123\")", "#\\3");
make_test ("(find 2 '(0 1 2 3 4) :end 2 :key #'=)", "NIL");
make_test ("(find-if 'digit-char-p \"0123\")", "#\\0");
make_test ("(find-if 'digit-char-p \"0123\" :from-end t)", "#\\3");
make_test ("(find-if 'digit-char-p \"0123a\" :from-end t :start 4)", "NIL");
make_test ("(find-if-not 'digit-char-p \"0123\")", "NIL");
make_test ("(find-if-not #'oddp #(1 2 5 4 7) :from-end t)", "4");
make_test ("(assoc 2 '((0 . 10) (1 . 11) (2 . 12)))", "(2 . 12)");
make_test ("(assoc 3 '((0 . 10) (1 . 11) (2 . 12)))", "NIL");
make_test ("(assoc 1 '(((0) . 10) ((1) . 11) ((2) . 12)) :test #'= :key #'car)", "((1) . 11)");
make_test ("(assoc 2 '(((0) . 10) nil ((2) . 12)) :test #'= :key #'car)", "((2) . 12)");
make_test ("(assoc-if 'zerop '((0 . 10) (1 . 11) (2 . 12)))", "(0 . 10)");
make_test ("(assoc-if-not 'zerop '((0 . 10) (1 . 11) (2 . 12)))", "(1 . 11)");
make_test ("(rassoc 1 '((10 0) nil (11 1) (12 2)) :test #'= :key #'car)", "(11 1)");
make_test ("(rassoc-if-not #'oddp '(nil (1 . 11) (2 . 12)))", "(2 . 12)");
make_test ("(position 2 '(3 2 1 0))", "1");
make_test ("(position 4 '(3 2 1 0))", "NIL");
make_test ("(position 2 '(3 2 2 0) :from-end t :start 2 :end 3)", "2");
make_test ("(position-if 'evenp '(3 2 1 0))", "1");
make_test ("(position-if 'evenp '(3 1))", "NIL");
make_test ("(position-if-not #'evenp '(3 1))", "0");
make_test ("(position-if-not 'evenp '(3 2 1 0) :from-end t)", "2");
make_test ("(count #\\a \"abcabc\")", "2");
make_test ("(count #\\a \"abcabc\" :start 3 :end 4 :from-end t)", "1");
make_test ("(count 2 '(1 2 3))", "1");
make_test ("(count-if #'upper-case-p \"aAbBcCdD\")", "4");
make_test ("(count-if-not #'evenp #(0 1 2))", "1");
make_test ("(>= (get-internal-run-time) 0)", "T");
make_test ("(machine-type)", "NIL");
make_test ("(machine-instance)", "NIL");
make_test ("(machine-version)", "NIL");
make_test ("(short-site-name)", "NIL");
make_test ("(long-site-name)", "NIL");
make_test ("(with-standard-io-syntax (write 10))", "10\n10");

make_test ("#+common-lisp \"\" \"\"", "\"\"\n\"\"");
make_test ("#-common-lisp \"\" \"\"", "\"\"");
make_test ("#+(and common-lisp foo) \"\" \"\"", "\"\"");
make_test ("#+(or common-lisp foo) \"\" \"\"", "\"\"\n\"\"");
make_test ("#-(not common-lisp) \"\" \"\"", "\"\"\n\"\"");
make_test ("#+common-lisp 'x", "X");
make_test ("#+common-lisp 0", "0");
make_test ("#-common-lisp 0 0", "0");
make_test ("#-common-lisp 'aaa 'bbb", "BBB");
make_test ("#+common-lisp '(1 2 3) \"\"", "(1 2 3)\n\"\"");
make_test ("#+foo '(1 2 3) \"\"", "\"\"");
make_test ("#+foo '((1 2) 3) 0", "0");
make_test ("#+(and\ncommon-lisp) 0 0", "0\n0");
make_test ("#-(\nand common-lisp) 0 0", "0");
make_test ("#+\n(and common-lisp) 0 0", "0\n0");
make_test ("#+common-lisp (\n)", "NIL");
make_test ("#+common-lisp '(1\n2)", "(1 2)");
make_test ("#-common-lisp (\n) 0", "0");
make_test ("#+foo \"aaa\nbbb\" 0", "0");
make_test ("#+foo (1\n2 (3)) 0", "0");
make_test ("#+foo (\"\") 0", "0");
make_test ("#+foo (foo \"a\") 0", "0");
make_test ("#-fooo :fooo 0", ":FOOO\n0");
make_test ("(#+foo\n0)", "NIL");
make_test ("'(1 2 #+foo\n3)", "(1 2)");
make_test ("`(#+foo ,@'(\"\"))", "NIL");
make_test ("(list `(#+foo ,@'(\"\")))", "(NIL)");
make_test ("`(+ #-foo ,0)", "(+ 0)");

make_test ("(encode-universal-time 10 10 10 1 2 1900 0)", "2715010");
make_test ("(encode-universal-time 10 11 12 19 8 2004 0)", "3301906270");
#make_test ("(encode-universal-time 10 11 12 19 8 2004)", "3301899070");
make_test ("(encode-universal-time 10 11 12 19 8 2004 3)", "3301917070");
make_test ("(decode-universal-time 2201917080 0)", "0\n38\n3\n11\n10\n1969\n5\nNIL\n0");
make_test ("(decode-universal-time 90000000000 0)", "0\n0\n16\n27\n12\n4751\n3\nNIL\n0");
make_test ("(decode-universal-time 90000000000 7)", "0\n0\n9\n27\n12\n4751\n3\nNIL\n7");

make_test ("*package*", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(find-package *package*)", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(find-package \"CL\")", "#<PACKAGE \"COMMON-LISP\">");
make_test ("(find-package 'cl-user)", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(find-package #\\a)", "NIL");
make_test ("(package-name \"CL\")", "\"COMMON-LISP\"");
make_test ("(package-name *package*)", "\"COMMON-LISP-USER\"");
make_test ("(package-nicknames *package*)", "(\"CL-USER\")");
make_test ("(package-nicknames (symbol-package :fff))", "NIL");
make_test ("(package-use-list *package*)", "(#<PACKAGE \"COMMON-LISP\">)");
make_test ("(package-use-list (symbol-package :fff))", "NIL");
make_test ("(package-used-by-list *package*)", "NIL");
make_test ("(package-used-by-list (symbol-package 'if))", "(#<PACKAGE \"COMMON-LISP-USER\">)");
make_test ("(list-all-packages)", "(#<PACKAGE \"COMMON-LISP-USER\"> #<PACKAGE \"COMMON-LISP\"> #<PACKAGE \"KEYWORD\">)");
make_test ("(make-package \"test\")", "#<PACKAGE \"test\">");
make_test ("(list-all-packages)", "(#<PACKAGE \"test\"> #<PACKAGE \"COMMON-LISP-USER\"> #<PACKAGE \"COMMON-LISP\"> #<PACKAGE \"KEYWORD\">)");
make_test ("(rename-package \"test\" \"test\")", "#<PACKAGE \"test\">");
make_test ("(rename-package \"test\" \"TEST\" '(nick1 #\\n \"nick3\"))", "#<PACKAGE \"TEST\">");
make_test ("(list-all-packages)", "(#<PACKAGE \"TEST\"> #<PACKAGE \"COMMON-LISP-USER\"> #<PACKAGE \"COMMON-LISP\"> #<PACKAGE \"KEYWORD\">)");
make_test ("(package-nicknames \"TEST\")", "(\"NICK1\" \"n\" \"nick3\")");
make_test ("(princ 'test::foo)", "FOO\nTEST::FOO");
make_test ("(in-package \"CL\")", "#<PACKAGE \"COMMON-LISP\">");
make_test ("*package*", "#<PACKAGE \"COMMON-LISP\">");
make_test ("(in-package cl-user)", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("*package*", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(use-package 'cl)", "T");
make_test ("(make-package \"newtest\")", "#<PACKAGE \"newtest\">");
make_test ("(rename-package (find-package '|newtest|) (find-package '|newtest|))", "#<PACKAGE \"newtest\">");
make_test ("(in-package \"newtest\")", "#<PACKAGE \"newtest\">");
make_test ("'a", "A");
make_test ("'b", "B");
make_test ("#+foo 0 1", "1");
make_test ("(cl:do-symbols (s '|newtest|) (cl:write s))", "ABS|newtest|\nCOMMON-LISP:NIL");
make_test ("(cl:do-symbols (s '|newtest|) 0 (cl:write s) (cl:go 2) 1 (cl:write 1) 2 (cl:write 2))", "A2B2S2|newtest|2\nCOMMON-LISP:NIL");
make_test ("(cl:package-use-list cl:*package*)", "COMMON-LISP:NIL");
make_test ("(cl:use-package 'cl)", "T");
make_test ("(cl:use-package nil \"newtest\")", "T");
make_test ("(cl:package-use-list cl:*package*)", "(#<PACKAGE \"COMMON-LISP\">)");
make_test ("(cadr '(0 1))", "1");
make_test ("(cl:import 'cl:car)", "T");
make_test ("(cl:import '(cl:car) '|newtest|)", "T");
make_test ("(cl:find-all-symbols \"CAR\")", "(CAR)");
make_test ("(cl:find-all-symbols 'cdr)", "(CDR)");
make_test ("(cl:find-all-symbols 'al-exit)", "(COMMON-LISP-USER:AL-EXIT AL-EXIT)");
make_test ("(cl:export nil \"newtest\")", "T");
make_test ("(cl:export 'cdr)", "T");
make_test ("'|newtest|:cdr", "CDR");
make_test ("(cl:export 'jjj)", "T");
make_test ("(cl:export '(jjj) cl:*package*)", "T");
make_test ("(do-external-symbols (s '|newtest|) (write s))", "CDRJJJ\nNIL");
make_test ("(do-external-symbols (s '|newtest|) 0 (write s) (go 2) 1 (write 1) 2 (write 2))", "CDR2JJJ2\nNIL");
make_test ("'|newtest|:jjj", "JJJ");
make_test ("(cl:unexport nil \"newtest\")", "T");
make_test ("(cl:unexport '(jjj) cl:*package*)", "T");
make_test ("(car ())", "NIL");
make_test ("(cl:unuse-package 'cl)", "COMMON-LISP:T");
make_test ("(cl:unuse-package cl:nil \"newtest\")", "COMMON-LISP:T");
make_test ("(cl:package-use-list cl:*package*)", "COMMON-LISP:NIL");
make_test ("(cl:package-used-by-list 'cl)", "(#<PACKAGE \"COMMON-LISP-USER\">)");
make_test ("(car ())", "COMMON-LISP:NIL");
make_test ("(cl:in-package cl-user)", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(make-package 'pack10)", "#<PACKAGE \"PACK10\">");
make_test ("(export 'pack10::foo 'pack10)", "T");
make_test ("(make-package 'pack11)", "#<PACKAGE \"PACK11\">");
make_test ("(use-package 'pack10 'pack11)", "T");
make_test ("(export 'pack11::foo 'pack11)", "T");
make_test ("(make-package 'pack12)", "#<PACKAGE \"PACK12\">");
make_test ("(use-package 'pack10 'pack12)", "T");
make_test ("(use-package 'pack11 'pack12)", "T");
make_test ("(defpackage \"aaa\" (:nicknames \"bbb\" |ccc|) (:use cl-user) (:export \"sym\") (:intern \"eee\" \"fff\") (:import-from \"CL\" car caar) (:import-from \"CL-USER\" al-argc))", "#<PACKAGE \"aaa\">");
make_test ("(package-nicknames \"aaa\")", "(\"bbb\" \"ccc\")");
make_test ("(package-use-list \"aaa\")", "(#<PACKAGE \"COMMON-LISP-USER\">)");
make_test ("'|aaa|:|sym|", "|aaa|:|sym|");
make_test ("(make-package 'whatever :nicknames '(\"APPLE\" pear))", "#<PACKAGE \"WHATEVER\">");
make_test ("(package-nicknames 'whatever)", "(\"APPLE\" \"PEAR\")");
make_test ("(defpackage \"aaa\" (:nicknames \"bbb\" |ccc|) (:use cl-user) (:export \"sym\") (:intern \"eee\" \"fff\") (:import-from \"CL\" car caar) (:import-from \"CL-USER\" al-argc))", "#<PACKAGE \"aaa\">");
make_test ("(make-package \"PACK1\")", "#<PACKAGE \"PACK1\">");
make_test ("(export 'pack1::foo 'pack1)", "T");
make_test ("(make-package \"PACK2\")", "#<PACKAGE \"PACK2\">");
make_test ("(shadow \"FOO\" \"PACK2\")", "T");
make_test ("'pack2::foo", "PACK2::FOO");
make_test ("(use-package 'pack1 'pack2)", "T");
make_test ("(find-symbol \"FOO\" 'pack2)", "PACK2::FOO\n:INTERNAL");
make_test ("(shadow \"BAR\" \"PACK1\")", "T");
make_test ("(package-shadowing-symbols 'pack1)", "(PACK1::BAR)");
make_test ("(package-shadowing-symbols 'pack2)", "(PACK2::FOO)");
make_test ("(import 'pack1::bar 'pack2)", "T");
make_test ("(package-shadowing-symbols 'pack2)", "(PACK2::FOO)");
make_test ("(in-package \"PACK2\")", "#<PACKAGE \"PACK2\">");
make_test ("'pack1:foo", "PACK1:FOO");
make_test ("(cl:in-package \"CL-USER\")", "#<PACKAGE \"COMMON-LISP-USER\">");
make_test ("(with-package-iterator (next 'cl :inherited) (do nil (nil) (multiple-value-bind (pred s tp p) (next) (if pred (write s) (return)))))", "NIL");


# arithmetic tests

make_test ("2/4", "1/2");
make_test ("0.1s2", "10.0");
make_test ("0.0", "0.0");
make_test ("0.1", "0.0999999999999999916733");
make_test ("(+)", "0");
make_test ("(+ 1)", "1");
make_test ("(+ 1 2 3)", "6");
make_test ("(+ 1/2 2 3 1000)", "2011/2");
make_test ("(+ a b)", "17");
make_test ("(+ .1 .1 .1)", "0.299999999999999988898");
make_test ("(+ 1 1 1.2)", "3.20000000000000017764");
make_test ("(+ #c(1 1) #c(2.0 3))", "#C(3.0 4.0)");
make_test ("(+ #c(1 1) #c(2 -1))", "3");
make_test ("(+ 1 #c(2 3.0))", "#C(3.0 3.0)");
make_test ("(- 3.5)", "-3.5");
make_test ("(- 3 4.5)", "-1.5");
make_test ("(- 1 1 1.2)", "-1.19999999999999995559");
make_test ("(- #c(0 1))", "#C(0 -1)");
make_test ("(- #c(1 2) #c(3.0 4))", "#C(-2.0 -2.0)");
make_test ("(- 1 #c(3.0 4))", "#C(-2.0 -4.0)");
make_test ("(*)", "1");
make_test ("(* 1)", "1");
make_test ("(* 3 5)", "15");
make_test ("(* 1 1 1.2)", "1.19999999999999995559");
make_test ("(* 3 #c(3.0 4))", "#C(9.0 12.0)");
make_test ("(* #c(0 1) #c(0 1))", "-1");
make_test ("(/ 12 4)", "3");
make_test ("(/ 13 4)", "13/4");
make_test ("(/ -8)", "-1/8");
make_test ("(/ 3 4 5)", "3/20");
make_test ("(/ 0.5)", "2.0");
make_test ("(/ 60 -2 3 5.0)", "-2.0");
make_test ("(/ #c(1 1) #c(1 1))", "1");
make_test ("(/ #c(1 2) #c(1 1))", "#C(3/2 1/2)");
make_test ("(/ #c(1 2) 3)", "#C(1/3 2/3)");
make_test ("(/ #c(1.1 2) 3)", "#C(0.36666666666666658525 0.666666666666666629659)");
make_test ("(floor 1)", "1\n0");
make_test ("(floor 1/2)", "0\n1/2");
make_test ("(ffloor 1)", "1.0\n0");
make_test ("(ffloor 1/2)", "0.0\n1/2");
make_test ("(ceiling 2)", "2\n0");
make_test ("(ceiling 1/2 1)", "1\n-1/2");
make_test ("(fceiling 2)", "2.0\n0");
make_test ("(fceiling 1/2 1)", "1.0\n-1/2");
make_test ("(truncate -3 2)", "-1\n-1");
make_test ("(truncate -1)", "-1\n0");
make_test ("(truncate -.1)", "0\n-0.0999999999999999916733");
make_test ("(ftruncate -3 2)", "-1.0\n-1");
make_test ("(ftruncate -1)", "-1.0\n0");
make_test ("(ftruncate -.1)", "0.0\n-0.0999999999999999916733");
make_test ("(round 5)", "5\n0");
make_test ("(round .5)", "0\n0.5");
make_test ("(round -1.5)", "-2\n0.5");
make_test ("(fround 5)", "5.0\n0");
make_test ("(fround .5)", "0.0\n0.5");
make_test ("(fround -1.5)", "-2.0\n0.5");
make_test ("(numerator 2)", "2");
make_test ("(numerator 2/3)", "2");
make_test ("(denominator 2)", "1");
make_test ("(denominator 2/3)", "3");
make_test ("(float 1)", "1.0");
make_test ("(float 1/2)", "0.5");
make_test ("(float 1.0)", "1.0");
make_test ("(gcd)", "0");
make_test ("(gcd -1)", "1");
make_test ("(gcd 0 -3)", "3");
make_test ("(gcd -4 6)", "2");
make_test ("(gcd 6 -27 81)", "3");
make_test ("(lcm)", "1");
make_test ("(lcm -1)", "1");
make_test ("(lcm 4 -6)", "12");
make_test ("(lcm -9 3 27)", "27");
make_test ("(sqrt 9)", "3.0");
make_test ("(sqrt 9.0)", "3.0");
make_test ("(isqrt 10)", "3");
make_test ("(complex 1)", "1");
make_test ("(complex 1.0)", "#C(1.0 0.0)");
make_test ("(complex 1 1)", "#C(1 1)");
make_test ("(complex 1 0)", "1");
make_test ("(complex 1 0.0)", "#C(1.0 0.0)");
make_test ("(complex 1.0 1)", "#C(1.0 1.0)");
make_test ("(complex 1 1.0)", "#C(1.0 1.0)");
make_test ("#c (0 1)", "#C(0 1)");
make_test ("#C(1 1.0)", "#C(1.0 1.0)");
make_test ("(realpart 1)", "1");
make_test ("(realpart (complex 1 2.0))", "1.0");
make_test ("(typep (realpart #c(1 1/2)) 'integer)", "T");
make_test ("(imagpart 1)", "0");
make_test ("(imagpart (complex 1 1/2))", "1/2");
make_test ("(conjugate 1)", "1");
make_test ("(conjugate (complex 1 1))", "#C(1 -1)");
make_test ("(cis 1.5)", "#C(0.0707372016677029064047 0.9974949866040544455)");
make_test ("(= 1)", "T");
make_test ("(= 1 1)", "T");
make_test ("(= 1 2)", "NIL");
make_test ("(= 1 1 1.0 1/1)", "T");
make_test ("(= 0 -0.0)", "T");
make_test ("(= #c(0 1) #c(0.0 1))", "T");
make_test ("(= #c(2 0) 1)", "NIL");
make_test ("(= #c(1.0 0) 1)", "T");
make_test ("(/= 0)", "T");
make_test ("(/= 0 1 2.0)", "T");
make_test ("(/= 0 1.0 2 1)", "NIL");
make_test ("(/= 1 2 #c(1 0) 3)", "NIL");
make_test ("(< 0)", "T");
make_test ("(< 0 1.0 2)", "T");
make_test ("(< 0 0.0 1)", "NIL");
make_test ("(<= 0 1 1)", "T");
make_test ("(<= 1 0)", "NIL");
make_test ("(> 1.1 0)", "T");
make_test ("(> 1 1)", "NIL");
make_test ("(>= 1 1.0)", "T");
make_test ("(min 0)", "0");
make_test ("(min 0 1 2)", "0");
make_test ("(min 0 1/2 -0.1)", "-0.0999999999999999916733");
make_test ("(max 0)", "0");
make_test ("(max 0 1.0 2)", "2");
make_test ("(max 0 1/2 -0.1)", "1/2");
make_test ("(sin 0)", "0.0");
make_test ("(sin 0.5)", "0.479425538604203005377");
make_test ("(cos 1/2)", "0.877582561890372758739");
make_test ("(tan 2.5)", "-0.747022297238660315521");
make_test ("(sinh 10)", "11013.2328747033934633");
make_test ("(cosh 1/2)", "1.12762596520638069819");
make_test ("(tanh 3.5)", "0.998177897611198683414");
make_test ("(exp 0)", "1.0");
make_test ("(exp 2.5)", "12.1824939607034732347");
make_test ("(expt 2 3)", "8");
make_test ("(expt 2 0)", "1");
make_test ("(expt 1/2 -2)", "4");
make_test ("(expt 2 1/2)", "1.41421356237309514547");
make_test ("(expt 1.2 -2.5)", "0.633938145260608987286");
make_test ("(log 1)", "0.0");
make_test ("(log 2.5)", "0.916290731874155106595");
make_test ("(log 1 0)", "0.0");
make_test ("(log 2.5 1.2)", "5.02568510266547630039");
make_test ("(lognot 1024)", "-1025");
make_test ("(lognot -1025)", "1024");
make_test ("(logior)", "0");
make_test ("(logior -1)", "-1");
make_test ("(logior 0 1 -4 27 259)", "-1");
make_test ("(logand)", "-1");
make_test ("(logand 15 7)", "7");
make_test ("(logand 1 2 4 129 -367)", "0");
make_test ("(logandc1 27 -49)", "-60");
make_test ("(logandc2 27 -49)", "16");
make_test ("(logeqv 1 2 3 4 13)", "9");
make_test ("(lognand 27 -49)", "-12");
make_test ("(lognor 27 -49)", "32");
make_test ("(logorc1 27 -49)", "-17");
make_test ("(logorc2 27 -49)", "59");
make_test ("(logxor 1 2 3 4 13)", "9");
make_test ("(boole boole-1 3 -7)", "3");
make_test ("(boole boole-2 3 -7)", "-7");
make_test ("(boole boole-andc1 3 -7)", "-8");
make_test ("(boole boole-andc2 3 -7)", "2");
make_test ("(boole boole-and 3 -7)", "1");
make_test ("(boole boole-c1 3 -7)", "-4");
make_test ("(boole boole-c2 3 -7)", "6");
make_test ("(boole boole-clr 3 -7)", "0");
make_test ("(boole boole-eqv 3 -7)", "5");
make_test ("(boole boole-ior 3 -7)", "-5");
make_test ("(boole boole-nand 3 -7)", "-2");
make_test ("(boole boole-nor 3 -7)", "4");
make_test ("(boole boole-orc1 3 -7)", "-3");
make_test ("(boole boole-orc2 3 -7)", "7");
make_test ("(boole boole-set 3 -7)", "-1");
make_test ("(boole boole-xor 3 -7)", "-6");
make_test ("(minusp 0)", "NIL");
make_test ("(minusp -0.5)", "T");
make_test ("(plusp 1)", "T");
make_test ("(plusp -1)", "NIL");
make_test ("(abs 0)", "0");
make_test ("(abs -0.5)", "0.5");
make_test ("(abs 1/2)", "1/2");
make_test ("(zerop -0)", "T");
make_test ("(zerop 1/2)", "NIL");
make_test ("(signum 0)", "0");
make_test ("(signum 3.5)", "1.0");
make_test ("(signum -1/2)", "-1");
make_test ("(mod 2.5 1)", "0.5");
make_test ("(mod -2.5 1)", "0.5");
make_test ("(rem 2.5 1)", "0.5");
make_test ("(rem -2.5 1)", "-0.5");
make_test ("(evenp 0)", "T");
make_test ("(evenp 1)", "NIL");
make_test ("(oddp 0)", "NIL");
make_test ("(oddp -1)", "T");
make_test ("*random-state*", "#<RANDOM-STATE ?>");
make_test ("(make-random-state)", "#<RANDOM-STATE ?>");
make_test ("(type-of (make-random-state))", "RANDOM-STATE");
make_test ("(make-random-state t)", "#<RANDOM-STATE ?>");
make_test ("(make-random-state nil)", "#<RANDOM-STATE ?>");
make_test ("(make-random-state *random-state*)", "#<RANDOM-STATE ?>");
make_test ("(< (random 10) 10)", "T");
make_test ("(< (random 4.5) 4.5)", "T");
make_test ("(< (random 10.5 (make-random-state)) 10.5)", "T");



print "\ntotal tests: " . $total_tests . "\n";
print "passed " . $passed_tests . " (" . $passed_tests / $total_tests * 100 . "%), failed " . $failed_tests . "\n";



kill 'TERM', $pid;
waitpid ($pid, 0);





sub make_test
{
    my $skip_only_first_line;
    my $i = 0;

    if (not defined ($_[2]))
    {
	$skip_only_first_line = 0;
    }
    else
    {
	$skip_only_first_line = $_[2];
    }

    print $_[0] . " -> ";

    my $in = $_[0] . "\n";

    my @inlines = split("\n", $in);

    foreach my $l (@inlines)
    {
	print $al_in $l . "\n";

	if ($i == 0 or $skip_only_first_line == 0)
	{
	    <$al_out>;
	}

	$i++;
    }

    my @expected_outlines = split("\n", $_[1]);

    my $result = 1;

    my $out;

    for ($i = 0; $i < scalar (@expected_outlines); $i++)
    {
	$out = <$al_out>;

	if (not defined ($out) and eof ($al_out))
	{
	    print "\n\nGot EOF from al, it probably crashed\n";

	    exit;
	}

	print $out;

	chomp ($out);

	if ($out ne $expected_outlines [$i])
	{
	    $result = 0;
	}
    }


    if ($result == 1)
    {
	print "OK!\n";
	$passed_tests++;
    }
    else
    {
	print "FAIL, expected " . $_[1] . " instead\n";
	$failed_tests++;
    }

    print "\n";

    $total_tests++;
}
