/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.utils;

import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;
import com.google.common.io.FileWriteMode;
import com.google.common.io.Files;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nullable;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.commons.compress.compressors.snappy.FramedSnappyCompressorInputStream;
import org.apache.commons.compress.compressors.xz.XZCompressorInputStream;
import org.apache.commons.compress.compressors.zstandard.ZstdCompressorInputStream;
import org.apache.commons.compress.utils.FileNameUtils;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.RetryUtils;
import org.apache.druid.java.util.common.StreamUtils;
import org.apache.druid.java.util.common.io.NativeIO;
import org.apache.druid.java.util.common.logger.Logger;

public class CompressionUtils {
    public static final int COMPRESSED_TEXT_WEIGHT_FACTOR = 4;
    private static final Logger log = new Logger(CompressionUtils.class);
    private static final int DEFAULT_RETRY_COUNT = 3;
    private static final int GZIP_BUFFER_SIZE = 8192;

    public static long zip(File directory, File outputZipFile, boolean fsync) throws IOException {
        if (!CompressionUtils.isZip(outputZipFile.getName())) {
            log.warn("No .zip suffix[%s], putting files from [%s] into it anyway.", outputZipFile, directory);
        }
        if (fsync) {
            return FileUtils.writeAtomically(outputZipFile, out -> CompressionUtils.zip(directory, out));
        }
        try (FileChannel fileChannel = FileChannel.open(outputZipFile.toPath(), StandardOpenOption.WRITE, StandardOpenOption.CREATE);){
            long l;
            block14: {
                OutputStream out2 = Channels.newOutputStream(fileChannel);
                try {
                    l = CompressionUtils.zip(directory, out2);
                    if (out2 == null) break block14;
                }
                catch (Throwable throwable) {
                    if (out2 != null) {
                        try {
                            out2.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                out2.close();
            }
            return l;
        }
    }

    public static long zip(File directory, File outputZipFile) throws IOException {
        return CompressionUtils.zip(directory, outputZipFile, false);
    }

    public static long zip(File directory, OutputStream out) throws IOException {
        if (!directory.isDirectory()) {
            throw new IOE("directory[%s] is not a directory", directory);
        }
        ZipOutputStream zipOut = new ZipOutputStream(out);
        long totalSize = 0L;
        for (File file : Arrays.stream(directory.listFiles()).sorted().collect(Collectors.toList())) {
            log.debug("Adding file[%s] with size[%,d].  Total size so far[%,d]", file, file.length(), totalSize);
            if (file.length() > Integer.MAX_VALUE) {
                zipOut.finish();
                throw new IOE("file[%s] too large [%,d]", file, file.length());
            }
            zipOut.putNextEntry(new ZipEntry(file.getName()));
            totalSize += Files.asByteSource((File)file).copyTo((OutputStream)zipOut);
        }
        zipOut.closeEntry();
        zipOut.flush();
        zipOut.finish();
        return totalSize;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static FileUtils.FileCopyResult unzip(ByteSource byteSource, File outDir, Predicate<Throwable> shouldRetry, boolean cacheLocally) throws IOException {
        FileUtils.FileCopyResult fileCopyResult;
        block7: {
            if (!cacheLocally) {
                try {
                    return RetryUtils.retry(() -> CompressionUtils.unzip(byteSource.openStream(), outDir), shouldRetry, 3);
                }
                catch (IOException e) {
                    throw e;
                }
                catch (Exception e) {
                    throw Throwables.propagate((Throwable)e);
                }
            }
            File tmpFile = File.createTempFile("compressionUtilZipCache", Format.ZIP.getSuffix());
            try {
                FileUtils.retryCopy(byteSource, tmpFile, shouldRetry, 3);
                fileCopyResult = CompressionUtils.unzip(tmpFile, outDir);
                if (tmpFile.delete()) break block7;
            }
            catch (Throwable throwable) {
                if (!tmpFile.delete()) {
                    log.warn("Could not delete zip cache at [%s]", tmpFile.toString());
                }
                throw throwable;
            }
            log.warn("Could not delete zip cache at [%s]", tmpFile.toString());
        }
        return fileCopyResult;
    }

    public static FileUtils.FileCopyResult unzip(File pulledFile, File outDir) throws IOException {
        if (!outDir.exists() || !outDir.isDirectory()) {
            throw new ISE("outDir[%s] must exist and be a directory", outDir);
        }
        log.info("Unzipping file[%s] to [%s]", pulledFile, outDir);
        FileUtils.FileCopyResult result = new FileUtils.FileCopyResult(new File[0]);
        try (final ZipFile zipFile = new ZipFile(pulledFile);){
            Enumeration<? extends ZipEntry> enumeration = zipFile.entries();
            while (enumeration.hasMoreElements()) {
                final ZipEntry entry = enumeration.nextElement();
                File outFile = new File(outDir, entry.getName());
                CompressionUtils.validateZipOutputFile(pulledFile.getCanonicalPath(), outFile, outDir);
                result.addFiles(FileUtils.retryCopy(new ByteSource(){

                    public InputStream openStream() throws IOException {
                        return new BufferedInputStream(zipFile.getInputStream(entry));
                    }
                }, outFile, FileUtils.IS_EXCEPTION, 3).getFiles());
            }
        }
        return result;
    }

    public static void validateZipOutputFile(String sourceFilename, File outFile, File outDir) throws IOException {
        File canonicalOutFile = outFile.getCanonicalFile();
        String canonicalOutDir = outDir.getCanonicalPath();
        if (!canonicalOutFile.toPath().startsWith(canonicalOutDir)) {
            throw new ISE("Unzipped output path[%s] of sourceFile[%s] does not start with outDir[%s].", canonicalOutFile, sourceFilename, canonicalOutDir);
        }
    }

    public static FileUtils.FileCopyResult unzip(InputStream in, File outDir) throws IOException {
        try (ZipInputStream zipIn = new ZipInputStream(in);){
            ZipEntry entry;
            FileUtils.FileCopyResult result = new FileUtils.FileCopyResult(new File[0]);
            while ((entry = zipIn.getNextEntry()) != null) {
                File file = new File(outDir, entry.getName());
                CompressionUtils.validateZipOutputFile("", file, outDir);
                NativeIO.chunkedCopy(zipIn, file);
                result.addFile(file);
                zipIn.closeEntry();
            }
            byte[] buf = new byte[512];
            while (in.read(buf) != -1) {
            }
            FileUtils.FileCopyResult fileCopyResult = result;
            return fileCopyResult;
        }
    }

    public static FileUtils.FileCopyResult gunzip(File pulledFile, File outFile) {
        return CompressionUtils.gunzip(Files.asByteSource((File)pulledFile), outFile);
    }

    public static FileUtils.FileCopyResult gunzip(InputStream in, File outFile) throws IOException {
        try (GZIPInputStream gzipInputStream = CompressionUtils.gzipInputStream(in);){
            NativeIO.chunkedCopy(gzipInputStream, outFile);
            FileUtils.FileCopyResult fileCopyResult = new FileUtils.FileCopyResult(outFile);
            return fileCopyResult;
        }
    }

    public static GZIPInputStream gzipInputStream(InputStream in) throws IOException {
        return new GZIPInputStream((InputStream)new FilterInputStream(in){

            @Override
            public int available() throws IOException {
                int otherAvailable = super.available();
                return otherAvailable == 0 ? 1024 : otherAvailable;
            }
        }, 8192);
    }

    public static long gunzip(InputStream in, OutputStream out) throws IOException {
        try {
            long l;
            block9: {
                GZIPInputStream gzipInputStream = CompressionUtils.gzipInputStream(in);
                try {
                    long result = ByteStreams.copy((InputStream)gzipInputStream, (OutputStream)out);
                    out.flush();
                    l = result;
                    if (gzipInputStream == null) break block9;
                }
                catch (Throwable throwable) {
                    if (gzipInputStream != null) {
                        try {
                            gzipInputStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                gzipInputStream.close();
            }
            return l;
        }
        finally {
            out.close();
        }
    }

    public static FileUtils.FileCopyResult gunzip(final ByteSource in, File outFile, Predicate<Throwable> shouldRetry) {
        return FileUtils.retryCopy(new ByteSource(){

            public InputStream openStream() throws IOException {
                return CompressionUtils.gzipInputStream(in.openStream());
            }
        }, outFile, shouldRetry, 3);
    }

    public static FileUtils.FileCopyResult gunzip(ByteSource in, File outFile) {
        return CompressionUtils.gunzip(in, outFile, FileUtils.IS_EXCEPTION);
    }

    public static long gzip(InputStream inputStream, OutputStream out) throws IOException {
        try {
            long l;
            try (GZIPOutputStream outputStream = new GZIPOutputStream(out);){
                long result = ByteStreams.copy((InputStream)inputStream, (OutputStream)outputStream);
                out.flush();
                l = result;
            }
            return l;
        }
        finally {
            inputStream.close();
        }
    }

    public static FileUtils.FileCopyResult gzip(File inFile, File outFile, Predicate<Throwable> shouldRetry) {
        CompressionUtils.gzip(Files.asByteSource((File)inFile), Files.asByteSink((File)outFile, (FileWriteMode[])new FileWriteMode[0]), shouldRetry);
        return new FileUtils.FileCopyResult(outFile);
    }

    public static long gzip(ByteSource in, final ByteSink out, Predicate<Throwable> shouldRetry) {
        return StreamUtils.retryCopy(in, new ByteSink(){

            public OutputStream openStream() throws IOException {
                return new GZIPOutputStream(out.openStream());
            }
        }, shouldRetry, 3);
    }

    public static FileUtils.FileCopyResult gzip(File inFile, File outFile) {
        return CompressionUtils.gzip(inFile, outFile, FileUtils.IS_EXCEPTION);
    }

    public static boolean isZip(String fName) {
        if (Strings.isNullOrEmpty((String)fName)) {
            return false;
        }
        return fName.endsWith(Format.ZIP.getSuffix());
    }

    public static boolean isGz(String fName) {
        if (Strings.isNullOrEmpty((String)fName)) {
            return false;
        }
        return fName.endsWith(Format.GZ.getSuffix()) && fName.length() > Format.GZ.getSuffix().length();
    }

    public static String getGzBaseName(String fname) {
        String reducedFname = Files.getNameWithoutExtension((String)fname);
        if (CompressionUtils.isGz(fname) && !reducedFname.isEmpty()) {
            return reducedFname;
        }
        throw new IAE("[%s] is not a valid gz file name", fname);
    }

    public static InputStream decompress(InputStream in, String fileName) throws IOException {
        if (fileName.endsWith(Format.GZ.getSuffix())) {
            return CompressionUtils.gzipInputStream(in);
        }
        if (fileName.endsWith(Format.BZ2.getSuffix())) {
            return new BZip2CompressorInputStream(in, true);
        }
        if (fileName.endsWith(Format.XZ.getSuffix())) {
            return new XZCompressorInputStream(in, true);
        }
        if (fileName.endsWith(Format.SNAPPY.getSuffix())) {
            return new FramedSnappyCompressorInputStream(in);
        }
        if (fileName.endsWith(Format.ZSTD.getSuffix())) {
            return new ZstdCompressorInputStream(in);
        }
        if (fileName.endsWith(Format.ZIP.getSuffix())) {
            ZipInputStream zipIn = new ZipInputStream(in, StandardCharsets.UTF_8);
            try {
                ZipEntry nextEntry = zipIn.getNextEntry();
                if (nextEntry == null) {
                    zipIn.close();
                    return new ByteArrayInputStream(new byte[0]);
                }
                return zipIn;
            }
            catch (IOException e) {
                try {
                    zipIn.close();
                }
                catch (IOException e2) {
                    e.addSuppressed(e2);
                }
                throw e;
            }
        }
        return in;
    }

    public static int estimatedCompressionFactor(@Nullable Format format) {
        if (format == Format.GZ) {
            return 4;
        }
        return 1;
    }

    public static enum Format {
        BZ2(".bz2", "bz2"),
        GZ(".gz", "gz"),
        SNAPPY(".sz", "sz"),
        XZ(".xz", "xz"),
        ZIP(".zip", "zip"),
        ZSTD(".zst", "zst");

        private static final Map<String, Format> EXTENSION_TO_COMPRESSION_FORMAT;
        private final String suffix;
        private final String extension;

        private Format(String suffix, String extension) {
            this.suffix = suffix;
            this.extension = extension;
        }

        public String getSuffix() {
            return this.suffix;
        }

        public String getExtension() {
            return this.extension;
        }

        @Nullable
        public static Format fromFileName(@Nullable String fileName) {
            String extension = FileNameUtils.getExtension((String)fileName);
            if (null == extension) {
                return null;
            }
            return EXTENSION_TO_COMPRESSION_FORMAT.get(extension);
        }

        static {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            builder.put((Object)BZ2.getExtension(), (Object)BZ2);
            builder.put((Object)GZ.getExtension(), (Object)GZ);
            builder.put((Object)SNAPPY.getExtension(), (Object)SNAPPY);
            builder.put((Object)XZ.getExtension(), (Object)XZ);
            builder.put((Object)ZIP.getExtension(), (Object)ZIP);
            builder.put((Object)ZSTD.getExtension(), (Object)ZSTD);
            EXTENSION_TO_COMPRESSION_FORMAT = builder.build();
        }
    }
}

