/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.s3.credential;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.gravitino.credential.AwsIrsaCredential;
import org.apache.gravitino.credential.CredentialContext;
import org.apache.gravitino.credential.CredentialGenerator;
import org.apache.gravitino.credential.PathBasedCredentialContext;
import org.apache.gravitino.credential.config.S3CredentialConfig;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.credentials.WebIdentityTokenFileCredentialsProvider;
import software.amazon.awssdk.policybuilder.iam.IamConditionOperator;
import software.amazon.awssdk.policybuilder.iam.IamEffect;
import software.amazon.awssdk.policybuilder.iam.IamPolicy;
import software.amazon.awssdk.policybuilder.iam.IamResource;
import software.amazon.awssdk.policybuilder.iam.IamStatement;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.sts.StsClient;
import software.amazon.awssdk.services.sts.StsClientBuilder;
import software.amazon.awssdk.services.sts.model.AssumeRoleWithWebIdentityRequest;
import software.amazon.awssdk.services.sts.model.AssumeRoleWithWebIdentityResponse;
import software.amazon.awssdk.services.sts.model.Credentials;

public class AwsIrsaCredentialGenerator
implements CredentialGenerator<AwsIrsaCredential> {
    private WebIdentityTokenFileCredentialsProvider baseCredentialsProvider;
    private String roleArn;
    private int tokenExpireSecs;
    private String region;
    private String stsEndpoint;

    public void initialize(Map<String, String> properties) {
        this.baseCredentialsProvider = WebIdentityTokenFileCredentialsProvider.create();
        S3CredentialConfig s3CredentialConfig = new S3CredentialConfig(properties);
        this.roleArn = s3CredentialConfig.s3RoleArn();
        this.tokenExpireSecs = s3CredentialConfig.tokenExpireInSecs();
        this.region = s3CredentialConfig.region();
        this.stsEndpoint = s3CredentialConfig.stsEndpoint();
    }

    public AwsIrsaCredential generate(CredentialContext context) {
        if (!(context instanceof PathBasedCredentialContext)) {
            AwsCredentials creds = this.baseCredentialsProvider.resolveCredentials();
            if (creds instanceof AwsSessionCredentials) {
                AwsSessionCredentials sessionCreds = (AwsSessionCredentials)creds;
                long expiration = sessionCreds.expirationTime().isPresent() ? ((Instant)sessionCreds.expirationTime().get()).toEpochMilli() : 0L;
                return new AwsIrsaCredential(sessionCreds.accessKeyId(), sessionCreds.secretAccessKey(), sessionCreds.sessionToken(), expiration);
            }
            throw new IllegalStateException("AWS IRSA credentials must be of type AwsSessionCredentials. Check your EKS/IRSA configuration. Got: " + creds.getClass().getName());
        }
        PathBasedCredentialContext pathBasedCredentialContext = (PathBasedCredentialContext)context;
        Credentials s3Token = this.createCredentialsWithSessionPolicy(pathBasedCredentialContext.getReadPaths(), pathBasedCredentialContext.getWritePaths(), pathBasedCredentialContext.getUserName());
        return new AwsIrsaCredential(s3Token.accessKeyId(), s3Token.secretAccessKey(), s3Token.sessionToken(), s3Token.expiration().toEpochMilli());
    }

    private Credentials createCredentialsWithSessionPolicy(Set<String> readLocations, Set<String> writeLocations, String userName) {
        this.validateInputParameters(readLocations, writeLocations, userName);
        IamPolicy sessionPolicy = this.createSessionPolicy(readLocations, writeLocations, this.region);
        String webIdentityTokenFile = this.getValidatedWebIdentityTokenFile();
        String effectiveRoleArn = this.getValidatedRoleArn(this.roleArn);
        try {
            String tokenContent = new String(Files.readAllBytes(Paths.get(webIdentityTokenFile, new String[0])), StandardCharsets.UTF_8);
            if (StringUtils.isBlank((CharSequence)tokenContent)) {
                throw new IllegalStateException("Web identity token file is empty: " + webIdentityTokenFile);
            }
            return this.assumeRoleWithSessionPolicy(effectiveRoleArn, userName, tokenContent, sessionPolicy);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to create credentials with session policy for user: " + userName, e);
        }
    }

    private IamPolicy createSessionPolicy(Set<String> readLocations, Set<String> writeLocations, String region) {
        IamPolicy.Builder policyBuilder = IamPolicy.builder();
        String arnPrefix = this.getArnPrefix(region);
        this.addReadPermissions(policyBuilder, readLocations, writeLocations, arnPrefix);
        if (!writeLocations.isEmpty()) {
            this.addWritePermissions(policyBuilder, writeLocations, arnPrefix);
        }
        this.addBucketPermissions(policyBuilder, readLocations, writeLocations, arnPrefix);
        return (IamPolicy)policyBuilder.build();
    }

    private void addReadPermissions(IamPolicy.Builder policyBuilder, Set<String> readLocations, Set<String> writeLocations, String arnPrefix) {
        IamStatement.Builder allowGetObjectStatementBuilder = IamStatement.builder().effect(IamEffect.ALLOW).addAction("s3:GetObject").addAction("s3:GetObjectVersion");
        Stream.concat(readLocations.stream(), writeLocations.stream()).distinct().forEach(location -> {
            URI uri = URI.create(location);
            allowGetObjectStatementBuilder.addResource(IamResource.create((String)this.getS3UriWithArn(arnPrefix, uri)));
        });
        policyBuilder.addStatement((IamStatement)allowGetObjectStatementBuilder.build());
    }

    private void addWritePermissions(IamPolicy.Builder policyBuilder, Set<String> writeLocations, String arnPrefix) {
        IamStatement.Builder allowPutObjectStatementBuilder = IamStatement.builder().effect(IamEffect.ALLOW).addAction("s3:PutObject").addAction("s3:DeleteObject");
        writeLocations.forEach(location -> {
            URI uri = URI.create(location);
            allowPutObjectStatementBuilder.addResource(IamResource.create((String)this.getS3UriWithArn(arnPrefix, uri)));
        });
        policyBuilder.addStatement((IamStatement)allowPutObjectStatementBuilder.build());
    }

    private void addBucketPermissions(IamPolicy.Builder policyBuilder, Set<String> readLocations, Set<String> writeLocations, String arnPrefix) {
        HashMap<String, IamStatement.Builder> bucketListStatementBuilder = new HashMap<String, IamStatement.Builder>();
        HashMap<String, IamStatement.Builder> bucketGetLocationStatementBuilder = new HashMap<String, IamStatement.Builder>();
        Stream.concat(readLocations.stream(), writeLocations.stream()).distinct().forEach(location -> {
            URI uri = URI.create(location);
            String bucketArn = arnPrefix + AwsIrsaCredentialGenerator.getBucketName(uri);
            String rawPath = AwsIrsaCredentialGenerator.trimLeadingSlash(uri.getPath());
            bucketListStatementBuilder.computeIfAbsent(bucketArn, key -> IamStatement.builder().effect(IamEffect.ALLOW).addAction("s3:ListBucket").addResource(key)).addConditions(IamConditionOperator.STRING_LIKE, "s3:prefix", Arrays.asList(rawPath, AwsIrsaCredentialGenerator.addWildcardToPath(rawPath)));
            bucketGetLocationStatementBuilder.computeIfAbsent(bucketArn, key -> IamStatement.builder().effect(IamEffect.ALLOW).addAction("s3:GetBucketLocation").addResource(key));
        });
        this.addStatementsToPolicy(policyBuilder, bucketListStatementBuilder);
        this.addStatementsToPolicy(policyBuilder, bucketGetLocationStatementBuilder);
    }

    private void addStatementsToPolicy(IamPolicy.Builder policyBuilder, Map<String, IamStatement.Builder> statementBuilders) {
        statementBuilders.values().forEach(builder -> policyBuilder.addStatement((IamStatement)builder.build()));
    }

    private String getS3UriWithArn(String arnPrefix, URI uri) {
        return arnPrefix + AwsIrsaCredentialGenerator.addWildcardToPath(AwsIrsaCredentialGenerator.removeSchemaFromS3Uri(uri));
    }

    private String getArnPrefix(String region) {
        if (StringUtils.isNotBlank((CharSequence)region)) {
            if (region.contains("cn-")) {
                return "arn:aws-cn:s3:::";
            }
            if (region.contains("us-gov-")) {
                return "arn:aws-us-gov:s3:::";
            }
        }
        return "arn:aws:s3:::";
    }

    private static String addWildcardToPath(String path) {
        return path.endsWith("/") ? path + "*" : path + "/*";
    }

    private static String removeSchemaFromS3Uri(URI uri) {
        String bucket = uri.getHost();
        String path = AwsIrsaCredentialGenerator.trimLeadingSlash(uri.getPath());
        return String.join((CharSequence)"/", (CharSequence[])Stream.of(bucket, path).filter(Objects::nonNull).toArray(String[]::new));
    }

    private static String trimLeadingSlash(String path) {
        return path.startsWith("/") ? path.substring(1) : path;
    }

    private static String getBucketName(URI uri) {
        return uri.getHost();
    }

    private void validateInputParameters(Set<String> readLocations, Set<String> writeLocations, String userName) {
        if (StringUtils.isBlank((CharSequence)userName)) {
            throw new IllegalArgumentException("userName cannot be null or empty");
        }
        if ((readLocations == null || readLocations.isEmpty()) && (writeLocations == null || writeLocations.isEmpty())) {
            throw new IllegalArgumentException("At least one read or write location must be specified");
        }
    }

    private String getValidatedWebIdentityTokenFile() {
        String webIdentityTokenFile = System.getenv("AWS_WEB_IDENTITY_TOKEN_FILE");
        if (StringUtils.isBlank((CharSequence)webIdentityTokenFile)) {
            throw new IllegalStateException("AWS_WEB_IDENTITY_TOKEN_FILE environment variable is not set. Ensure IRSA is properly configured in your EKS cluster.");
        }
        if (!Files.exists(Paths.get(webIdentityTokenFile, new String[0]), new LinkOption[0])) {
            throw new IllegalStateException("Web identity token file does not exist: " + webIdentityTokenFile);
        }
        return webIdentityTokenFile;
    }

    private String getValidatedRoleArn(String configRoleArn) {
        String effectiveRoleArn;
        String string = effectiveRoleArn = StringUtils.isNotBlank((CharSequence)configRoleArn) ? configRoleArn : System.getenv("AWS_ROLE_ARN");
        if (StringUtils.isBlank((CharSequence)effectiveRoleArn)) {
            throw new IllegalStateException("No role ARN available. Either configure s3-role-arn or ensure AWS_ROLE_ARN environment variable is set.");
        }
        if (!effectiveRoleArn.startsWith("arn:aws")) {
            throw new IllegalArgumentException("Invalid role ARN format: " + effectiveRoleArn);
        }
        return effectiveRoleArn;
    }

    private Credentials assumeRoleWithSessionPolicy(String roleArn, String userName, String webIdentityToken, IamPolicy sessionPolicy) {
        StsClientBuilder stsBuilder = StsClient.builder();
        if (StringUtils.isNotBlank((CharSequence)this.region)) {
            stsBuilder.region(Region.of((String)this.region));
        }
        if (StringUtils.isNotBlank((CharSequence)this.stsEndpoint)) {
            stsBuilder.endpointOverride(URI.create(this.stsEndpoint));
        }
        try (StsClient stsClient = (StsClient)stsBuilder.build();){
            AssumeRoleWithWebIdentityRequest request = (AssumeRoleWithWebIdentityRequest)AssumeRoleWithWebIdentityRequest.builder().roleArn(roleArn).roleSessionName("gravitino_irsa_session_" + userName).durationSeconds(Integer.valueOf(this.tokenExpireSecs)).webIdentityToken(webIdentityToken).policy(sessionPolicy.toJson()).build();
            AssumeRoleWithWebIdentityResponse response = stsClient.assumeRoleWithWebIdentity(request);
            Credentials credentials = response.credentials();
            return credentials;
        }
    }

    public void close() throws IOException {
    }
}

