/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.ozone.debug.ratis.parse;

import java.util.concurrent.Callable;
import org.apache.hadoop.hdds.cli.HddsVersionProvider;
import org.apache.hadoop.hdds.scm.ha.SCMRatisRequest;
import org.apache.hadoop.ozone.container.common.transport.server.ratis.ContainerStateMachine;
import org.apache.hadoop.ozone.om.helpers.OMRatisHelper;
import org.apache.ratis.proto.RaftProtos.StateMachineLogEntryProto;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import picocli.CommandLine;

/**
 * Parse Ratis Log CLI implementation.
 */
@CommandLine.Command(
    name = "parse",
    description = "Shell for printing Ratis Log in understandable text",
    versionProvider = HddsVersionProvider.class,
    mixinStandardHelpOptions = true)
public class RatisLogParser extends BaseLogParser implements Callable<Void> {
  @CommandLine.Option(names = {"--role"},
      defaultValue = "generic",
      showDefaultValue = CommandLine.Help.Visibility.ALWAYS,
      description = "Component role for parsing. Values: om, scm, datanode")
  private String role;

  private static final RaftGroupId DUMMY_PIPELINE_ID =
      RaftGroupId.valueOf(ByteString.copyFromUtf8("ADummyRatisGroup"));

  public static String smToContainerLogString(
      StateMachineLogEntryProto logEntryProto) {
    return ContainerStateMachine.smProtoToString(DUMMY_PIPELINE_ID, null, logEntryProto);
  }

  @Override
  public Void call() throws Exception {
    switch (role.toLowerCase()) {
    case "om":
      System.out.println("Dumping OM Ratis Log");
      parseRatisLogs(OMRatisHelper::smProtoToString);
      break;
    case "scm":
      System.out.println("Dumping SCM Ratis Log");
      parseRatisLogs(SCMRatisRequest::smProtoToString);
      break;
    case "datanode":
      System.out.println("Dumping Datanode Ratis Log");
      System.out.println("Using Dummy PipelineID:" + DUMMY_PIPELINE_ID + "\n\n");
      parseRatisLogs(RatisLogParser::smToContainerLogString);
      break;
    default:
      System.out.println("Dumping Generic Ratis Log");
      parseRatisLogs(null);
      break;
    }
    return null;
  }
}
