/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.utils;

import com.google.common.annotations.VisibleForTesting;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdds.HddsUtils;
import org.apache.hadoop.hdds.utils.FaultInjector;
import org.apache.hadoop.hdds.utils.HAUtils;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.hdds.utils.db.DBCheckpoint;
import org.apache.hadoop.hdds.utils.db.RocksDBCheckpoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class RDBSnapshotProvider
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(RDBSnapshotProvider.class);
    private final File snapshotDir;
    private final File candidateDir;
    private final String dbName;
    private final AtomicReference<String> lastLeaderRef;
    private final AtomicLong numDownloaded;
    private FaultInjector injector;
    private final AtomicLong initCount;

    public RDBSnapshotProvider(File snapshotDir, String dbName) {
        this.snapshotDir = snapshotDir;
        this.candidateDir = new File(snapshotDir, dbName + ".candidate");
        this.dbName = dbName;
        this.injector = null;
        this.lastLeaderRef = new AtomicReference<Object>(null);
        this.numDownloaded = new AtomicLong();
        this.initCount = new AtomicLong();
        this.init();
    }

    public synchronized void init() {
        if (!this.snapshotDir.exists()) {
            HddsUtils.createDir((String)this.snapshotDir.toString());
        }
        LOG.info("Cleaning up the candidate dir: {}", (Object)this.candidateDir);
        if (this.candidateDir.exists()) {
            FileUtil.fullyDeleteContents((File)this.candidateDir);
        } else {
            HddsUtils.createDir((String)this.candidateDir.toString());
        }
        this.lastLeaderRef.set(null);
        this.initCount.incrementAndGet();
    }

    public DBCheckpoint downloadDBSnapshotFromLeader(String leaderNodeID) throws IOException {
        RocksDBCheckpoint checkpoint;
        LOG.info("Prepare to download the snapshot from leader OM {} and reloading state from the snapshot.", (Object)leaderNodeID);
        this.checkLeaderConsistency(leaderNodeID);
        do {
            String snapshotFileName = this.getSnapshotFileName(leaderNodeID);
            File targetFile = new File(this.snapshotDir, snapshotFileName);
            this.downloadSnapshot(leaderNodeID, targetFile);
            LOG.info("Successfully download the latest snapshot {} from leader OM: {}", (Object)targetFile, (Object)leaderNodeID);
            this.numDownloaded.incrementAndGet();
            this.injectPause();
            checkpoint = this.getCheckpointFromSnapshotFile(targetFile, this.candidateDir, true);
            LOG.info("Successfully untar the downloaded snapshot {} at {}.", (Object)targetFile, (Object)checkpoint.getCheckpointLocation());
        } while (!HddsServerUtil.ratisSnapshotComplete(checkpoint.getCheckpointLocation()));
        LOG.info("Ratis snapshot transfer is complete.");
        return checkpoint;
    }

    @VisibleForTesting
    void checkLeaderConsistency(String currentLeader) throws IOException {
        String lastLeader = this.lastLeaderRef.get();
        if (lastLeader != null) {
            if (!lastLeader.equals(currentLeader)) {
                LOG.info("Last leader for install snapshot is {}, but current leader is {}. ", (Object)lastLeader, (Object)currentLeader);
                this.init();
                this.lastLeaderRef.set(currentLeader);
            }
            return;
        }
        List<String> files = HAUtils.getExistingSstFiles(this.candidateDir);
        if (!files.isEmpty()) {
            LOG.warn("Candidate DB directory {} is not empty when last leader is null.", (Object)this.candidateDir);
            this.init();
        }
        this.lastLeaderRef.set(currentLeader);
    }

    public String getSnapshotFileName(String leaderNodeID) {
        String snapshotTime = Long.toString(System.currentTimeMillis());
        return this.dbName + "-" + leaderNodeID + "-" + snapshotTime + ".tar";
    }

    public RocksDBCheckpoint getCheckpointFromSnapshotFile(File snapshot, File untarDir, boolean deleteSnapshot) throws IOException {
        Path untarredDbDir = untarDir.toPath();
        FileUtil.unTar((File)snapshot, (File)untarredDbDir.toFile());
        if (deleteSnapshot) {
            FileUtil.fullyDelete((File)snapshot);
        }
        return new RocksDBCheckpoint(untarredDbDir);
    }

    public abstract void downloadSnapshot(String var1, File var2) throws IOException;

    private void injectPause() throws IOException {
        if (this.injector != null) {
            this.injector.pause();
        }
    }

    @VisibleForTesting
    public File getSnapshotDir() {
        return this.snapshotDir;
    }

    @VisibleForTesting
    public File getCandidateDir() {
        return this.candidateDir;
    }

    @VisibleForTesting
    public FaultInjector getInjector() {
        return this.injector;
    }

    @VisibleForTesting
    public void setInjector(FaultInjector injector) {
        this.injector = injector;
    }

    @VisibleForTesting
    public long getNumDownloaded() {
        return this.numDownloaded.get();
    }

    @VisibleForTesting
    public long getInitCount() {
        return this.initCount.get();
    }
}

