/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Configuration,
  TalerWireGatewayHttpClient,
  TransactionIdStr,
  TransactionMajorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

function adjustExchangeConfig(config: Configuration): void {
  configureCommonKyc(config);

  config.setString("KYC-RULE-R1", "operation_type", "balance");
  config.setString("KYC-RULE-R1", "enabled", "yes");
  config.setString("KYC-RULE-R1", "exposed", "yes");
  config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
  config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:10");
  config.setString("KYC-RULE-R1", "timeframe", "forever");
  config.setString("KYC-RULE-R1", "next_measures", "M1");

  config.setString("KYC-MEASURE-M1", "check_name", "C1");
  config.setString("KYC-MEASURE-M1", "context", "{}");
  config.setString("KYC-MEASURE-M1", "program", "NONE");

  config.setString("KYC-CHECK-C1", "type", "INFO");
  config.setString("KYC-CHECK-C1", "description", "my check!");
  config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
}

export async function runKycBalanceWithdrawalChangeManualTest(
  t: GlobalTestState,
) {
  // Set up test environment

  const { walletClient, bankClient, exchange, exchangeBankAccount, bank } =
    await createKycTestkudosEnvironment(t, {
      adjustExchangeConfig,
    });

  const mwResp = await walletClient.call(
    WalletApiOperation.AcceptManualWithdrawal,
    {
      // Specify larger amount than what will be in the reserve.
      amount: "TESTKUDOS:20",
      exchangeBaseUrl: exchange.baseUrl,
    },
  );

  await exchange.stop();

  await exchange.modifyConfig(async (config) => {
    config.setString("KYC-RULE-R1", "enabled", "NO");
  });

  await exchange.start();

  const user = await bankClient.createRandomBankUser();

  // Next, do a manual withdrawal.
  const wireGatewayApiClient = new TalerWireGatewayHttpClient(
    exchangeBankAccount.wireGatewayApiBaseUrl,
  );

  const reservePub = mwResp.reservePub;

  await wireGatewayApiClient.addIncoming({
    auth: bank.getAdminAuth(),
    body: {
      amount: "TESTKUDOS:20",
      debit_account: user.accountPaytoUri,
      reserve_pub: reservePub,
    },
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: mwResp.transactionId as TransactionIdStr,
    txState: {
      major: TransactionMajorState.Done,
    },
  });
}

runKycBalanceWithdrawalChangeManualTest.suites = ["wallet"];
