/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { assertUnreachable } from "./errors.js";
import { canonicalJson } from "./helpers.js";
import { Logger } from "./logging.js";
import {
  decodeCrock,
  encodeCrock,
  getRandomBytes,
  hash,
  kdf,
  stringToBytes,
} from "./taler-crypto.js";
import {
  AmountString,
  Integer,
} from "./types-taler-common.js";
import {
  MerchantContractTerms,
  MerchantContractTermsV0,
  MerchantContractTermsV1,
  MerchantContractTokenKind,
  MerchantContractVersion,
} from "./types-taler-merchant.js";

const logger = new Logger("contractTerms.ts");

export namespace ContractTermsUtil {
  export function forgetAllImpl(
    anyJson: any,
    path: string[],
    pred: PathPredicate,
  ): any {
    const dup = JSON.parse(JSON.stringify(anyJson));
    if (Array.isArray(dup)) {
      for (let i = 0; i < dup.length; i++) {
        dup[i] = forgetAllImpl(dup[i], [...path, `${i}`], pred);
      }
    } else if (typeof dup === "object" && dup != null) {
      if (typeof dup.$forgettable === "object") {
        for (const x of Object.keys(dup.$forgettable)) {
          if (!pred([...path, x])) {
            continue;
          }
          if (!dup.$forgotten) {
            dup.$forgotten = {};
          }
          if (!dup.$forgotten[x]) {
            const membValCanon = stringToBytes(
              canonicalJson(scrub(dup[x])) + "\0",
            );
            const membSalt = stringToBytes(dup.$forgettable[x] + "\0");
            const h = kdf(64, membValCanon, membSalt, new Uint8Array([]));
            dup.$forgotten[x] = encodeCrock(h);
          }
          delete dup[x];
          delete dup.$forgettable[x];
        }
        if (Object.keys(dup.$forgettable).length === 0) {
          delete dup.$forgettable;
        }
      }
      for (const x of Object.keys(dup)) {
        if (x.startsWith("$")) {
          continue;
        }
        dup[x] = forgetAllImpl(dup[x], [...path, x], pred);
      }
    }
    return dup;
  }

  export type PathPredicate = (path: string[]) => boolean;

  /**
   * Scrub all forgettable members from an object.
   */
  export function scrub(anyJson: any): any {
    return forgetAllImpl(anyJson, [], () => true);
  }

  /**
   * Recursively forget all forgettable members of an object,
   * where the path matches a predicate.
   */
  export function forgetAll(anyJson: any, pred: PathPredicate): any {
    return forgetAllImpl(anyJson, [], pred);
  }

  /**
   * Generate a salt for all members marked as forgettable,
   * but which don't have an actual salt yet.
   */
  export function saltForgettable(anyJson: any): any {
    const dup = JSON.parse(JSON.stringify(anyJson));
    if (Array.isArray(dup)) {
      for (let i = 0; i < dup.length; i++) {
        dup[i] = saltForgettable(dup[i]);
      }
    } else if (typeof dup === "object" && dup !== null) {
      if (typeof dup.$forgettable === "object") {
        for (const k of Object.keys(dup.$forgettable)) {
          if (dup.$forgettable[k] === true) {
            dup.$forgettable[k] = encodeCrock(getRandomBytes(32));
          }
        }
      }
      for (const x of Object.keys(dup)) {
        if (x.startsWith("$")) {
          continue;
        }
        dup[x] = saltForgettable(dup[x]);
      }
    }
    return dup;
  }

  const nameRegex = /^[0-9A-Za-z_]+$/;

  /**
   * Check that the given JSON object is well-formed with regards
   * to forgettable fields and other restrictions for forgettable JSON.
   */
  export function validateForgettable(anyJson: any): boolean {
    if (anyJson === undefined) {
      return true;
    }
    if (typeof anyJson === "string") {
      return true;
    }
    if (typeof anyJson === "number") {
      return (
        Number.isInteger(anyJson) &&
        anyJson >= Number.MIN_SAFE_INTEGER &&
        anyJson <= Number.MAX_SAFE_INTEGER
      );
    }
    if (typeof anyJson === "boolean") {
      return true;
    }
    if (anyJson === null) {
      return true;
    }
    if (Array.isArray(anyJson)) {
      return anyJson.every((x) => validateForgettable(x));
    }
    if (typeof anyJson === "object") {
      for (const k of Object.keys(anyJson)) {
        if (k.match(nameRegex)) {
          if (validateForgettable(anyJson[k])) {
            continue;
          } else {
            return false;
          }
        }
        if (k === "$forgettable") {
          const fga = anyJson.$forgettable;
          if (!fga || typeof fga !== "object") {
            return false;
          }
          for (const fk of Object.keys(fga)) {
            if (!fk.match(nameRegex)) {
              return false;
            }
            if (!(fk in anyJson)) {
              return false;
            }
            const fv = anyJson.$forgettable[fk];
            if (typeof fv !== "string") {
              return false;
            }
          }
        } else if (k === "$forgotten") {
          const fgo = anyJson.$forgotten;
          if (!fgo || typeof fgo !== "object") {
            return false;
          }
          for (const fk of Object.keys(fgo)) {
            if (!fk.match(nameRegex)) {
              return false;
            }
            // Check that the value has actually been forgotten.
            if (fk in anyJson) {
              return false;
            }
            const fv = anyJson.$forgotten[fk];
            if (typeof fv !== "string") {
              return false;
            }
            try {
              const decFv = decodeCrock(fv);
              if (decFv.length != 64) {
                return false;
              }
            } catch (e) {
              return false;
            }
            // Check that salt has been deleted after forgetting.
            if (anyJson.$forgettable?.[k] !== undefined) {
              return false;
            }
          }
        } else {
          return false;
        }
      }
      return true;
    }
    return false;
  }

  /**
   * Check that no forgettable information has been forgotten.
   *
   * Must only be called on an object already validated with validateForgettable.
   */
  export function validateNothingForgotten(contractTerms: any): boolean {
    throw Error("not implemented yet");
  }

  export function validateParsed(contractTerms: MerchantContractTerms): boolean {
    // validate trusted/expected domains
    if (contractTerms.version === MerchantContractVersion.V1) {
      const regex = new RegExp("^(\\*\\.)?([\\w\\d]+\\.)+[\\w\\d]+$");
      for (const slug in contractTerms.token_families) {
        const family = contractTerms.token_families[slug];
        var domains: string[] = [];
        switch (family.details.class) {
          case MerchantContractTokenKind.Subscription:
            domains.push(...family.details.trusted_domains);
            break;
          case MerchantContractTokenKind.Discount:
            domains.push(...family.details.expected_domains);
            break;
          default:
            assertUnreachable(family.details);
        }

        for (const domain in domains) {
          if (domain !== "*" && !regex.test(domain)) {
            return false;
          }
        }
      }
    }

    return true;
  }

  /**
   * Hash a contract terms object.  Forgettable fields
   * are scrubbed and JSON canonicalization is applied
   * before hashing.
   */
  export function hashContractTerms(contractTerms: unknown): string {
    const cleaned = scrub(contractTerms);
    const canon = canonicalJson(cleaned) + "\0";
    const bytes = stringToBytes(canon);
    return encodeCrock(hash(bytes));
  }

  /**
   * Extract raw amount and max fee.
   */
  export function extractAmounts(
    contractTerms: MerchantContractTerms,
    choiceIndex: Integer | undefined,
  ): {
    available: true,
    amountRaw: AmountString,
    maxFee: AmountString,
  } | {
    available: false,
    amountRaw: undefined,
    maxFee: undefined,
  } {
    let amountRaw: AmountString;
    let maxFee: AmountString;
    switch (contractTerms.version) {
      case undefined:
      case MerchantContractVersion.V0:
        amountRaw = contractTerms.amount;
        maxFee = contractTerms.max_fee;
        break;
      case MerchantContractVersion.V1:
        if (choiceIndex === undefined) {
          logger.trace("choice index not specified for contract v1");
          return {
            available: false,
            amountRaw: undefined,
            maxFee: undefined,
          };
        }
        if (contractTerms.choices[choiceIndex] === undefined)
          throw Error(`invalid choice index ${choiceIndex}`);
        amountRaw = contractTerms.choices[choiceIndex].amount;
        maxFee = contractTerms.choices[choiceIndex].max_fee;
        break;
      default:
        assertUnreachable(contractTerms);
    }

    return {
      available: true,
      amountRaw,
      maxFee,
    };
  }

  export function getV0CompatChoiceIndex(terms: MerchantContractTermsV1): number | undefined {
    // Select the first choice that doesn't have
    // and non-currency inputs.
    let firstGood: number | undefined = undefined;
    for (let i = 0; i < terms.choices.length; i++) {
      if (terms.choices[i].inputs.length == 0) {
        firstGood = i;
        break;
      }
    }
    if (firstGood != null) {
      return firstGood;
    }
    return undefined;
  }

  /**
   * Try to downgrade contract terms in the v1 format to the v0 format.
   * Returns undefined if downgrading is not possible. This can happen
   * when the contract only offers token payments.
   */
  export function downgradeContractTerms(terms: MerchantContractTerms): MerchantContractTermsV0 | undefined {
    if (terms.version == MerchantContractVersion.V0) {
      return terms;
    }
    if (terms.version !== MerchantContractVersion.V1) {
      return undefined;
    }
    const firstGood = getV0CompatChoiceIndex(terms);
    if (firstGood == null) {
      return undefined;
    }
    return {
      amount: terms.choices[firstGood].amount,
      exchanges: terms.exchanges,
      h_wire: terms.h_wire,
      max_fee: terms.choices[firstGood].max_fee,
      merchant: terms.merchant,
      merchant_base_url: terms.merchant_base_url,
      merchant_pub: terms.merchant_pub,
      nonce: terms.nonce,
      order_id: terms.order_id,
      pay_deadline: terms.pay_deadline,
      refund_deadline: terms.refund_deadline,
      summary: terms.summary,
      timestamp: terms.timestamp,
      wire_method: terms.wire_method,
      wire_transfer_deadline: terms.wire_transfer_deadline,
      auto_refund: terms.auto_refund,
      delivery_date: terms.delivery_date,
      delivery_location: terms.delivery_location,
      extra: terms.extra,
      fulfillment_message: terms.fulfillment_message,
      fulfillment_message_i18n: terms.fulfillment_message_i18n,
      fulfillment_url: terms.fulfillment_url,
      minimum_age: terms.minimum_age,
      products: terms.products,
      public_reorder_url: terms.public_reorder_url,
      summary_i18n: terms.summary_i18n,
      version: MerchantContractVersion.V0,
    }
  }
}
