﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/qapps/QAppsRequest.h>
#include <aws/qapps/QApps_EXPORTS.h>
#include <aws/qapps/model/CardValue.h>

#include <utility>

namespace Aws {
namespace QApps {
namespace Model {

/**
 */
class StartQAppSessionRequest : public QAppsRequest {
 public:
  AWS_QAPPS_API StartQAppSessionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartQAppSession"; }

  AWS_QAPPS_API Aws::String SerializePayload() const override;

  AWS_QAPPS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The unique identifier of the Amazon Q Business application environment
   * instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  StartQAppSessionRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the Q App to start a session for.</p>
   */
  inline const Aws::String& GetAppId() const { return m_appId; }
  inline bool AppIdHasBeenSet() const { return m_appIdHasBeenSet; }
  template <typename AppIdT = Aws::String>
  void SetAppId(AppIdT&& value) {
    m_appIdHasBeenSet = true;
    m_appId = std::forward<AppIdT>(value);
  }
  template <typename AppIdT = Aws::String>
  StartQAppSessionRequest& WithAppId(AppIdT&& value) {
    SetAppId(std::forward<AppIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the Q App to use for the session.</p>
   */
  inline int GetAppVersion() const { return m_appVersion; }
  inline bool AppVersionHasBeenSet() const { return m_appVersionHasBeenSet; }
  inline void SetAppVersion(int value) {
    m_appVersionHasBeenSet = true;
    m_appVersion = value;
  }
  inline StartQAppSessionRequest& WithAppVersion(int value) {
    SetAppVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional initial input values to provide for the Q App session.</p>
   */
  inline const Aws::Vector<CardValue>& GetInitialValues() const { return m_initialValues; }
  inline bool InitialValuesHasBeenSet() const { return m_initialValuesHasBeenSet; }
  template <typename InitialValuesT = Aws::Vector<CardValue>>
  void SetInitialValues(InitialValuesT&& value) {
    m_initialValuesHasBeenSet = true;
    m_initialValues = std::forward<InitialValuesT>(value);
  }
  template <typename InitialValuesT = Aws::Vector<CardValue>>
  StartQAppSessionRequest& WithInitialValues(InitialValuesT&& value) {
    SetInitialValues(std::forward<InitialValuesT>(value));
    return *this;
  }
  template <typename InitialValuesT = CardValue>
  StartQAppSessionRequest& AddInitialValues(InitialValuesT&& value) {
    m_initialValuesHasBeenSet = true;
    m_initialValues.emplace_back(std::forward<InitialValuesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier of the a Q App session.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  StartQAppSessionRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional tags to associate with the new Q App session.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartQAppSessionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartQAppSessionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_appId;

  int m_appVersion{0};

  Aws::Vector<CardValue> m_initialValues;

  Aws::String m_sessionId;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_instanceIdHasBeenSet = false;
  bool m_appIdHasBeenSet = false;
  bool m_appVersionHasBeenSet = false;
  bool m_initialValuesHasBeenSet = false;
  bool m_sessionIdHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace QApps
}  // namespace Aws
