;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2017, 2018, 2019 Rob Wieringa <rma.wieringa@gmail.com>
;;; Copyright © 2017, 2021, 2022, 2023 Rutger van Beusekom <rutger@dezyne.org>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (dzn commands code)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)

  #:use-module (ice-9 getopt-long)
  #:use-module (ice-9 poe)

  #:use-module (dzn ast)
  #:use-module (dzn ast ast)
  #:use-module (dzn ast lookup)
  #:use-module (dzn code)
  #:use-module (dzn config)
  #:use-module (dzn misc)
  #:use-module (dzn shell-util)
  #:use-module (dzn command-line)
  #:use-module (dzn commands parse)
  #:use-module (dzn parse)
  #:use-module (dzn shell-util)
  #:export (%languages
            parse-opts
            main))

(define %default-language "c++")
(define (list-languages dir)
  (let* ((uninstalled? (getenv "DZN_UNINSTALLED"))
         (ext path (if uninstalled? (values ".scm" %load-path)
                       (values ".go" %load-compiled-path)))
         (regex (string-append "\\" ext "$")))
    (map (cut basename <> ext)
         (filter (cute string-contains <> dir)
                 (append-map (cute find-files <> regex)
                             (filter directory-exists? path))))))

(define list-languages (pure-funcq list-languages))

(define %languages
  (sort (delete-duplicates (list-languages "/dzn/code/language") string=?)
        string<))

(define (parse-opts args)
  (let* ((option-spec
          '((calling-context (single-char #\c) (value #t))
            (help (single-char #\h))
            (import (single-char #\I) (value #t))
            (init (value #t))
            (language (single-char #\l) (value #t))
            (locations (single-char #\L))
            (model (single-char #\m) (value #t))
            (no-constraint (single-char #\C))
            (no-unreachable (single-char #\U))
            (output (single-char #\o) (value #t))
            (queue-size (single-char #\q) (value #t))
            (queue-size-defer (value #t))
            (queue-size-external (value #t))
            (shell (single-char #\s) (value #t))
            (touch-empty-files (single-char #\t))))
         (options (getopt-long args option-spec))
         (help? (option-ref options 'help #f))
         (files (option-ref options '() '()))
         (usage? (and (not help?) (null? files))))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: dzn code [OPTION]... {DZN-FILE|DIRECTORY}...
Generate code for Dezyne models in DZN-FILEs and DIRECTORY(s)

  -c, --calling-context=TYPE  generate extra parameter of TYPE for every event
  -C, --no-constraint         do not use a constraining process
  -h, --help                  display this help and exit
  -I, --import=DIR+           add DIR to import path
      --init=PROCESS          use init PROCESS for mCRL2
  -l, --language=LANG         generate code for language=LANG [~a]
  -L, --locations             prepend locations to output trace
  -m, --model=MODEL           generate main for MODEL
  -o, --output=DIR            write output to DIR (use - for stdout)
  -q, --queue-size=SIZE       use queue size SIZE [~a]
      --queue-size-defer=SIZE
                              use queue size=SIZE [~a] for defer
      --queue-size-external=SIZE
                              use queue size=SIZE [~a] for external
  -s, --shell=MODEL           generate thread safe system shell for MODEL
  -t, --touch-empty-files     generate empty files for consistency
  -U, --no-unreachable        do not generate unreachable code tags

Languages: ~a
" %default-language
(%queue-size) (%queue-size-defer) (%queue-size-external)
(string-join %languages ", "))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define* (file->code file-name
                     #:key
                     backtrace?
                     calling-context
                     dir
                     empty-files?
                     model
                     language
                     locations?
                     (options '())
                     shell
                     verbose?)

  (define (shell-error ast model-name)
    (let* ((model (ast:lookup ast model-name))
           (message
            (cond
             ((not model)
              (format #f "No such model for --shell: `~a'." model-name))
             ((not (is-a? model <system>))
              (format #f "Option `--shell' cannot be used with ~a: `~a'."
                      (ast-name model)
                      (ast:dotted-name model)))
             ((ast:imported? model)
              (format #f "Option `--shell' cannot be used with imported `~a'."
                      (ast:dotted-name model)))
             (else
              #f))))
      (and message
           (make <error> #:ast (or model ast) #:message message))))

  ;; Parse --model=MODEL cuts MODEL from AST; avoid that
  (let* ((ast (parse options file-name))
         (errors (filter-map (cute shell-error ast <>) shell)))

    (parse:call-with-handle-exceptions
     (lambda _
       (when (pair? errors)
         (apply throw 'well-formedness-error errors))
       (code ast
             #:calling-context calling-context
             #:dir dir
             #:empty-files? empty-files?
             #:model model
             #:language language
             #:locations? locations?
             #:shell shell
             #:verbose? verbose?))
     #:backtrace? backtrace?
     #:file-name file-name)))


;;;
;;; Entry point.
;;;
(define (main args)
  (let* ((options (parse-opts args))
         (files (option-ref options '() '()))
         (files (append-map file-name->dzn-files files))
         (multiple? (> (length files) 1))
         (debug? (dzn:command-line:get 'debug #f))
         (dir (option-ref options 'output #f))
         (calling-context (option-ref options 'calling-context #f))
         (empty-files? (command-line:get 'touch-empty-files #f))
         (language (option-ref options 'language %default-language))
         (locations? (option-ref options 'locations #f))
         (model (option-ref options 'model #f))
         (queue-size (command-line:get-number 'queue-size (%queue-size)))
         (queue-size-defer (command-line:get-number 'queue-size
                                                    (%queue-size-defer)))
         (queue-size-external (command-line:get-number 'queue-size-external
                                                       (%queue-size-external)))
         (no-constraint? (command-line:get 'no-constraint))
         (no-unreachable? (command-line:get 'no-unreachable))
         (shell (multi-opt options 'shell)))

    (when (and (> (length files) 1)
               (not (equal? language "c++")))
      (format (current-error-port) "dzn code: multiple files only supported for c++\n")
      (exit EXIT_OTHER_FAILURE))

    (parameterize ((%calling-context calling-context)
                   (%no-constraint? no-constraint?)
                   (%language language)
                   (%locations? locations?)
                   (%no-unreachable? no-unreachable?)
                   (%queue-size queue-size)
                   (%queue-size-defer queue-size-defer)
                   (%queue-size-external queue-size-external)
                   (%shell shell))
      (parameterize ((%context (%context)))
        (for-each (cut file->code <>
                       #:backtrace? debug?
                       #:calling-context calling-context
                       #:dir dir
                       #:empty-files? empty-files?
                       #:model model
                       #:language language
                       #:locations? locations?
                       #:shell shell
                       #:verbose? multiple?)
                  files)))))
