/* marray/marray_long.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_long_H__
#define __MARRAY_long_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  long * data;
} marray_long;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_long *
marray_long_alloc(const unsigned int rank, const size_t * dimension);

marray_long *
marray_long_calloc(const unsigned int rank, const size_t * dimension);

marray_long *
marray_long_copy(marray_long * t);

void marray_long_free(marray_long * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_long * marray_long_2matrix(marray_long * t);
gsl_vector_long * marray_long_2vector(marray_long * t);


/* Operations */

long marray_long_get(const marray_long * t, const size_t * indices);
void marray_long_set(marray_long * t, const size_t * indices, const long x);


long * marray_long_ptr(marray_long * t, const size_t * indices);
const long * marray_long_const_ptr(const marray_long * t, const size_t * indices);

void marray_long_set_zero(marray_long * t);
void marray_long_set_all(marray_long * t, long x);

int marray_long_fread(FILE * stream, marray_long * t);
int marray_long_fwrite(FILE * stream, const marray_long * t);
int marray_long_fscanf(FILE * stream, marray_long * t);
int marray_long_fprintf(FILE * stream, const marray_long * t, const char * format);

int marray_long_memcpy(marray_long * dest, const marray_long * src);
int marray_long_swap(marray_long * t1, marray_long * t2);

long marray_long_max(const marray_long * t);
long marray_long_min(const marray_long * t);
void marray_long_minmax(const marray_long * t, long * min_out, long * max_out);

void marray_long_max_index(const marray_long * t, size_t * indices);
void marray_long_min_index(const marray_long * t, size_t * indices);
void marray_long_minmax_index(const marray_long * t, size_t * imin, size_t * imax);

int marray_long_isnull(const marray_long * t);

int marray_long_add(marray_long * a, const marray_long * b);
int marray_long_sub(marray_long * a, const marray_long * b);
int marray_long_mul_elements(marray_long * a, const marray_long * b);
int marray_long_div_elements(marray_long * a, const marray_long * b);
int marray_long_scale(marray_long * a, const double x);
int marray_long_add_constant(marray_long * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_long_position(const size_t * indices, const marray_long * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
long
marray_long_get(const marray_long * t, const size_t * indices)
{
  size_t position;

  position = marray_long_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_long_set(marray_long * t, const size_t * indices, const long x)
{
  size_t position;

  position = marray_long_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
long *
marray_long_ptr(marray_long * t, const size_t * indices)
{
  size_t position;

  position = marray_long_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (long *) (t->data + position);
}


extern inline
const long *
marray_long_const_ptr(const marray_long * t, const size_t * indices)
{
  size_t position;

  position = marray_long_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const long *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_long_H__ */
