/* marray/marray_long_double.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_long_double_H__
#define __MARRAY_long_double_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  long double * data;
} marray_long_double;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_long_double *
marray_long_double_alloc(const unsigned int rank, const size_t * dimension);

marray_long_double *
marray_long_double_calloc(const unsigned int rank, const size_t * dimension);

marray_long_double *
marray_long_double_copy(marray_long_double * t);

void marray_long_double_free(marray_long_double * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_long_double * marray_long_double_2matrix(marray_long_double * t);
gsl_vector_long_double * marray_long_double_2vector(marray_long_double * t);


/* Operations */

long double marray_long_double_get(const marray_long_double * t, const size_t * indices);
void marray_long_double_set(marray_long_double * t, const size_t * indices, const long double x);


long double * marray_long_double_ptr(marray_long_double * t, const size_t * indices);
const long double * marray_long_double_const_ptr(const marray_long_double * t, const size_t * indices);

void marray_long_double_set_zero(marray_long_double * t);
void marray_long_double_set_all(marray_long_double * t, long double x);

int marray_long_double_fread(FILE * stream, marray_long_double * t);
int marray_long_double_fwrite(FILE * stream, const marray_long_double * t);
int marray_long_double_fscanf(FILE * stream, marray_long_double * t);
int marray_long_double_fprintf(FILE * stream, const marray_long_double * t, const char * format);

int marray_long_double_memcpy(marray_long_double * dest, const marray_long_double * src);
int marray_long_double_swap(marray_long_double * t1, marray_long_double * t2);

long double marray_long_double_max(const marray_long_double * t);
long double marray_long_double_min(const marray_long_double * t);
void marray_long_double_minmax(const marray_long_double * t, long double * min_out, long double * max_out);

void marray_long_double_max_index(const marray_long_double * t, size_t * indices);
void marray_long_double_min_index(const marray_long_double * t, size_t * indices);
void marray_long_double_minmax_index(const marray_long_double * t, size_t * imin, size_t * imax);

int marray_long_double_isnull(const marray_long_double * t);

int marray_long_double_add(marray_long_double * a, const marray_long_double * b);
int marray_long_double_sub(marray_long_double * a, const marray_long_double * b);
int marray_long_double_mul_elements(marray_long_double * a, const marray_long_double * b);
int marray_long_double_div_elements(marray_long_double * a, const marray_long_double * b);
int marray_long_double_scale(marray_long_double * a, const double x);
int marray_long_double_add_constant(marray_long_double * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_long_double_position(const size_t * indices, const marray_long_double * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
long double
marray_long_double_get(const marray_long_double * t, const size_t * indices)
{
  size_t position;

  position = marray_long_double_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_long_double_set(marray_long_double * t, const size_t * indices, const long double x)
{
  size_t position;

  position = marray_long_double_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
long double *
marray_long_double_ptr(marray_long_double * t, const size_t * indices)
{
  size_t position;

  position = marray_long_double_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (long double *) (t->data + position);
}


extern inline
const long double *
marray_long_double_const_ptr(const marray_long_double * t, const size_t * indices)
{
  size_t position;

  position = marray_long_double_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const long double *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_long_double_H__ */
