/* marray/marray_short.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_short_H__
#define __MARRAY_short_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  short * data;
} marray_short;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_short *
marray_short_alloc(const unsigned int rank, const size_t * dimension);

marray_short *
marray_short_calloc(const unsigned int rank, const size_t * dimension);

marray_short *
marray_short_copy(marray_short * t);

void marray_short_free(marray_short * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_short * marray_short_2matrix(marray_short * t);
gsl_vector_short * marray_short_2vector(marray_short * t);


/* Operations */

short marray_short_get(const marray_short * t, const size_t * indices);
void marray_short_set(marray_short * t, const size_t * indices, const short x);


short * marray_short_ptr(marray_short * t, const size_t * indices);
const short * marray_short_const_ptr(const marray_short * t, const size_t * indices);

void marray_short_set_zero(marray_short * t);
void marray_short_set_all(marray_short * t, short x);

int marray_short_fread(FILE * stream, marray_short * t);
int marray_short_fwrite(FILE * stream, const marray_short * t);
int marray_short_fscanf(FILE * stream, marray_short * t);
int marray_short_fprintf(FILE * stream, const marray_short * t, const char * format);

int marray_short_memcpy(marray_short * dest, const marray_short * src);
int marray_short_swap(marray_short * t1, marray_short * t2);

short marray_short_max(const marray_short * t);
short marray_short_min(const marray_short * t);
void marray_short_minmax(const marray_short * t, short * min_out, short * max_out);

void marray_short_max_index(const marray_short * t, size_t * indices);
void marray_short_min_index(const marray_short * t, size_t * indices);
void marray_short_minmax_index(const marray_short * t, size_t * imin, size_t * imax);

int marray_short_isnull(const marray_short * t);

int marray_short_add(marray_short * a, const marray_short * b);
int marray_short_sub(marray_short * a, const marray_short * b);
int marray_short_mul_elements(marray_short * a, const marray_short * b);
int marray_short_div_elements(marray_short * a, const marray_short * b);
int marray_short_scale(marray_short * a, const double x);
int marray_short_add_constant(marray_short * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_short_position(const size_t * indices, const marray_short * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
short
marray_short_get(const marray_short * t, const size_t * indices)
{
  size_t position;

  position = marray_short_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_short_set(marray_short * t, const size_t * indices, const short x)
{
  size_t position;

  position = marray_short_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
short *
marray_short_ptr(marray_short * t, const size_t * indices)
{
  size_t position;

  position = marray_short_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (short *) (t->data + position);
}


extern inline
const short *
marray_short_const_ptr(const marray_short * t, const size_t * indices)
{
  size_t position;

  position = marray_short_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const short *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_short_H__ */
