/* marray/marray_uint.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_uint_H__
#define __MARRAY_uint_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  unsigned int * data;
} marray_uint;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_uint *
marray_uint_alloc(const unsigned int rank, const size_t * dimension);

marray_uint *
marray_uint_calloc(const unsigned int rank, const size_t * dimension);

marray_uint *
marray_uint_copy(marray_uint * t);

void marray_uint_free(marray_uint * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_uint * marray_uint_2matrix(marray_uint * t);
gsl_vector_uint * marray_uint_2vector(marray_uint * t);


/* Operations */

unsigned int marray_uint_get(const marray_uint * t, const size_t * indices);
void marray_uint_set(marray_uint * t, const size_t * indices, const unsigned int x);


unsigned int * marray_uint_ptr(marray_uint * t, const size_t * indices);
const unsigned int * marray_uint_const_ptr(const marray_uint * t, const size_t * indices);

void marray_uint_set_zero(marray_uint * t);
void marray_uint_set_all(marray_uint * t, unsigned int x);

int marray_uint_fread(FILE * stream, marray_uint * t);
int marray_uint_fwrite(FILE * stream, const marray_uint * t);
int marray_uint_fscanf(FILE * stream, marray_uint * t);
int marray_uint_fprintf(FILE * stream, const marray_uint * t, const char * format);

int marray_uint_memcpy(marray_uint * dest, const marray_uint * src);
int marray_uint_swap(marray_uint * t1, marray_uint * t2);

unsigned int marray_uint_max(const marray_uint * t);
unsigned int marray_uint_min(const marray_uint * t);
void marray_uint_minmax(const marray_uint * t, unsigned int * min_out, unsigned int * max_out);

void marray_uint_max_index(const marray_uint * t, size_t * indices);
void marray_uint_min_index(const marray_uint * t, size_t * indices);
void marray_uint_minmax_index(const marray_uint * t, size_t * imin, size_t * imax);

int marray_uint_isnull(const marray_uint * t);

int marray_uint_add(marray_uint * a, const marray_uint * b);
int marray_uint_sub(marray_uint * a, const marray_uint * b);
int marray_uint_mul_elements(marray_uint * a, const marray_uint * b);
int marray_uint_div_elements(marray_uint * a, const marray_uint * b);
int marray_uint_scale(marray_uint * a, const double x);
int marray_uint_add_constant(marray_uint * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_uint_position(const size_t * indices, const marray_uint * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
unsigned int
marray_uint_get(const marray_uint * t, const size_t * indices)
{
  size_t position;

  position = marray_uint_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_uint_set(marray_uint * t, const size_t * indices, const unsigned int x)
{
  size_t position;

  position = marray_uint_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
unsigned int *
marray_uint_ptr(marray_uint * t, const size_t * indices)
{
  size_t position;

  position = marray_uint_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (unsigned int *) (t->data + position);
}


extern inline
const unsigned int *
marray_uint_const_ptr(const marray_uint * t, const size_t * indices)
{
  size_t position;

  position = marray_uint_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const unsigned int *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_uint_H__ */
