/* marray/marray_ulong.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_ulong_H__
#define __MARRAY_ulong_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  unsigned long * data;
} marray_ulong;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_ulong *
marray_ulong_alloc(const unsigned int rank, const size_t * dimension);

marray_ulong *
marray_ulong_calloc(const unsigned int rank, const size_t * dimension);

marray_ulong *
marray_ulong_copy(marray_ulong * t);

void marray_ulong_free(marray_ulong * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_ulong * marray_ulong_2matrix(marray_ulong * t);
gsl_vector_ulong * marray_ulong_2vector(marray_ulong * t);


/* Operations */

unsigned long marray_ulong_get(const marray_ulong * t, const size_t * indices);
void marray_ulong_set(marray_ulong * t, const size_t * indices, const unsigned long x);


unsigned long * marray_ulong_ptr(marray_ulong * t, const size_t * indices);
const unsigned long * marray_ulong_const_ptr(const marray_ulong * t, const size_t * indices);

void marray_ulong_set_zero(marray_ulong * t);
void marray_ulong_set_all(marray_ulong * t, unsigned long x);

int marray_ulong_fread(FILE * stream, marray_ulong * t);
int marray_ulong_fwrite(FILE * stream, const marray_ulong * t);
int marray_ulong_fscanf(FILE * stream, marray_ulong * t);
int marray_ulong_fprintf(FILE * stream, const marray_ulong * t, const char * format);

int marray_ulong_memcpy(marray_ulong * dest, const marray_ulong * src);
int marray_ulong_swap(marray_ulong * t1, marray_ulong * t2);

unsigned long marray_ulong_max(const marray_ulong * t);
unsigned long marray_ulong_min(const marray_ulong * t);
void marray_ulong_minmax(const marray_ulong * t, unsigned long * min_out, unsigned long * max_out);

void marray_ulong_max_index(const marray_ulong * t, size_t * indices);
void marray_ulong_min_index(const marray_ulong * t, size_t * indices);
void marray_ulong_minmax_index(const marray_ulong * t, size_t * imin, size_t * imax);

int marray_ulong_isnull(const marray_ulong * t);

int marray_ulong_add(marray_ulong * a, const marray_ulong * b);
int marray_ulong_sub(marray_ulong * a, const marray_ulong * b);
int marray_ulong_mul_elements(marray_ulong * a, const marray_ulong * b);
int marray_ulong_div_elements(marray_ulong * a, const marray_ulong * b);
int marray_ulong_scale(marray_ulong * a, const double x);
int marray_ulong_add_constant(marray_ulong * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_ulong_position(const size_t * indices, const marray_ulong * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
unsigned long
marray_ulong_get(const marray_ulong * t, const size_t * indices)
{
  size_t position;

  position = marray_ulong_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_ulong_set(marray_ulong * t, const size_t * indices, const unsigned long x)
{
  size_t position;

  position = marray_ulong_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
unsigned long *
marray_ulong_ptr(marray_ulong * t, const size_t * indices)
{
  size_t position;

  position = marray_ulong_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (unsigned long *) (t->data + position);
}


extern inline
const unsigned long *
marray_ulong_const_ptr(const marray_ulong * t, const size_t * indices)
{
  size_t position;

  position = marray_ulong_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const unsigned long *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_ulong_H__ */
