/* marray/marray_ushort.h
 *
 * Copyright (C) 2003, 2004, 2007 Jordi Burguet-Castell
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   Free Software Foundation, Inc.
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */

/*
 * This file contains the basic declarations for a marray.
 */
#ifndef __MARRAY_ushort_H__
#define __MARRAY_ushort_H__

#include <stdlib.h>


#include <gsl/gsl_types.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_check_range.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>

#include "marray_utilities.h"

#undef __BEGIN_DECLS
#undef __END_DECLS
#ifdef __cplusplus
# define __BEGIN_DECLS extern "C" {
# define __END_DECLS }
#else
# define __BEGIN_DECLS /* empty */
# define __END_DECLS /* empty */
#endif

__BEGIN_DECLS


/*
 * A marray is a struct with the rank of the marray (number of
 * indices it has), an array with the number of elements available for
 * each index, and an array to store the n1*n2*...*nm values.
 *
 * For the moment, there is no tda, as opossed to matrices, because it
 * would complicate quite a bit the algorithms and probably it is not
 * worth it.
 */
typedef struct
{
  unsigned int rank;
  size_t * dimension;
  size_t size;
  unsigned short * data;
} marray_ushort;


/*
 * There is not such a thing as "marray views", in contrast with the
 * case for gsl_matrix.
 */

/* Allocation */

marray_ushort *
marray_ushort_alloc(const unsigned int rank, const size_t * dimension);

marray_ushort *
marray_ushort_calloc(const unsigned int rank, const size_t * dimension);

marray_ushort *
marray_ushort_copy(marray_ushort * t);

void marray_ushort_free(marray_ushort * t);


/* Views */

/*
 * There are no views.
 */


/* Conversions */

gsl_matrix_ushort * marray_ushort_2matrix(marray_ushort * t);
gsl_vector_ushort * marray_ushort_2vector(marray_ushort * t);


/* Operations */

unsigned short marray_ushort_get(const marray_ushort * t, const size_t * indices);
void marray_ushort_set(marray_ushort * t, const size_t * indices, const unsigned short x);


unsigned short * marray_ushort_ptr(marray_ushort * t, const size_t * indices);
const unsigned short * marray_ushort_const_ptr(const marray_ushort * t, const size_t * indices);

void marray_ushort_set_zero(marray_ushort * t);
void marray_ushort_set_all(marray_ushort * t, unsigned short x);

int marray_ushort_fread(FILE * stream, marray_ushort * t);
int marray_ushort_fwrite(FILE * stream, const marray_ushort * t);
int marray_ushort_fscanf(FILE * stream, marray_ushort * t);
int marray_ushort_fprintf(FILE * stream, const marray_ushort * t, const char * format);

int marray_ushort_memcpy(marray_ushort * dest, const marray_ushort * src);
int marray_ushort_swap(marray_ushort * t1, marray_ushort * t2);

unsigned short marray_ushort_max(const marray_ushort * t);
unsigned short marray_ushort_min(const marray_ushort * t);
void marray_ushort_minmax(const marray_ushort * t, unsigned short * min_out, unsigned short * max_out);

void marray_ushort_max_index(const marray_ushort * t, size_t * indices);
void marray_ushort_min_index(const marray_ushort * t, size_t * indices);
void marray_ushort_minmax_index(const marray_ushort * t, size_t * imin, size_t * imax);

int marray_ushort_isnull(const marray_ushort * t);

int marray_ushort_add(marray_ushort * a, const marray_ushort * b);
int marray_ushort_sub(marray_ushort * a, const marray_ushort * b);
int marray_ushort_mul_elements(marray_ushort * a, const marray_ushort * b);
int marray_ushort_div_elements(marray_ushort * a, const marray_ushort * b);
int marray_ushort_scale(marray_ushort * a, const double x);
int marray_ushort_add_constant(marray_ushort * a, const double x);


/* inline functions if you are using GCC */

#ifdef HAVE_INLINE
extern inline
size_t
marray_ushort_position(const size_t * indices, const marray_ushort * t)
{
  size_t shift, position;
  unsigned int i;

  shift = t->size;
  position = 0;
  for (i = 0; i < t->rank; i++)
    {
#if GSL_RANGE_CHECK
      if (indices[i] >= t->dimension[i])
        return t->size;
#endif
      shift /= t->dimension[i];
      position += shift * indices[i];
    }

  return position;
}


extern inline
unsigned short
marray_ushort_get(const marray_ushort * t, const size_t * indices)
{
  size_t position;

  position = marray_ushort_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VAL("index out of range", GSL_EINVAL, 0);
#endif

  return t->data[position];
}


extern inline
void
marray_ushort_set(marray_ushort * t, const size_t * indices, const unsigned short x)
{
  size_t position;

  position = marray_ushort_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_VOID("index out of range", GSL_EINVAL);
#endif

  t->data[position] = x;
}


extern inline
unsigned short *
marray_ushort_ptr(marray_ushort * t, const size_t * indices)
{
  size_t position;

  position = marray_ushort_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (unsigned short *) (t->data + position);
}


extern inline
const unsigned short *
marray_ushort_const_ptr(const marray_ushort * t, const size_t * indices)
{
  size_t position;

  position = marray_ushort_position(indices, t);
#if GSL_RANGE_CHECK
  if (position >= t->size)
    GSL_ERROR_NULL("index out of range", GSL_EINVAL);
#endif

  return (const unsigned short *) (t->data + position);
}

#endif

__END_DECLS

#endif /* __MARRAY_ushort_H__ */
