{
  Copyright 2014-2024 Michalis Kamburelis.

  This file is part of "Castle Game Engine".

  "Castle Game Engine" is free software; see the file COPYING.txt,
  included in this distribution, for details about the copyright.

  "Castle Game Engine" is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

  ----------------------------------------------------------------------------
}

{ Project information (from CastleEngineManifest.xml) and operations. }
unit ToolProject;

{$I castleconf.inc}

interface

uses SysUtils, Classes, Generics.Collections,
  CastleFindFiles, CastleStringUtils, CastleUtils, CastleInternalTools,
  ToolArchitectures, ToolCompile, ToolUtils, ToolServices, ToolAssocDocTypes,
  ToolPackage, ToolManifest, ToolProcess, ToolPackageFormat;

type
  ECannotGuessManifest = class(Exception);

  TCastleProject = class
  private
    ManifestFile: string;
    Manifest: TCastleManifest;
    DeletedFiles: Cardinal; //< only for DeleteFoundFile
    // Helpers only for ExtractTemplateFoundFile.
    // @groupBegin
    ExtractTemplateDestinationPath, ExtractTemplateDir: string;
    ExtractTemplateOverrideExisting: Boolean;
    // @groupEnd
    { Use to define macros containing the Android architecture names.
      Must be set by all commands that may use our macro system. }
    AndroidCPUS: TCPUS;
    IOSExportMethod: String; // set by DoPackage based on PackageFormat, otherwise ''
    FLaunchImageStoryboardInitialized: Boolean;
    FLaunchImageStoryboardWidth, FLaunchImageStoryboardHeight: Integer;
    FGuidFromName: Boolean;

    procedure DeleteFoundFile(const FileInfo: TFileInfo; var StopSearch: boolean);
    function PackageName(const OS: TOS; const CPU: TCPU; const PackageFormat: TPackageFormatNoDefault;
      const PackageNameIncludeVersion: Boolean): string;
    function SourcePackageName(const PackageNameIncludeVersion: Boolean;
      const PackageFormatFinal: TPackageFormatNoDefault): string;
    procedure ExtractTemplateFoundFile(const FileInfo: TFileInfo; var StopSearch: boolean);

    { Detect welcome file. Returns '' or a relative filename inside the project.

      Right now it is only used to display in Delphi IDE
      (specified in DPROJ, makes project look nice when it is opened,
      esp. since CGE examples and templates recommend README.md and Delphi
      handles Markdown nicely).
      And right now it is only detected automatically.

      In the future:
      - We may use it for more things, e.g. display in CGE editor.
      - We may allow to set it explicitly in CastleEngineManifest.xml. }
    function WelcomePage: String;

    { Convert Name to a valid Pascal identifier. }
    function NamePascal: string;

    { Extract a single file using the template system.
      SourceFileName and DestinationFileName should be absolute filenames
      of source and destination files.
      DestinationRelativeFileName should be a relative version of DestinationFileName,
      relative to the template root.

      This is used internally by ExtractTemplateFoundFile, which is in turn used
      by ExtractTemplate that can extract a whole template directory.

      It can also be used directly to expand a single file. }
    procedure ExtractTemplateFile(
      const SourceFileName, DestinationFileName, DestinationRelativeFileName: string;
      const OverrideExisting: boolean);

    { Generate a Pascal file from template. }
    procedure GeneratedSourceFile(
      const TemplateRelativeUrl, TargetRelativePath, ErrorMessageMissingGameUnits: string;
      const CreateIfNecessary: boolean;
      out RelativeResult, AbsoluteResult: string);
    procedure GeneratedSourceFile(
      const TemplateRelativeUrl, TargetRelativePath, ErrorMessageMissingGameUnits: string;
      const CreateIfNecessary: boolean);

    function AndroidSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
    function IOSSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
    function NXSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
    function StandaloneSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;

    { The standalone program filename that is generated by "castle-engine generate-program".

      Unlike StandaloneSourceFile, this is explicit,
      i.e. it never points to internal name in "castle-engine-output".
      Also this method never creates the file, it merely calculates the filename.

      The returned filename is relative. It is often without any path
      (like "my_program_standalone.dpr") but it may contain a relative path
      (like "code/my_program_standalone.dpr") if such path was specified
      in CastleEngineManifest.xml. }
    function ExplicitStandaloneFile(const Ext: String): String;

    { Check the PackageFormat (looking at what is allowed for this Target/OS/CPU),
      finalize the pfDefault to something more concrete. }
    function PackageFormatFinalize(
      const Target: TTarget; const OS: TOS; const CPU: TCPU;
      const PackageFormat: TPackageFormat): TPackageFormatNoDefault;

    procedure AddMacrosAndroid(const Macros: TStringStringMap);
    procedure AddMacrosIOS(const Macros: TStringStringMap);
  public
    constructor Create;
    constructor Create(const APath: string);
    constructor CreateNewProject(const ParentDir,
      ProjectName, TemplateName, ProjectCaption, MainView: String);
    destructor Destroy; override;

    { Commands on a project, used by the main program code. }
    { }

    procedure DoCreateManifest;
    procedure DoCompile(const OverrideCompiler: TCompiler;
      const Target: TTarget; const OS: TOS; const CPU: TCPU;
      const Mode: TCompilationMode; const CompilerExtraOptions: TStrings = nil;
      const AllowCache: Boolean = true);
    procedure DoPackage(const Target: TTarget;
      const OS: TOS; const CPU: TCPU; const Mode: TCompilationMode;
      const PackageFormat: TPackageFormat;
      const PackageNameIncludeVersion, UpdateOnlyCode: Boolean);
    procedure DoInstall(const Target: TTarget; const OS: TOS; const CPU: TCPU;
      const Mode: TCompilationMode;
      const PackageFormat: TPackageFormat;
      const PackageNameIncludeVersion: Boolean);
    procedure DoRun(const Target: TTarget; const OS: TOS; const CPU: TCPU;
      const Params: TCastleStringList);
    procedure DoPackageSource(
      const PackageFormat: TPackageFormat;
      const PackageNameIncludeVersion: Boolean);
    procedure DoClean;
    procedure DoAutoGenerateTextures;
    procedure DoAutoGenerateClean(const CleanAll: Boolean);
    procedure DoGenerateProgram(const GuidFromName: Boolean);
    procedure DoEditor;
    procedure DoEditorRebuildIfNeeded;
    procedure DoEditorRun(const WaitForProcessId: TProcessId);
    procedure DoOutput(const OutputKey: String);

    { Information about the project, derived from CastleEngineManifest.xml. }
    { }

    function Version: TProjectVersion;
    function ManifestCompiler: TCompiler;
    function QualifiedName: string;
    function Dependencies: TDependencies;
    function Name: string;
    { Project path. Always ends with path delimiter, like a slash or backslash. }
    function Path: string;
    function DataExists: Boolean;
    { Project data path. Always ends with path delimiter, like a slash or backslash.
      Should be ignored if not @link(DataExists). }
    function DataPath: string;
    function Caption: string;
    function Author: string;
    function ExecutableName: string;
    function FullscreenImmersive: boolean;
    function ScreenOrientation: TScreenOrientation;
    function AndroidCompileSdkVersion: Cardinal;
    function AndroidMinSdkVersion: Cardinal;
    function AndroidTargetSdkVersion: Cardinal;
    function Icons: TImageFileNames;
    function LaunchImages: TImageFileNames;
    function AndroidServices: TServiceList;
    function IOSServices: TServiceList;
    function AssociateDocumentTypes: TAssociatedDocTypeList;
    function LocalizedAppNames: TLocalizedAppNameList;
    function LaunchImageStoryboard: TLaunchImageStoryboard;

    function ReplaceMacros(const Source: string): string;

    { Recursively copy a directory from TemplatePath (this is relative
      to the build tool data) to the DestinationPath (this should be an absolute
      existing directory name).

      Each file is processed by the ReplaceMacros method.

      OverrideExisting says what happens when the destination file already exists.

      - OverrideExisting = @false (default) means that the
        destination file will be left unchanged
        (to preserve possible user customization),
        or source will be merged into destination
        (in case of special filenames;
        this allows to e.g. merge AndroidManifest.xml).

      - OverrideExisting = @true means that the destination file
        will simply be overridden, without any warning, without
        any merging.
    }
    procedure ExtractTemplate(const TemplatePath, DestinationPath: string;
      const OverrideExisting: Boolean = false);

    { Output Android library resulting from compilation.
      Relative to @link(Path) if AbsolutePath = @false,
      otherwise a complete absolute path.

      CPU should be one of CPUs supported on Android platform (arm, aarch64)
      or cpuNone to get the library name without the CPU suffix. }
    function AndroidLibraryFile(const CPU: TCPU; const AbsolutePath: boolean = true): string;

    { Get platform-independent files that should be included in a package,
      remove files that should be excluded.

      If OnlyData, then only takes stuff inside DataPath,
      and Files will contain URLs relative to DataPath.
      Otherwise, takes all files to be packaged in a project,
      and Files will contain URLs relative to @link(Path).

      The copy will only contain files useful on given TargetPlatform.
      Right now this means we will exclude auto-generated textures not suitable
      for TargetPlatform.

      The associated objects of the returned TCastleStringList,
      if non nil, are TIncludePath instances that caused inclusion
      of given file in the package. }
    function PackageFiles(const OnlyData: boolean;
      const TargetPlatform: TCastlePlatform): TCastleStringList;

    { Output iOS library resulting from compilation.
      Relative to @link(Path) if AbsolutePath = @false,
      otherwise a complete absolute path. }
    function IOSLibraryFile(const AbsolutePath: boolean = true): string;

    { Output Nintendo Switch library resulting from compilation.
      Relative to @link(Path) if AbsolutePath = @false,
      otherwise a complete absolute path. }
    function NXLibraryFile(const AbsolutePath: boolean = true): string;

    { Where should we place our output files, calculated looking at OutputPathBase
      and project path. Always an absolute filename ending with path delimiter. }
    function OutputPath: string;

    { Copy project data subdirectory to given path.
      OutputDataPath may but doesn't have to end with PathDelim.

      The path will be created if necessary (even if there are no files,
      this is useful at least for XCode as it references the resulting directory,
      so it must exist).

      The copy will only contain files useful on given TargetPlatform.
      Right now this means we will exclude auto-generated textures not suitable
      for TargetPlatform.

      We also generate the auto_generated/CastleDataInformation.xml inside.
      (Actually, this means the resulting directory is never empty now.) }
    procedure CopyData(OutputDataPath: string; const TargetPlatform: TCastlePlatform);

    { Is this filename created by some DoPackage or DoPackageSource command.
      FileName must be relative to project root directory. }
    function PackageOutput(const FileName: String): Boolean;

    function IOSHasLaunchImageStoryboard: Boolean;
  end;

implementation

uses {$ifdef UNIX} BaseUnix, {$endif}
  StrUtils, DOM, Process,
  CastleUriUtils, CastleXmlUtils, CastleLog, CastleFilesUtils, CastleImages,
  CastleTimeUtils,
  ToolResources, ToolAndroid, ToolMacOS,
  ToolTextureGeneration, ToolIOS, ToolAndroidMerging, ToolNintendoSwitch,
  ToolCommonUtils, ToolMacros, ToolCompilerInfo, ToolPackageCollectFiles;

const
  SErrDataDir = 'Make sure you have installed the data files of the Castle Game Engine build tool. Usually it is easiest to set the $CASTLE_ENGINE_PATH environment variable to the location of castle_game_engine/ or castle-engine/ directory, the build tool will then find its data correctly.'
    {$ifdef UNIX}
    + ' Or place the data in system-wide location /usr/share/castle-engine/ or /usr/local/share/castle-engine/.'
    {$endif};

{ Insert 'lib' prefix at the beginning of file name. }
function InsertLibPrefix(const S: string): string;
begin
  Result := ExtractFilePath(S) + 'lib' + ExtractFileName(S);
end;

{ Compiled library name (.so, .dll etc.) from given source code filename. }
function CompiledLibraryFile(const S: string; const OS: TOS): string;
begin
  Result := ChangeFileExt(S, LibraryExtensionOS(OS));
  if OS in AllUnixOSes then
    Result := InsertLibPrefix(Result);
end;

{ List filenames of external libraries used by the Dependencies, on given OS/CPU. }
procedure ExternalLibraries(const OS: TOS; const CPU: TCPU;
  const ProjectDependencies: TProjectDependencies; const List: TStrings;
  const CheckFilesExistence: Boolean = true);

  { Path to the external library in data/external_libraries/ .
    Right now, these host various Windows-specific DLL files.
    If CheckFilesExistence then this checks existence of appropriate files along the way,
    and raises exception in case of trouble. }
  function ExternalLibraryPath(const OS: TOS; const CPU: TCPU; const LibraryName: string): string;
  var
    LibraryUrl: String;
  begin
    LibraryUrl := 'castle-data:/' + LibraryName;
    Result := UriToFilenameSafe(LibraryUrl);
    if CheckFilesExistence and (not RegularFileExists(Result)) then
      raise Exception.Create('Cannot find dependency library in "' + Result + '". ' + SErrDataDir);
  end;

  procedure AddExternalLibrary(const LibraryName: string);
  begin
    List.Add(ExternalLibraryPath(OS, CPU, LibraryName));
  end;

var
  Files: TStringList;
  F: String;
begin
  Files := TStringList.Create;
  try
    case OS of
      win32: ProjectDependencies.DeployFiles(dpWin32, Files);
      win64: ProjectDependencies.DeployFiles(dpWin64, Files);
      else ; { no need to do anything on other OSes }
    end;

    for F in Files do
      AddExternalLibrary(F);
  finally FreeAndNil(Files) end;
end;

{ TCastleProject ------------------------------------------------------------- }

constructor TCastleProject.Create;
var
  { look for CastleEngineManifest.xml in this dir, or parents }
  Dir, ParentDir: string;
begin
  Dir := GetCurrentDir;
  while not RegularFileExists(InclPathDelim(Dir) + ManifestName) do
  begin
    ParentDir := ExtractFileDir(ExclPathDelim(Dir));
    if (ParentDir = '') or (ParentDir = Dir) then
    begin
      { no parent directory, give up, assume auto-guessed values in current dir }
      Create(GetCurrentDir);
      Exit;
    end;
    {if Verbose then
      Writeln('Manifest not found, looking in parent directory: ', ParentDir);}
    Dir := ParentDir;
  end;
  Create(Dir);
end;

constructor TCastleProject.Create(const APath: string);

  procedure ReadManifest;

    function GuessStandaloneSource: string;
    var
      FileInfo: TFileInfo;
    begin
      Result := ExtractFileName(ExtractFileDir(ManifestFile)) + '.lpr';
      if RegularFileExists(Result) then
        Exit;

      Result := ExtractFileName(ExtractFileDir(ManifestFile)) + '.dpr';
      if RegularFileExists(Result) then
        Exit;

      if FindFirstFile(GetCurrentDir, '*.lpr', false, [], FileInfo) then
        Exit(FileInfo.Name);

      if FindFirstFile(GetCurrentDir, '*.dpr', false, [], FileInfo) then
        Exit(FileInfo.Name);

      raise ECannotGuessManifest.Create('Cannot find any *.lpr or *.dpr file in this directory, cannot guess which file to compile.' + NL +
        'Please create a CastleEngineManifest.xml to instruct Castle Game Engine build tool how to build your project.');
    end;

  var
    ManifestUrl: String;
  begin
    ManifestFile := InclPathDelim(APath) + ManifestName;
    ManifestUrl := FilenameToUriSafe(ManifestFile);

    if not RegularFileExists(ManifestFile) then
    begin
      Writeln('Manifest file not found: ' + ManifestFile);
      Writeln('Guessing project values. Use create-manifest command to write these guesses into new CastleEngineManifest.xml');
      Manifest := TCastleManifest.CreateGuess(APath, GuessStandaloneSource);
    end else
    begin
      WritelnVerbose('Manifest file found: ' + ManifestFile);
      Manifest := TCastleManifest.CreateFromUrl(APath, ManifestUrl);
    end;
  end;

  function DependenciesToStr(const S: TDependencies): string;
  var
    D: TDependency;
  begin
    Result := '';
    for D in S do
    begin
      if Result <> '' then Result += ', ';
      Result += DependencyToString(D);
    end;
    Result := '[' + Result + ']';
  end;

begin
  inherited Create;

  ReadManifest;

  if Verbose then
    Writeln('Project "' + Name + '" dependencies: ' + DependenciesToStr(Dependencies));
end;

constructor TCastleProject.CreateNewProject(const ParentDir,
  ProjectName, TemplateName, ProjectCaption, MainView: String);
var
  Options: TProjectCreationOptions;
  ProjectDirUrl, ProjectDir: String;
begin
  Options.ParentDir := ParentDir;
  Options.TemplateName := TemplateName;
  Options.ProjectName := ProjectName;
  Options.ProjectCaption := ProjectCaption;
  Options.MainView := MainView;
  ProjectCreateFromTemplate(CastleEnginePath, Options, ProjectDirUrl);

  ProjectDir := UriToFilenameSafe(ProjectDirUrl);
  Writeln('Created new project in ' + ProjectDir + ' based on template ' + TemplateName);

  Create(ProjectDir);

  DoGenerateProgram(false);
end;

destructor TCastleProject.Destroy;
begin
  FreeAndNil(Manifest);
  inherited;
end;

procedure TCastleProject.DoCreateManifest;
var
  Contents: string;
begin
  if RegularFileExists(ManifestFile) then
    raise Exception.CreateFmt('Manifest file "%s" already exists, refusing to overwrite it',
      [ManifestFile]);
  Contents := '<?xml version="1.0" encoding="utf-8"?>' +NL+
    '<project name="' + Name + '" standalone_source="' + Manifest.StandaloneSource + '">' +NL+
    '</project>' + NL;
  StringToFile(ManifestFile, Contents);
  Writeln('Created manifest ' + ManifestFile);
end;

procedure TCastleProject.DoCompile(const OverrideCompiler: TCompiler; const Target: TTarget;
  const OS: TOS; const CPU: TCPU; const Mode: TCompilationMode;
  const CompilerExtraOptions: TStrings;
  const AllowCache: Boolean);

  { Copy external libraries to LibrariesOutputPath.
    LibrariesOutputPath must be empty (current dir) or ending with path delimiter. }
  procedure AddExternalLibraries(const LibrariesOutputPath: String);
  var
    List: TCastleStringList;
    OutputFile, FileName: String;
  begin
    List := TCastleStringList.Create;
    try
      ExternalLibraries(OS, CPU, Manifest.ProjectDependencies, List);
      for FileName in List do
      begin
        OutputFile := LibrariesOutputPath + ExtractFileName(FileName);
        WritelnVerbose('Copying library to ' + OutputFile);
        CheckCopyFile(FileName, OutputFile);
      end;
    finally FreeAndNil(List) end;
  end;

var
  SourceExe, DestExe, MainSource: string;
  CompilerOptions: TCompilerOptions;
  FinalCompiler: TCompiler;
begin
  Writeln(Format('Compiling project "%s" for %s in mode "%s".',
    [Name, TargetCompleteToString(Target, OS, CPU), ModeToString(Mode)]));

  CompilerOptions := TCompilerOptions.Create;
  try
    CompilerOptions.AllowCache := AllowCache;
    CompilerOptions.OS := OS;
    CompilerOptions.CPU := CPU;
    CompilerOptions.Mode := Mode;
    CompilerOptions.DetectMemoryLeaks := Manifest.DetectMemoryLeaks;
    CompilerOptions.ExtraOptions.AddRange(Manifest.ExtraCompilerOptions);
    if CompilerExtraOptions <> nil then
      CompilerOptions.ExtraOptions.AddRange(CompilerExtraOptions);
    CompilerOptions.SearchPaths.AddRange(Manifest.SearchPaths);
    CompilerOptions.LibraryPaths.AddRange(Manifest.LibraryPaths);
    CompilerOptions.Defines.AddRange(Manifest.Defines);

    { TODO: "lazbuild" should be another Compiler option, called "lazbuild (calls FPC using options from LPI file)" }
    if Manifest.BuildUsingLazbuild then
    begin
      CompileLazbuild(Path, Manifest.LazarusProject, CompilerOptions);
      Exit;
    end;

    { use compiler specified in CastleEngineManifest.xml,
      overridden by Compiler specified on command-line (given as parameter). }
    FinalCompiler := OverrideCompiler;
    if FinalCompiler = coAutodetect then
      FinalCompiler := ManifestCompiler;

    case Target of
      targetAndroid:
        begin
          CompileAndroid(FinalCompiler, Self, Path, AndroidSourceFile(true, true), CompilerOptions);
        end;
      targetIOS:
        begin
          CompileIOS(FinalCompiler, Path, IOSSourceFile(true, true), CompilerOptions);
          LinkIOSLibrary(FinalCompiler, Path, IOSLibraryFile);
          Writeln('Compiled library for iOS in ', IOSLibraryFile(false));
        end;
      targetNintendoSwitch:
        begin
          CompileNintendoSwitchLibrary(Self, Path, NXSourceFile(true, true), CompilerOptions);
        end;
      targetCustom:
        begin
          case OS of
            Android:
              begin
                Compile(FinalCompiler, Path, AndroidSourceFile(true, true), CompilerOptions);
                { Our default compilation output doesn't contain CPU suffix,
                  but we need the CPU suffix to differentiate between Android/ARM and Android/Aarch64. }
                CheckRenameFile(AndroidLibraryFile(cpuNone), AndroidLibraryFile(CPU));
                Writeln('Compiled library for Android in ', AndroidLibraryFile(CPU, false));
              end;
            else
              begin
                MainSource := StandaloneSourceFile(false, true);
                if MainSource = '' then
                  raise Exception.Create('standalone_source property for project not defined, cannot compile standalone version');

                if MakeAutoGeneratedResources(Self, Path + ExtractFilePath(MainSource), OS, CPU) then
                  CompilerOptions.ExtraOptions.Add('-dCASTLE_AUTO_GENERATED_RESOURCES');

                Compile(FinalCompiler, Path, MainSource, CompilerOptions);

                SourceExe := ChangeFileExt(MainSource, ExeExtensionOS(OS));
                DestExe := ChangeFileExt(ExecutableName, ExeExtensionOS(OS));
                AddExternalLibraries(OutputPath);
                if not SameFileName(SourceExe, DestExe) then
                begin
                  { move exe to top-level (in case MainSource is in subdirectory
                    like code/) and eventually rename to follow ExecutableName }
                  Writeln('Moving ', SourceExe, ' to ', DestExe);
                  CheckRenameFile(
                    CombinePaths(Path, SourceExe),
                    CombinePaths(OutputPath, DestExe));
                end;
              end;
          end;
        end;
      {$ifndef COMPILER_CASE_ANALYSIS}
      else raise EInternalError.Create('Unhandled --target for DoCompile');
      {$endif}
    end;
  finally FreeAndNil(CompilerOptions) end;
end;

function TCastleProject.PackageFiles(const OnlyData: boolean;
  const TargetPlatform: TCastlePlatform): TCastleStringList;
var
  Collector: TBinaryPackageFiles;
begin
  Result := TCastleStringList.Create;
  try
    Collector := TBinaryPackageFiles.Create(Self);
    try
      Collector.IncludePaths := Manifest.IncludePaths;
      Collector.ExcludePaths := Manifest.ExcludePaths;
      Collector.OnlyData := OnlyData;
      Collector.TargetPlatform := TargetPlatform;
      Collector.Run;
      Result.Assign(Collector.CollectedFiles);
    finally FreeAndNil(Collector) end;
  except FreeAndNil(Result); raise; end;
end;

function TCastleProject.PackageFormatFinalize(
  const Target: TTarget; const OS: TOS; const CPU: TCPU;
  const PackageFormat: TPackageFormat): TPackageFormatNoDefault;
begin
  // calculate Result

  if PackageFormat = pfDefault then
  begin
    if Target = targetIOS then
      Result := pfIosXcodeProject
    else
    if (Target = targetAndroid) or (OS = Android) then
      Result := pfAndroidApk
    else
    if Target = targetNintendoSwitch then
      Result := pfNintendoSwitchProject
    else
    if OS in AllWindowsOSes then
      Result := pfZip
    else
    if (OS = Darwin) and Manifest.MacAppBundle then
      Result := pfMacAppBundleZip
    else
      Result := pfTarGz;
  end else
    Result := PackageFormat;

  // check Result

  if (Result in [
      pfIosXcodeProject,
      pfIosArchiveDevelopment,
      pfIosArchiveAdHoc,
      pfIosArchiveAppStore
    ]) and (Target <> targetIOS) then
  begin
    raise Exception.CreateFmt('Package format "%s" only makes sense when target is iOS', [
      PackageFormatToString(Result)
    ]);
  end;

  if Result in [pfDeb] then
  begin
    if (OS <> Linux) then
      raise Exception.Create('Cannot package DEB for OS = ' + OSToString(OS) + ' Expected: Linux.');
    {$ifndef Linux}
    raise Exception.Create('Currently DEB package can only be created under Linux OS, as it depends on several Linux-specific tools.');
    {$endif}
  end;

  if (Result in [
      pfAndroidApk,
      pfAndroidAppBundle
    ]) and (Target <> targetAndroid) and (OS <> Android) then
  begin
    raise Exception.CreateFmt('Package format "%s" only makes sense when target is Android.', [
      PackageFormatToString(Result)
    ]);
  end;

  if (Result in [
      pfNintendoSwitchProject
    ]) and (Target <> targetNintendoSwitch) then
  begin
    raise Exception.CreateFmt('Package format "%s" only makes sense when target is Nintendo Switch.', [
      PackageFormatToString(Result)
    ]);
  end;
end;

procedure TCastleProject.DoPackage(const Target: TTarget;
  const OS: TOS; const CPU: TCPU;
  const Mode: TCompilationMode; const PackageFormat: TPackageFormat;
  const PackageNameIncludeVersion, UpdateOnlyCode: Boolean);
var
  Pack: TPackageDirectory;

  function UnixPermissionsMatter: Boolean;
  begin
    Result := not (OS in AllWindowsOSes);
  end;

  procedure AddExecutable;
  var
    ExecutableNameExt, ExecutableNameFull: string;
  begin
    if OS in [linux, go32v2, win32, os2, freebsd, beos, netbsd,
              amiga, atari, solaris, qnx, netware, openbsd, wdosx,
              palmos, macosclassic, darwin, emx, watcom, morphos, netwlibc,
              win64, wince, gba,nds, embedded, symbian, haiku, {iphonesim,}
              aix, java, {android,} nativent, msdos, wii] then
    begin
      ExecutableNameExt := ExecutableName + ExeExtensionOS(OS);
      ExecutableNameFull := OutputPath + ExecutableNameExt;
      Pack.Add(ExecutableNameFull, ExecutableNameExt, UnixPermissionsMatter);
    end;
  end;

  procedure AddExternalLibraries;
  var
    List: TCastleStringList;
    FileName: String;
  begin
    List := TCastleStringList.Create;
    try
      ExternalLibraries(OS, CPU, Manifest.ProjectDependencies, List);
      for FileName in List do
        Pack.Add(FileName, ExtractFileName(FileName));
    finally FreeAndNil(List) end;
  end;

  { How the targets are detected (at build (right here) and inside the compiled application
    (in Platform implementation)) is a bit complicated.

    - nintendo-switch:

        At build: building for [[Nintendo Switch]] using CGE build tool with --target=nintendo-switch .

        Inside the application: if code was compiled with CASTLE_NINTENDO_SWITCH.

    - Android

        When OS is Android (currently possible values: Android/Arm, Android/Aarch64), and it is *not* detected as _Nintendo Switch_ (for internal reasons, right now _Nintendo Switch_ is also treated as Android by FPC).

        This logic is used both at build, and inside the application.

    - iOS: When
           (OS is iPhoneSim) or
           (FPC >= 3.2.2 and OS = iOS) or
           (FPC  < 3.2.2 and OS/architecture are Darwin/Arm or Darwin/Aarch64).

        This logic is used inside the application.
        At build, it is simpler, as our build tool just says "darwin is not iOS"
        (just like FPC >= 3.2.2 says) to support new macOS 11 (desktop on arm).

    - desktop: everything else.
  }
  function TargetPlatform: TCastlePlatform;
  begin
    case Target of
      targetIOS: Result := cpIOS;
      targetAndroid: Result := cpAndroid;
      targetNintendoSwitch: Result := cpNintendoSwitch;
      else // only targetCustom for now
      begin
        if OS = Android then
          Result := cpAndroid
        else
        if OS in [iphonesim, iOS] then
          Result := cpIOS
        else
          Result := cpDesktop;
      end;
    end;
  end;

  { Handle packaging using TPackageDirectory (to pfDirectory, pfZip, pfTarGz formats,
    TPackageDirectory.Make will check it). }
  procedure PackageDirectory(const PackageFormatFinal: TPackageFormatNoDefault);
  var
    I: Integer;
    PackageFileName: string;
    Files: TCastleStringList;
    ExecutablePermission: Boolean;
  begin
    Pack := TPackageDirectory.Create(Name);
    try
      { executable is added 1st, since it's the most likely file
        to not exist, so we'll fail earlier }
      AddExecutable;
      AddExternalLibraries;

      Files := PackageFiles(false, TargetPlatform);
      try
        for I := 0 to Files.Count - 1 do
        begin
          ExecutablePermission := (Files.Objects[I] <> nil) and (TIncludePath(Files.Objects[I]).ExecutablePermission);
          Pack.Add(Path + Files[I], Files[I], ExecutablePermission);
        end;
      finally FreeAndNil(Files) end;

      Pack.AddDataInformation(TCastleManifest.DataName);

      PackageFileName := PackageName(OS, CPU, PackageFormatFinal, PackageNameIncludeVersion);

      Pack.Cpu := Cpu;
      Pack.Manifest := Manifest;
      Pack.Make(OutputPath, PackageFileName, PackageFormatFinal);
    finally FreeAndNil(Pack) end;
  end;

var
  PackageFormatFinal: TPackageFormatNoDefault;
  WantsIOSArchive: Boolean;
  IOSArchiveType: TIosArchiveType;
begin
  PackageFormatFinal := PackageFormatFinalize(Target, OS, CPU, PackageFormat);

  Writeln(Format('Packaging project "%s" for %s (platform: %s).' + NL + 'Format: %s.', [
    Name,
    TargetCompleteToString(Target, OS, CPU),
    PlatformToStr(TargetPlatform),
    PackageFormatToString(PackageFormatFinal)
  ]));

  case PackageFormatFinal of
    pfIosXcodeProject, pfIosArchiveDevelopment, pfIosArchiveAdHoc, pfIosArchiveAppStore:
      begin
        // set IOSExportMethod early, as it determines IOS_EXPORT_METHOD macro
        WantsIOSArchive := PackageFormatWantsIOSArchive(PackageFormatFinal, IOSArchiveType, IOSExportMethod);
        PackageIOS(Self, UpdateOnlyCode);
        if WantsIOSArchive then
          ArchiveIOS(Self, IOSArchiveType);
      end;
    pfAndroidApk, pfAndroidAppBundle:
      begin
        if Target = targetAndroid then
          AndroidCPUS := DetectAndroidCPUS
        else
          AndroidCPUS := [CPU];
        PackageAndroid(Self, OS, AndroidCPUS, Mode, PackageFormatFinal, PackageNameIncludeVersion);
      end;
    pfNintendoSwitchProject:
      PackageNintendoSwitch(Self);
    pfDirectory, pfZip, pfTarGz, pfDeb:
      PackageDirectory(PackageFormatFinal);
    pfMacAppBundle, pfMacAppBundleZip:
      begin
        CreateMacAppBundle(Self, OutputPath, false);
        if PackageFormatFinal = pfMacAppBundleZip then
          ZipMacAppBundle(Self, OutputPath, PackageName(OS, CPU, PackageFormatFinal, PackageNameIncludeVersion));
      end;
    {$ifndef COMPILER_CASE_ANALYSIS}
    else raise EInternalError.Create('Unhandled PackageFormatFinal in DoPackage');
    {$endif}
  end;
end;

procedure TCastleProject.DoInstall(const Target: TTarget;
  const OS: TOS; const CPU: TCPU; const Mode: TCompilationMode;
  const PackageFormat: TPackageFormat;
  const PackageNameIncludeVersion: Boolean);
var
  PackageFormatFinal: TPackageFormatNoDefault;
begin
  Writeln(Format('Installing project "%s" for %s.',
    [Name, TargetCompleteToString(Target, OS, CPU)]));

  PackageFormatFinal := PackageFormatFinalize(Target, OS, CPU, PackageFormat);

  case PackageFormatFinal of
    pfAndroidApk, pfAndroidAppBundle:
      InstallAndroid(Self, Mode, PackageFormatFinal, PackageNameIncludeVersion);
    else
      raise Exception.Create('The "install" command is not useful for this target / OS / CPU right now. Install the application manually.');
  end;
end;

procedure TCastleProject.DoRun(const Target: TTarget;
  const OS: TOS; const CPU: TCPU;
  const Params: TCastleStringList);

  procedure MaybeUseWrapperToRun(var ExeName: String);
  var
    S: String;
  begin
    if OS in AllUnixOSes then
    begin
      S := Path + ChangeFileExt(ExecutableName, '') + '_run.sh';
      if RegularFileExists(S) then
      begin
        ExeName := S;
        Exit;
      end;

      S := Path + 'run.sh';
      if RegularFileExists(S) then
      begin
        ExeName := S;
        Exit;
      end;
    end;
  end;

  procedure MaybeUseWineToRun(var ExeName: String; const NewParams: TStrings);
  var
    WineExe: String;
  begin
    if (OS in AllWindowsOSes) and (DefaultOS in AllUnixOSes) then
    begin
      Writeln('Running Windows EXE on Unix, trying to use WINE.');

      WineExe := '';
      // try to find with suffix 32 or 64
      case CPU of
        i386: WineExe := FindExe('wine32');
        x86_64: WineExe := FindExe('wine64');
        else ;
      end;
      // try to use wine without a suffix
      if WineExe = '' then
        WineExe := FindExe('wine');

      if WineExe <> '' then
      begin
        NewParams.Insert(0, ExeName);
        ExeName := WineExe;
      end;
    end;
  end;

  procedure MaybeRunThroughMacAppBundle(var RunWorkingDir, ExeName: String; const NewParams: TStrings);
  var
    ExeInBundle: String;
  begin
    if (OS = Darwin) and Manifest.MacAppBundle then
    begin
      CreateMacAppBundle(Self, TempOutputPath(Path) + 'macos' + PathDelim, true, ExeInBundle);

      RunWorkingDir := ExtractFilePath(ExeInBundle);
      ExeName := ExeInBundle;

      { Executing using "open" is also a reasonable option, this simulates more what Finder
        is doing when double-clicking the application.
        However
        - it doesn't seem to have any benefit for us, over executing ExeInBundle directly
        - it doesn't capture the stdout/stderr of the application.

      ExeName := 'open';
      NewParams.Insert(0, '--new');
      NewParams.Insert(1, '--wait-apps');
      NewParams.Insert(2, TempOutputPath(Path) + 'macos' + PathDelim + Name + '.app');
      NewParams.Insert(3, '--args');
      }
    end;
  end;

var
  RunWorkingDir: String;
  ExeName: string;
  NewParams: TCastleStringList;
begin
  Writeln(Format('Running project "%s" for %s.',
    [Name, TargetCompleteToString(Target, OS, CPU)]));

  if Target = targetIOS then
    RunIOS(Self)
  else
  if (Target = targetAndroid) or (OS = Android) then
    RunAndroid(Self)
  else
  if Target = targetCustom then
  begin
    ExeName := Path + ChangeFileExt(ExecutableName, ExeExtensionOS(OS));
    RunWorkingDir := Path;

    MaybeUseWrapperToRun(ExeName);
    Writeln('Running ' + ExeName);
    NewParams := TCastleStringList.Create;
    try
      NewParams.Assign(Params);
      MaybeUseWineToRun(ExeName, NewParams);
      MaybeRunThroughMacAppBundle(RunWorkingDir, ExeName, NewParams);
      Flush(Output); // needed to see "Running Windows EXE on Unix, trying to use WINE." in right order in editor
      { We set current path to Path, not OutputPath, because data/ subdirectory is under Path. }
      RunCommandSimple(RunWorkingDir, ExeName, NewParams.ToArray, 'CASTLE_LOG', 'stdout');
    finally FreeAndNil(NewParams) end;
  end else
    raise Exception.Create('The "run" command is not useful for this OS / CPU right now. Run the application manually.');
end;

procedure TCastleProject.DoPackageSource(const PackageFormat: TPackageFormat;
  const PackageNameIncludeVersion: Boolean);
var
  PackageFormatFinal: TPackageFormatNoDefault;
  Pack: TPackageDirectory;
  Files: TCastleStringList;
  I: Integer;
  PackageFileName: string;
  Collector: TSourcePackageFiles;
  ExecutablePermission: Boolean;
begin
  Writeln(Format('Packaging source code of project "%s".', [Name]));

  if PackageFormat = pfDefault then
    PackageFormatFinal := pfZip
  else
    PackageFormatFinal := PackageFormat;

  Pack := TPackageDirectory.Create(Name);
  try
    Collector := TSourcePackageFiles.Create(Self);
    try
      Collector.Run;
      Files := Collector.CollectedFiles;
      for I := 0 to Files.Count - 1 do
      begin
        ExecutablePermission := (Files.Objects[I] <> nil) and (TIncludePath(Files.Objects[I]).ExecutablePermission);
        Pack.Add(Path + Files[I], Files[I], ExecutablePermission);
      end;
    finally FreeAndNil(Collector) end;

    PackageFileName := SourcePackageName(PackageNameIncludeVersion, PackageFormatFinal);
    Pack.Make(OutputPath, PackageFileName, PackageFormatFinal);
  finally FreeAndNil(Pack) end;
end;

function TCastleProject.PackageName(const OS: TOS; const CPU: TCPU;
  const PackageFormat: TPackageFormatNoDefault;
  const PackageNameIncludeVersion: Boolean): string;
begin
  Result := Name;
  if PackageNameIncludeVersion and (Version.DisplayValue <> '') then
    Result += '-' + Version.DisplayValue;
  Result += '-' + OSToString(OS) + '-' + CPUToString(CPU);
  case PackageFormat of
    pfZip, pfMacAppBundleZip: Result += '.zip';
    pfTarGz: Result += '.tar.gz';
    pfDeb: Result += '.deb';
    else ; // leave without extension for pfDirectory
  end;
end;

function TCastleProject.SourcePackageName(const PackageNameIncludeVersion: Boolean;
  const PackageFormatFinal: TPackageFormatNoDefault): string;
begin
  Result := Name;
  if PackageNameIncludeVersion and (Version.DisplayValue <> '') then
    Result += '-' + Version.DisplayValue;
  Result += '-src';

  case PackageFormatFinal of
    pfZip  : Result += '.zip';
    pfTarGz: Result += '.tar.gz';
    else raise Exception.CreateFmt('Package format "%s" not supported for source package', [
      PackageFormatToString(PackageFormatFinal)
    ]);
  end;
end;

procedure TCastleProject.DeleteFoundFile(const FileInfo: TFileInfo; var StopSearch: boolean);
begin
  if Verbose then
    Writeln('Deleting ' + FileInfo.AbsoluteName);
  CheckDeleteFile(FileInfo.AbsoluteName);
  Inc(DeletedFiles);
end;

function TCastleProject.NamePascal: string;
begin
  Result := MakeProjectPascalName(Name);
end;

procedure TCastleProject.GeneratedSourceFile(
  const TemplateRelativeUrl, TargetRelativePath, ErrorMessageMissingGameUnits: string;
  const CreateIfNecessary: boolean;
  out RelativeResult, AbsoluteResult: string);
var
  TemplateFile: string;
begin
  AbsoluteResult := TempOutputPath(Path, CreateIfNecessary) + TargetRelativePath;
  if CreateIfNecessary then
  begin
    TemplateFile := UriToFilenameSafe('castle-data:/' +TemplateRelativeUrl);
    if Manifest.GameUnits = '' then
      raise Exception.Create(ErrorMessageMissingGameUnits);
    ExtractTemplateFile(TemplateFile, AbsoluteResult, TemplateRelativeUrl, true);
  end;
  // This may not be true anymore, if user changes OutputPathBase
  // if not IsPrefix(Path, AbsoluteResult, true) then
  //   raise EInternalError.CreateFmt('Something is wrong with the temporary source location "%s", it is not within the project "%s"',
  //     [AbsoluteResult, Path]);
  RelativeResult := PrefixRemove(Path, AbsoluteResult, true);
end;

procedure TCastleProject.GeneratedSourceFile(
  const TemplateRelativeUrl, TargetRelativePath, ErrorMessageMissingGameUnits: string;
  const CreateIfNecessary: boolean);
var
  RelativeResult, AbsoluteResult: string;
begin
  GeneratedSourceFile(TemplateRelativeUrl, TargetRelativePath, ErrorMessageMissingGameUnits,
    CreateIfNecessary, RelativeResult, AbsoluteResult);
  // just ignore RelativeResult, AbsoluteResult output values
end;

function TCastleProject.AndroidSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;

  procedure InvalidAndroidSource(const FinalAndroidSource: string);
  begin
    raise Exception.Create('The android source library in "' + FinalAndroidSource + '" must export the necessary JNI functions for our integration to work. See the examples in "castle-engine/tools/build-tool/data/android/library_template_xxx.lpr".' + NL + 'It''s simplest to fix this error by removing the "android_source" from CastleEngineManifest.xml, and using only the "game_units" attribute in CastleEngineManifest.xml. Then the correct Android code will be auto-generated for you.');
  end;

const
  ErrorMessageMissingGameUnits = 'You must specify game_units="..." in the CastleEngineManifest.xml to enable build tool to create an Android project. Alternatively, you can specify android_source="..." in the CastleEngineManifest.xml, to explicitly indicate the Android library source code.';
var
  AndroidSourceContents, RelativeResult, AbsoluteResult, TemplateRelativeUrl: String;
begin
  { calculate RelativeResult, AbsoluteResult }
  if Manifest.AndroidSource <> '' then
  begin
    RelativeResult := Manifest.AndroidSource;
    AbsoluteResult := Path + RelativeResult;
  end else
  begin
    TemplateRelativeUrl := 'android/library_template_integrated.lpr';
    GeneratedSourceFile(TemplateRelativeUrl,
      'android' + PathDelim + NamePascal + '_android.lpr',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary, RelativeResult, AbsoluteResult);
    GeneratedSourceFile('castleautogenerated_template.pas',
      'android' + PathDelim + 'castleautogenerated.pas',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary);
  end;

  // for speed, do not check correctness if CreateIfNecessary = false
  if CreateIfNecessary then
  begin
    { check Android lpr file correctness.
      For now, do it even if we generated Android project from our own template
      (when AndroidSource = ''), to check our own work. }
    AndroidSourceContents := FileToString(AbsoluteResult);
    if Pos('ANativeActivity_onCreate', AndroidSourceContents) = 0 then
      InvalidAndroidSource(AbsoluteResult);
    if Pos('Java_io_castleengine_MainActivity_jniMessage', AndroidSourceContents) = 0 then
      InvalidAndroidSource(AbsoluteResult);
  end;

  if AbsolutePath then
    Result := AbsoluteResult
  else
    Result := RelativeResult;
end;

function TCastleProject.IOSSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
const
  ErrorMessageMissingGameUnits = 'You must specify game_units="..." in the CastleEngineManifest.xml to enable build tool to create an iOS project. Alternatively, you can specify ios_source="..." in the CastleEngineManifest.xml, to explicitly indicate the iOS library source code.';
var
  RelativeResult, AbsoluteResult: string;
begin
  if Manifest.IOSSource <> '' then
  begin
    RelativeResult := Manifest.IOSSource;
    AbsoluteResult := Path + RelativeResult;
  end else
  begin
    GeneratedSourceFile('ios/library_template.lpr',
      'ios' + PathDelim + NamePascal + '_ios.lpr',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary, RelativeResult, AbsoluteResult);
    GeneratedSourceFile('castleautogenerated_template.pas',
      'ios' + PathDelim + 'castleautogenerated.pas',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary);
  end;

  if AbsolutePath then
    Result := AbsoluteResult
  else
    Result := RelativeResult;
end;

function TCastleProject.NXSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
const
  ErrorMessageMissingGameUnits = 'You must specify game_units="..." in the CastleEngineManifest.xml to enable build tool to create a Nintendo Switch project.';
var
  RelativeResult, AbsoluteResult: string;
begin
  { Without this, we would also have an error, but NxNotSupported makes
    nicer error message. }
  NxNotSupported;

  GeneratedSourceFile('nintendo_switch/library_template.lpr',
    'nintendo_switch' + PathDelim + 'castle_nx.lpr',
    ErrorMessageMissingGameUnits,
    CreateIfNecessary, RelativeResult, AbsoluteResult);
  GeneratedSourceFile('castleautogenerated_template.pas',
    'nintendo_switch' + PathDelim + 'castleautogenerated.pas',
    ErrorMessageMissingGameUnits,
    CreateIfNecessary);

  if AbsolutePath then
    Result := AbsoluteResult
  else
    Result := RelativeResult;
end;

function TCastleProject.StandaloneSourceFile(const AbsolutePath, CreateIfNecessary: boolean): string;
const
  ErrorMessageMissingGameUnits = 'You must specify game_units or standalone_source in the CastleEngineManifest.xml to compile for the standalone platform';
var
  RelativeResult, AbsoluteResult: string;
begin
  if Manifest.StandaloneSource <> '' then
  begin
    RelativeResult := Manifest.StandaloneSource;
    AbsoluteResult := Path + RelativeResult;
  end else
  begin
    GeneratedSourceFile('standalone/program_template.dpr',
      'standalone' + PathDelim + NamePascal + '_standalone.dpr',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary, RelativeResult, AbsoluteResult);
    GeneratedSourceFile('castleautogenerated_template.pas',
      'standalone' + PathDelim + 'castleautogenerated.pas',
      ErrorMessageMissingGameUnits,
      CreateIfNecessary);
  end;

  if AbsolutePath then
    Result := AbsoluteResult
  else
    Result := RelativeResult;
end;

function TCastleProject.AndroidLibraryFile(const CPU: TCPU;
  const AbsolutePath: boolean): string;
var
  Ext: String;
begin
  Ext := '.so';
  if CPU <> cpuNone then
    Ext := '_' + CPUToString(CPU) + Ext;
  Result := InsertLibPrefix(ChangeFileExt(AndroidSourceFile(AbsolutePath, false), Ext));
end;

function TCastleProject.IOSLibraryFile(const AbsolutePath: boolean): string;
begin
  Result := InsertLibPrefix(ChangeFileExt(IOSSourceFile(AbsolutePath, false), '.a'));
end;

function TCastleProject.NXLibraryFile(const AbsolutePath: boolean): string;
begin
  Result := InsertLibPrefix(ChangeFileExt(NXSourceFile(AbsolutePath, false), '.a'));
end;

procedure TCastleProject.DoClean;

  { Delete a file, given as absolute FileName. }
  procedure TryDeleteAbsoluteFile(FileName: string);
  begin
    if RegularFileExists(FileName) then
    begin
      if Verbose then
        Writeln('Deleting ' + FileName);
      CheckDeleteFile(FileName);
      Inc(DeletedFiles);
    end;
  end;

  { Delete a file, given as FileName relative to project root. }
  procedure TryDeleteFile(FileName: string);
  begin
    TryDeleteAbsoluteFile(Path + FileName);
  end;

  procedure DeleteFilesRecursive(const Mask: string);
  begin
    FindFiles(Path, Mask, false, @DeleteFoundFile, [ffRecursive]);
  end;

  procedure DeleteExternalLibraries(const LibrariesOutputPath: String; const OS: TOS; const CPU: TCPU);
  var
    List: TCastleStringList;
    OutputFile, FileName: String;
  begin
    List := TCastleStringList.Create;
    try
      { CheckFilesExistence parameter for ExternalLibraries may be false.
        This way you can run "castle-engine clean" without setting $CASTLE_ENGINE_PATH . }
      ExternalLibraries(OS, CPU, Manifest.ProjectDependencies, List, false);
      for FileName in List do
      begin
        OutputFile := LibrariesOutputPath + ExtractFileName(FileName);
        TryDeleteFile(OutputFile);
      end;
    finally FreeAndNil(List) end;
  end;

var
  OS: TOS;
  CPU: TCPU;
  OutputP: string;
begin
  { delete OutputPath first, this also removes many files
    (but RemoveNonEmptyDir does not count them) }
  OutputP := TempOutputPath(Path, false);
  if DirectoryExists(OutputP) then
  begin
    RemoveNonEmptyDir(OutputP);
    Writeln('Deleted ', OutputP);
  end;

  DeletedFiles := 0;

  TryDeleteFile(ChangeFileExt(ExecutableName, ''));
  TryDeleteFile(ChangeFileExt(ExecutableName, '.exe'));
  TryDeleteFile(ChangeFileExt(ExecutableName, '.log'));

  if Manifest.AndroidSource <> '' then
  begin
    TryDeleteAbsoluteFile(AndroidLibraryFile(arm));
    TryDeleteAbsoluteFile(AndroidLibraryFile(aarch64));
  end;
  if Manifest.IOSSource <> '' then
    TryDeleteAbsoluteFile(IOSLibraryFile);

  for OS in TOS do
    for CPU in TCPU do
      if OSCPUSupported[OS, CPU] then
      begin
        { packages created by DoPackage? Or not, it's safer to not remove them. }
        // TryDeleteFile(PackageName(OS, CPU));

        DeleteExternalLibraries(ExtractFilePath(ExecutableName), OS, CPU);
      end;

  { compilation and editor backups }
  DeleteFilesRecursive('*~'); // editor backup, e.g. Emacs
  DeleteFilesRecursive('*.ppu'); // compilation
  DeleteFilesRecursive('*.o'); // compilation
  DeleteFilesRecursive('*.or'); // compilation
  DeleteFilesRecursive('*.compiled'); // Lazarus compilation
  DeleteFilesRecursive('*.rst'); // resource strings
  DeleteFilesRecursive('*.rsj'); // resource strings
  TryDeleteFile('castle-auto-generated-resources.res');

  Writeln('Deleted ', DeletedFiles, ' files');
end;

procedure TCastleProject.DoAutoGenerateTextures;
begin
  AutoGenerateTextures(Self);
end;

procedure TCastleProject.DoAutoGenerateClean(const CleanAll: Boolean);
begin
  if CleanAll then
    AutoGenerateCleanAll(Self)
  else
    AutoGenerateCleanUnused(Self);
end;

function TCastleProject.ExplicitStandaloneFile(const Ext: String): String;
begin
  if Manifest.StandaloneSource <> '' then
  begin
    { If we wanted to generate dpr/lpr file, then just use standalone_source value from manifest
      instead, preserving extension (whether it is dpr,lpr or something else). }
    if (Ext = '.dpr') or
       (Ext = '.lpr') then
      Result := Manifest.StandaloneSource
    else
      Result := ChangeFileExt(Manifest.StandaloneSource, Ext);
  end else
    Result := NamePascal + '_standalone' + Ext;
end;

procedure TCastleProject.DoGenerateProgram(const GuidFromName: Boolean);

  procedure Generate(const TemplateRelativePath, TargeRelativePath: string);
  var
    TemplateFile, TargetFile: string;
  begin
    TemplateFile := UriToFilenameSafe('castle-data:/' +TemplateRelativePath);
    TargetFile := Path + TargeRelativePath;
    ExtractTemplateFile(TemplateFile, TargetFile, TemplateRelativePath, true);
    Writeln('Generated ', ExtractRelativePath(Path, TargetFile));
  end;

  procedure GenerateStandaloneProgram(const Ext: string);
  var
    TargetFile: String;
  begin
    TargetFile := ExplicitStandaloneFile(Ext);
    Generate(
      'standalone/program_template' + Ext,
      TargetFile);
  end;

begin
  FGuidFromName := GuidFromName;

  { Generate castleautogenerated.pas before generating LPI,
    because LPI contains list of units that should include castleautogenerated.pas. }
  Generate('castleautogenerated_template.pas', 'castleautogenerated.pas');

  if Manifest.GameUnits = '' then
  begin
    WritelnWarning('CastleEngineManifest.xml does not specify game_units="...".' + NL +
      '  We cannot auto-generate main Pascal program (dpr, lpr) in this case.' + NL +
      '  We will still auto-generate other project files, assuming you have hand-crafted dpr/lpr.');
  end else
    GenerateStandaloneProgram('.dpr');

  GenerateStandaloneProgram('.lpi');
  GenerateStandaloneProgram('.dproj');

  if (Manifest.StandaloneSource <> '') and
     (not SameText(ExtractFileExt(Manifest.StandaloneSource), '.dpr')) then
    WritelnWarning('Main program (standalone_source) does not have ".dpr" extension.' + NL +
      '  It will not be possible to compile this project with Delphi.' + NL +
      '  We recommend changing the standalone_source extension to ".dpr".' + NL +
      '  The ".dpr" extension will work with both FPC/Lazarus and Delphi.');
end;

procedure TCastleProject.DoEditor;
begin
  DoEditorRebuildIfNeeded;
  DoEditorRun(0);
end;

procedure TCastleProject.DoEditorRebuildIfNeeded;
var
  CgePath, EditorPath: String;
begin
  if Trim(Manifest.EditorUnits) <> '' then
  begin
    { Check CastleEnginePath, since without this, compiling custom castle-editor.lpi
      will always fail. }
    CgePath := CastleEnginePath;
    if CgePath = '' then
      raise Exception.Create(SCannotFindCgePath);

    // create custom editor directory
    EditorPath := TempOutputPath(Path) + 'editor' + PathDelim;
    { Do not remove previous directory contents,
      allows to reuse previous lazbuild compilation results.
      Just silence ExtractTemplate warnings when overriding. }
    ExtractTemplate('custom_editor_template/', EditorPath, true);

    // use lazbuild to compile CGE packages and CGE editor
    RunLazbuild(Path, [CgePath + 'packages/castle_base.lpk']);
    RunLazbuild(Path, [CgePath + 'packages/castle_components.lpk']);
    RunLazbuild(Path, [CgePath + 'packages/castle_editor_components.lpk']);
    RunLazbuild(Path, [EditorPath + 'castle_editor_automatic_package.lpk']);
    RunLazbuild(Path, [EditorPath + 'castle_editor.lpi']);
  end;
end;

procedure TCastleProject.DoEditorRun(const WaitForProcessId: TProcessId);

  {$ifdef UNIX}
  { Copied from FPC packages/fcl-extra/src/unix/daemonapp.inc .
    We need to become a daemon to reliably keep working even if parent process
    terminates. Otherwise restarting CGE editor after rebuilding new editor
    fails. }
  const
    SErrFailedToFork          = 'Failed to fork daemon process.';

  procedure DaemonizeProgram;
  var
    pid, sid: TPid;
  begin
    pid := FpFork;
    if (pid<0) then
      raise Exception.Create(SErrFailedToFork);
    if pid>0 then
    begin
      // We are now in the main program, which has to terminate
      FpExit(0);
    end
    else
    begin
      // Here we are in the daemonized proces
      sid := FpSetsid;
      if sid < 0 then
        raise Exception.Create(SErrFailedToFork);
      // Reset the file-mask
      FpUmask(0);
      // Change the current directory, to avoid locking the current directory
      chdir('/');
      FpClose(StdInputHandle);
      FpClose(StdOutputHandle);
      FpClose(StdErrorHandle);
    end;
  end;
  {$endif}

  { Copy external libraries to LibrariesOutputPath. }
  procedure AddExternalLibraries(const LibrariesOutputPath: String);
  var
    List: TCastleStringList;
    OutputFile, FileName: String;
    EditorProjectDependencies: TProjectDependencies;
  begin
    List := TCastleStringList.Create;
    try
      EditorProjectDependencies := TProjectDependencies.Create;
      try
        EditorProjectDependencies.Dependencies := [
          // to read fonts
          depFreetype,
          // to read PNG
          depZlib, depPng,
          // to play sound
          depSound,
          // to read OggVorbis
          depOggVorbis,
          // not used now by the editor -- but likely will be used in the future, e.g. to check for new version by HTTPS query.
          depHttps
        ];
        ExternalLibraries(DefaultOS, DefaultCPU, EditorProjectDependencies, List);
        for FileName in List do
        begin
          OutputFile := LibrariesOutputPath + ExtractFileName(FileName);
          WritelnVerbose('Copying library to ' + OutputFile);
          CheckCopyFile(FileName, OutputFile);
        end;
      finally FreeAndNil(EditorProjectDependencies) end;
    finally FreeAndNil(List) end;
  end;

var
  EditorPath: String;

  function GetEditorExe(const Suffix: String): String;
  begin
    Result := EditorPath + 'castle-editor' + Suffix + ExeExtension;
    {$ifdef DARWIN}
    // on macOS, run new editor through app bundle
    Result := Result + '.app/Contents/MacOS/castle-editor' + Suffix;
    {$endif}
  end;

var
  EditorExe{$ifndef DARWIN}, NewEditorExe{$endif}: String;
  NewEnvironment: TStringList;
begin
  {$ifdef UNIX}
  DaemonizeProgram;
  {$endif}

  if WaitForProcessId <> 0 then
    WaitForProcessExit(WaitForProcessId);

  NewEnvironment := nil;

  if Trim(Manifest.EditorUnits) = '' then
  begin
    EditorExe := FindExeCastleTool('castle-editor');
    if EditorExe = '' then
      raise Exception.Create('Cannot find "castle-editor" program on $PATH or within $CASTLE_ENGINE_PATH/bin directory.');
  end else
  begin
    // here EditorPath and EditorExe are calculated like in DoEditorRebuildIfNeeded
    EditorPath := TempOutputPath(Path) + 'editor' + PathDelim;

    { This can be done only once previous editor process finished,
      to not block EXE and DLL files on Windows. }
    AddExternalLibraries(EditorPath);

    {$ifdef DARWIN}
    { on macOS, run new editor through app bundle.
      We never rename it (no need to, and this is simplest), it always remains with -new suffix). }
    EditorExe := GetEditorExe('-new');
    {$else}
    NewEditorExe := GetEditorExe('-new');
    EditorExe := GetEditorExe('');
    { Rename NewEditorExe -> EditorExe.
      If you use DoEditorRebuildIfNeeded + DoEditorRun, the NewEditorExe must exist.
      But if you directly use DoEditorRun (e.g. using "Run Last Editor" from CGE editor)
      then NewEditorExe may not exist. }
    if RegularFileExists(NewEditorExe) then
      CheckRenameFile(NewEditorExe, EditorExe);
    {$endif}

    { When running custom editor build, we must make sure it can find CGE location,
      in particular so it can find editor's data (which means InternalCastleDesignData
      must be useful, and it must be able to load gfx stuff like gizmo images and
      layer_physics_simulation.castle-user-interface).

      So it is not sufficient that only build tool can find the CGE path (and use it for compilation).
      CGE editor must be able to find it too.

      Note that in usual user installation (when CASTLE_ENGINE_PATH is undefined):
      - Build tool will be inside CGE/bin/  and can detect CGE location reliably
        based on exe name.
      - OTOH editor, when it is a custom editor build, is in project's directory and will
        not be able to detect CGE location reliably based on exe name.
      This means build tool must pass it to CGE editor.

      Relevant bugreport: https://forum.castle-engine.io/t/castle-lines-2d/701/5
    }
    if (CastleEnginePath <> '') and
       (GetEnvironmentVariable('CASTLE_ENGINE_PATH') = '') then
    begin
      NewEnvironment := EnvironmentStrings;
      NewEnvironment.Values['CASTLE_ENGINE_PATH'] := CastleEnginePath;
    end;
  end;

  { Regardless of all above conditions (
    whether Manifest.EditorUnits is empty or not,
    whether we renamed NewEditorExe or not,
    whether it's in macOS bundle or not)... EditorExe must exist. }
  if not RegularFileExists(EditorExe) then
    raise Exception.Create('Editor should be compiled, but we cannot find file "' + EditorExe + '"');

  { Running with CurrentDirectory = Path, so that at least on Windows
    editor can automatically use the DLL files inside the project, like libeffekseer.dll.

    TODO: In the long run, this should change to use EditorPath as current path.
    Running editor should not "lock" the project DLLs on Windows.
    We should have a system of services for desktop, to manage DLLs, including custom
    DLLs like Effekseer and FMOD. }
  RunCommandNoWait(Path, EditorExe, [ManifestFile], [], NewEnvironment);
end;

procedure TCastleProject.DoOutput(const OutputKey: String);
begin
  case OutputKey of
    'executable-name': Writeln(ExecutableName);
    'name': Writeln(Name);
    'pascal-name': Writeln(NamePascal);
    'search-paths': Writeln(Manifest.SearchPaths.Text);
    'version': Writeln(Manifest.Version.DisplayValue);
    'version-code': Writeln(Manifest.Version.Code);
    else raise Exception.CreateFmt('Unsupported output key: "%s"', [OutputKey]);
  end;
end;

procedure TCastleProject.AddMacrosAndroid(const Macros: TStringStringMap);
const
  AndroidScreenOrientation: array [TScreenOrientation] of string =
  ('unspecified', 'sensorLandscape', 'sensorPortrait');

  AndroidScreenOrientationFeature: array [TScreenOrientation] of string =
  ('',
   '<uses-feature android:name="android.hardware.screen.landscape"/>',
   '<uses-feature android:name="android.hardware.screen.portrait"/>');

  function AndroidActivityTheme: string;
  begin
    if FullscreenImmersive then
      Result := 'android:Theme.NoTitleBar.Fullscreen'
    else
      Result := 'android:Theme.NoTitleBar';
  end;

  function AndroidActivityLoadLibraries: string;
  begin
    { Some Android devices work without this clause, some don't.
      TODO: This should be moved to CastleEngineService.xml, not hardcoded here. }
    Result := '';
    if depSound in Dependencies then
      Result += 'safeLoadLibrary("openal");' + NL;
    if depOggVorbis in Dependencies then
      Result += 'safeLoadLibrary("tremolo");' + NL;
    if depFreetype in Dependencies then
      Result += 'safeLoadLibrary("freetype");' + NL;
    if depPng in Dependencies then
      Result += 'safeLoadLibrary("png");' + NL;
    { Necessary, otherwise FMOD is not initialized correctly, and reports
        [ERR] FMOD_JNI_GetEnv                          : JNI_OnLoad has not run, should have occurred during System.LoadLibrary.
      and reports internal error even from FMOD_System_Create. }
    if AndroidServices.HasService('fmod') then
      Result += 'safeLoadLibrary("fmod");' + NL;
  end;

  { Android ABI list like '"armeabi-v7a","arm64-v8a"' }
  function AndroidAbiList: String;
  var
    CPU: TCPU;
  begin
    Result := '';
    for CPU in AndroidCPUS do
      Result := SAppendPart(Result, ',', '"' + CPUToAndroidArchitecture(CPU) + '"');
  end;

  { Android ABI list like 'armeabi-v7a arm64-v8a' }
  function AndroidAbiListMakefile: String;
  var
    CPU: TCPU;
  begin
    Result := '';
    for CPU in AndroidCPUS do
      Result := SAppendPart(Result, ' ', CPUToAndroidArchitecture(CPU));
  end;

var
  AndroidLibraryName: string;
  Service: TService;
begin
  AndroidLibraryName := ChangeFileExt(ExtractFileName(AndroidSourceFile(true, false)), '');
  Macros.Add('ANDROID_LIBRARY_NAME'                , AndroidLibraryName);
  Macros.Add('ANDROID_ACTIVITY_THEME'              , AndroidActivityTheme);
  Macros.Add('ANDROID_SCREEN_ORIENTATION'          , AndroidScreenOrientation[ScreenOrientation]);
  Macros.Add('ANDROID_SCREEN_ORIENTATION_FEATURE'  , AndroidScreenOrientationFeature[ScreenOrientation]);
  Macros.Add('ANDROID_ACTIVITY_LOAD_LIBRARIES'     , AndroidActivityLoadLibraries);
  Macros.Add('ANDROID_COMPILE_SDK_VERSION'         , IntToStr(AndroidCompileSdkVersion));
  Macros.Add('ANDROID_MIN_SDK_VERSION'             , IntToStr(AndroidMinSdkVersion));
  Macros.Add('ANDROID_TARGET_SDK_VERSION'          , IntToStr(AndroidTargetSdkVersion));
  Macros.Add('ANDROID_ASSOCIATE_DOCUMENT_TYPES'    , AssociateDocumentTypes.ToIntentFilter);
  Macros.Add('ANDROID_LOG_TAG'                     , Copy(Name, 1, MaxAndroidTagLength));
  Macros.Add('ANDROID_ABI_LIST'                    , AndroidAbiList);
  Macros.Add('ANDROID_ABI_LIST_MAKEFILE'           , AndroidAbiListMakefile);

  for Service in AndroidServices do
    ParametersAddMacros(Macros, Service.Parameters, 'ANDROID.' + Service.Name + '.');
end;

procedure TCastleProject.AddMacrosIOS(const Macros: TStringStringMap);
const
  IOSScreenOrientation: array [TScreenOrientation] of string = (
    #9#9'<string>UIInterfaceOrientationPortrait</string>' + NL +
    #9#9'<string>UIInterfaceOrientationPortraitUpsideDown</string>' + NL +
    #9#9'<string>UIInterfaceOrientationLandscapeLeft</string>' + NL +
    #9#9'<string>UIInterfaceOrientationLandscapeRight</string>' + NL,

    #9#9'<string>UIInterfaceOrientationLandscapeLeft</string>' + NL +
    #9#9'<string>UIInterfaceOrientationLandscapeRight</string>' + NL,

    #9#9'<string>UIInterfaceOrientationPortrait</string>' + NL +
    #9#9'<string>UIInterfaceOrientationPortraitUpsideDown</string>' + NL
  );

  IOSCapabilityEnable =
    #9#9#9#9#9#9#9'com.apple.%s = {' + NL +
    #9#9#9#9#9#9#9#9'enabled = 1;' + NL +
    #9#9#9#9#9#9#9'};' + NL;

  { QualifiedName for iOS: either qualified_name, or ios.override_qualified_name. }
  function IOSQualifiedName: string;
  begin
    if Manifest.IOSOverrideQualifiedName <> '' then
      Result := Manifest.IOSOverrideQualifiedName
    else
      Result := QualifiedName;
  end;

  procedure LaunchImageStoryboardInitialize;
  var
    Img: TCastleImage;
  begin
    if Manifest.LaunchImageStoryboard.Path <> '' then
    begin
      if ExtractFileExt(Manifest.LaunchImageStoryboard.Path) <> '.png' then
        raise Exception.CreateFmt('Launch image storyboard must be a PNG file, but is "%s"', [
          Manifest.LaunchImageStoryboard.Path
        ]);
      Img := LoadImage(Manifest.LaunchImageStoryboard.Path);
      try
        FLaunchImageStoryboardWidth := Img.Width;
        FLaunchImageStoryboardHeight := Img.Height;
      finally FreeAndNil(Img) end;
    end;
  end;

var
  InfoPList, IOSTargetAttributes, IOSRequiredDeviceCapabilities, IOSSystemCapabilities: string;
  Service: TService;
  IOSVersion: TProjectVersion;
  GccPreprocessorDefinitions: String;
begin
  InfoPList := '';

  // first time: initialize launch image storyboard
  if not FLaunchImageStoryboardInitialized then
  begin
    FLaunchImageStoryboardInitialized := true;
    LaunchImageStoryboardInitialize;
  end;
  // define macros related to launch image storyboard
  if IOSHasLaunchImageStoryboard then
  begin
    Macros.Add('IOS_LAUNCH_IMAGE_WIDTH' , IntToStr(FLaunchImageStoryboardWidth));
    Macros.Add('IOS_LAUNCH_IMAGE_HEIGHT', IntToStr(FLaunchImageStoryboardHeight));
    Macros.Add('IOS_LAUNCH_IMAGE_DISPLAY_WIDTH' , IntToStr(Round(256 * Manifest.LaunchImageStoryboard.Scale)));
    Macros.Add('IOS_LAUNCH_IMAGE_DISPLAY_HEIGHT', IntToStr(Round(256 * Manifest.LaunchImageStoryboard.Scale)));
    Macros.Add('IOS_LAUNCH_BACKGROUND_RED'  , FloatToStrDot(Manifest.LaunchImageStoryboard.BackgroundColor[0]));
    Macros.Add('IOS_LAUNCH_BACKGROUND_GREEN', FloatToStrDot(Manifest.LaunchImageStoryboard.BackgroundColor[1]));
    Macros.Add('IOS_LAUNCH_BACKGROUND_BLUE' , FloatToStrDot(Manifest.LaunchImageStoryboard.BackgroundColor[2]));
    Macros.Add('IOS_LAUNCH_BACKGROUND_ALPHA', FloatToStrDot(Manifest.LaunchImageStoryboard.BackgroundColor[3]));
    InfoPList := SAppendPart(InfoPList, NL,
      '<key>UILaunchStoryboardName</key> <string>Launch Screen</string>');
  end;

  if Manifest.IOSOverrideVersion <> nil then
    IOSVersion := Manifest.IOSOverrideVersion
  else
    IOSVersion := Manifest.Version;
  Macros.Add('IOS_QUALIFIED_NAME', IOSQualifiedName);
  Macros.Add('IOS_VERSION', IOSVersion.DisplayValue);
  Macros.Add('IOS_VERSION_CODE', IntToStr(IOSVersion.Code));
  Macros.Add('IOS_LIBRARY_BASE_NAME' , ExtractFileName(IOSLibraryFile));
  Macros.Add('IOS_STATUSBAR_HIDDEN', Iff(FullscreenImmersive, 'YES', 'NO'));
  Macros.Add('IOS_SCREEN_ORIENTATION', IOSScreenOrientation[ScreenOrientation]);
  InfoPList := SAppendPart(InfoPList, NL,
    AssociateDocumentTypes.ToPListSection(IOSQualifiedName, 'AppIcon'));
  if not Manifest.UsesNonExemptEncryption then
    InfoPList := SAppendPart(InfoPList, NL,
      '<key>ITSAppUsesNonExemptEncryption</key> <false/>');
  Macros.Add('IOS_EXTRA_INFO_PLIST', InfoPList);

  IOSTargetAttributes := '';
  IOSRequiredDeviceCapabilities := '';
  if Manifest.IOSTeam <> '' then
  begin
    IOSTargetAttributes := IOSTargetAttributes +
      #9#9#9#9#9#9'DevelopmentTeam = ' + Manifest.IOSTeam + ';' + NL;
    Macros.Add('IOS_DEVELOPMENT_TEAM_LINE', 'DEVELOPMENT_TEAM = ' + Manifest.IOSTeam + ';');
  end else
  begin
    Macros.Add('IOS_DEVELOPMENT_TEAM_LINE', '');
  end;

  IOSSystemCapabilities := '';
  // TODO: These service-specifics should be in CastleEngineService.xml now
  if IOSServices.HasService('apple_game_center') then
  begin
    IOSSystemCapabilities := IOSSystemCapabilities +
      Format(IOSCapabilityEnable, ['GameCenter']);
    IOSRequiredDeviceCapabilities := IOSRequiredDeviceCapabilities +
      #9#9'<string>gamekit</string>' + NL;
  end;
  if IOSServices.HasService('icloud_for_save_games') then
    IOSSystemCapabilities := IOSSystemCapabilities +
      Format(IOSCapabilityEnable, ['iCloud']);
  if IOSServices.HasService('in_app_purchases') then
    IOSSystemCapabilities := IOSSystemCapabilities +
      Format(IOSCapabilityEnable, ['InAppPurchase']);
  // If not empty, add IOSSystemCapabilities to IOSTargetAttributes,
  // wrapped in SystemCapabilities = { } block.
  if IOSSystemCapabilities <> '' then
      IOSTargetAttributes := IOSTargetAttributes +
        #9#9#9#9#9#9'SystemCapabilities = {' + NL +
        IOSSystemCapabilities +
        #9#9#9#9#9#9'};' + NL;

  Macros.Add('IOS_TARGET_ATTRIBUTES', IOSTargetAttributes);
  Macros.Add('IOS_REQUIRED_DEVICE_CAPABILITIES', IOSRequiredDeviceCapabilities);
  Macros.Add('IOS_EXPORT_METHOD', IOSExportMethod);

  if IOSServices.HasService('icloud_for_save_games') then
    Macros.Add('IOS_CODE_SIGN_ENTITLEMENTS', 'CODE_SIGN_ENTITLEMENTS = "' + Name + '/icloud_for_save_games.entitlements";')
  else
    Macros.Add('IOS_CODE_SIGN_ENTITLEMENTS', '');

  GccPreprocessorDefinitions := '';
  // Since right now we always compile with CASTLE_TREMOLO_STATIC,
  // we just always behave like ogg_vorbis service was included.
  //if depOggVorbis in Dependencies then
    GccPreprocessorDefinitions := GccPreprocessorDefinitions + '"ONLY_C=1",' + NL;
  Macros.Add('IOS_GCC_PREPROCESSOR_DEFINITIONS', GccPreprocessorDefinitions);

  for Service in IOSServices do
    ParametersAddMacros(Macros, Service.Parameters, 'IOS.' + Service.Name + '.');
end;

function TCastleProject.ReplaceMacros(const Source: string): string;

  function MakePathsStr(const Paths: TStringList; const Absolute: Boolean): String;
  var
    S, Dir: string;
  begin
    Result := '';
    for S in Paths do
    begin
      if Result <> '' then
        Result := Result + ';';
      if Absolute then
        Dir := CombinePaths(Path, S)
      else
        Dir := S;
      Result := Result + Dir;
    end;

    { For ABSOLUTE_xxx macros, add Path (project directory).
      It is not necessary for relative paths, as their handling always includes current dir for now.
      Testcase: examples/advanced_editor/CastleEngineManifest.xml ,
      without this the "castle-engine editor" would not find GameControls. }
    if Absolute then
      Result := SAppendPart(Result, ';', Path);
  end;

  function DelphiSearchPaths: String;

    { Make DPROJ generated on Windows and Unix the same.
      We use slashes as they work on both Windows and Unix
      (through Delphi IDE is for now only on Windows, so using backslashes would be OK too.) }
    function PathNormalizeDelimiter(const S: String): String;
    begin
      Result := SReplaceChars(S, '\', '/');
    end;

  var
    RelativeEnginePaths: Boolean;
    S, EnginePathPrefix: String;
  begin
    RelativeEnginePaths := IsPrefix(
      ExpandFileName(CastleEnginePath),
      ExpandFileName(Path),
      not FileNameCaseSensitive);
    if RelativeEnginePaths then
      EnginePathPrefix := ExtractRelativePath(Path, CastleEnginePath) + 'src/'
    else
      EnginePathPrefix := CastleEnginePath + 'src/';
    Result := '';
    for S in EnginePaths do
      Result := SAppendPart(Result, ';', PathNormalizeDelimiter(EnginePathPrefix + S));
    for S in EnginePathsDelphi do
      Result := SAppendPart(Result, ';', PathNormalizeDelimiter(EnginePathPrefix + S));
    for S in Manifest.SearchPaths do
      Result := SAppendPart(Result, ';', PathNormalizeDelimiter(S));
  end;

  procedure AddMacrosLazarusProject(const Macros: TStringStringMap);
  var
    PascalFiles: TStringList;
    PascalFile, PascalFilesXml: String;
    I: Integer;
  begin
    { As an optimization, do not calculate LAZARUS_PROJECT_ALL_UNITS
      macro value when not needed. }
    if Pos('${LAZARUS_PROJECT_ALL_UNITS}', Source) <> 0 then
    begin
      PascalFiles := Manifest.FindPascalFiles;
      try
        { Make the files' list sorted,
          to have the `castle-engine generate-program`
          output fully deterministic for a given project,
          and indepedent from source OS, undefined ordef of FindFiles etc.
          This avoids diffs in version control if nothing changed. }
        PascalFiles.CaseSensitive := true;
        PascalFiles.Sort;

        PascalFilesXml := '';
        I := 1;
        for PascalFile in PascalFiles do
        begin
          if LowerCase(ExtractFileExt(PascalFile)) = '.inc' then
          begin
            PascalFilesXml += Format(
              '      <Unit%d>' + NL +
              '        <Filename Value="%s"/>' + NL +
              '        <IsPartOfProject Value="True"/>' + NL +
              '      </Unit%d>' + NL, [
              I,
              PascalFile,
              I
            ]);
          end else
          begin
            PascalFilesXml += Format(
              '      <Unit%d>' + NL +
              '        <Filename Value="%s"/>' + NL +
              '        <IsPartOfProject Value="True"/>' + NL +
              '        <UnitName Value="%s"/>' + NL +
              '      </Unit%d>' + NL, [
              I,
              PascalFile,
              DeleteFileExt(ExtractFileName(PascalFile)),
              I
            ]);
          end;
          Inc(I);
        end;
        Macros.Add('LAZARUS_PROJECT_ALL_UNITS_COUNT', IntToStr(PascalFiles.Count + 1));
        Macros.Add('LAZARUS_PROJECT_ALL_UNITS', PascalFilesXml);
      finally FreeAndNil(PascalFiles) end;
    end;
  end;

  function IcoPath: String;
  begin
    Result := Icons.FindExtension(['.ico']);
    { Let it be '' if not found, don't make warning, as this is only used by Delphi DPROJ. }
  end;

  function ProjectGuid: String;
  var
    MyGuid: TGUID;
  begin
    if FGuidFromName then
      MyGuid := CreateGUIDFromHash(QualifiedName)
    else
      CreateGUID(MyGuid);
    Result := GUIDToString(MyGuid);
  end;

  function DefinesAsCompilerOptions: String;
  var
    D: String;
  begin
    Result := '';
    for D in Manifest.Defines do
      Result := Result + '-d' + D + NL;
  end;

  function DefinesSemicolonSeparated: String;
  var
    D: String;
  begin
    Result := '';
    for D in Manifest.Defines do
      Result := Result + D + ';';
  end;

  { Generate proper DeployFile elements for DPROJ,
    to deploy programs with their data using PAServer.

    Sample:
      <DeployFile LocalName="data\Dino.gltf" Configuration="Debug" Class="File">
          <Platform Name="Linux64">
              <RemoteDir>./data/</RemoteDir>
              <RemoteName>Dino.gltf</RemoteName>
              <Overwrite>true</Overwrite>
          </Platform>
          ... // repeat for all platforms
      </DeployFile>
      // repeat for release too
  }
  function DelphiDprojDeployFiles: String;
  const
    AllDelphiPlatformNames: array [0..2] of String = (
      { For now, commented out platforms that we don't use.
        These platforms make DPROJ extra large
        ( see https://forum.castle-engine.io/t/there-is-a-problem-with-generating-files-for-testing-new-features/1264/6 )
        so for now, at least make it smaller but commenting out platforms that we don't use. }
      // 'Android',
      // 'Android64',
      // 'iOSDevice64',
      // 'iOSSimARM64',
      'Linux64',
      // 'OSX64',
      // 'OSXARM64',
      'Win32',
      'Win64'
    );
    AllDelphiConfigNames: array [0..1] of String = (
      'Debug',
      'Release'
    );
  var
    ConfigName, PlatformName, FileRelativeName: String;
    Files: TStringList;
    ResultBuilder: TStringBuilder;
  begin
    ResultBuilder := TStringBuilder.Create;
    try
      Files := PackageFiles(true, cpDesktop);
      try
        { Make the files' list sorted,
          to have the `castle-engine generate-program`
          output fully deterministic for a given project,
          and indepedent from source OS, undefined ordef of FindFiles etc.
          This avoids diffs in version control if nothing changed. }
        Files.CaseSensitive := true;
        Files.Sort;

        { Note that we try to minimize output below, as it may get really
          large for many files.
          ( see https://forum.castle-engine.io/t/there-is-a-problem-with-generating-files-for-testing-new-features/1264/6 )
          That's also why we don't add too much whitespace and indentation below.
        }

        if Files.Count <> 0 then
        begin
          ResultBuilder.Append(
            '<!-- Auto-generated list of deploy files.' + NL +
            '     We need to list all "data" files to deploy Delphi applications' + NL +
            '     to non-Windows targets,' + NL +
            '     see https://castle-engine.io/delphi_linux#data_files_deployment .' + NL +
            '-->' + NL);
        end;

        for ConfigName in AllDelphiConfigNames do
        begin
          for FileRelativeName in Files do
          begin
            ResultBuilder.Append(Format(
              '<DeployFile LocalName="%s\%s" Configuration="%s" Class="File">' + NL, [
                TCastleManifest.DataName,
                FileRelativeName,
                ConfigName
              ]));
            for PlatformName in AllDelphiPlatformNames do
              ResultBuilder.Append(Format(
                ' <Platform Name="%s">' + NL +
                '  <RemoteDir>./%s/%s</RemoteDir>' + NL +
                '  <RemoteName>%s</RemoteName>' + NL +
                '  <Overwrite>true</Overwrite>' + NL +
                ' </Platform>' + NL, [
                  PlatformName,
                  TCastleManifest.DataName,
                  ExtractFilePath(FileRelativeName),
                  ExtractFileName(FileRelativeName)
                ]));
            ResultBuilder.Append(
              '</DeployFile>' + NL);
          end;
        end;
      finally FreeAndNil(Files) end;
      Result := ResultBuilder.ToString;
    finally FreeAndNil(ResultBuilder) end;
  end;

  { Add macros specifically useful by Delphi project files. }
  procedure AddMacrosDproj(const Macros: TStringStringMap;
    const StandaloneSource: String);
  var
    WelcomePageFile, IncludedFiles, WelcomePageXml: String;
    // PascalFiles unused -- see comment below
    // PascalFiles: TStringList;
    // PascalFile, UnitFileForDelphi: String;
  begin
    WelcomePageFile := WelcomePage;

    { Optimize: do not calculate DPROJ_INCLUDED_FILES when not necessary. }
    if Pos('${DPROJ_INCLUDED_FILES}', Source) <> 0 then
    begin
      IncludedFiles := '';

      { Add project units to IncludedFiles

        Commented out: Note that we don't add units (like castleautogenerated.pas,
        code/gameinitialize.pas, code/gameview*.pas) to DPROJ.
        While possible, it would be pointless: Delphi IDE doesn't actually
        display them as part of the project, unless they are repeated in DPR.

        And we don't want to add Manifest.FindPascalFiles to DPR.
        Because we are not 100% certain that all units in Manifest.FindPascalFiles
        should be linked into the application, and for all platforms.
        We obtain Manifest.FindPascalFiles by scanning code directories,
        we could pick units like code/unused.pas or code/onlymobile.pas .

        Ultimately, we'd have to ask user to specify every used unit
        explicitly in CastleEngineManifest.xml, for us to add it to Delphi project.
        Doing it automatically based on Manifest.FindPascalFiles is too risky. }
      (*
      PascalFiles := Manifest.FindPascalFiles;
      try
        { Make the files' list sorted,
          to have the `castle-engine generate-program`
          output fully deterministic for a given project,
          and indepedent from source OS, undefined ordef of FindFiles etc.
          This avoids diffs in version control if nothing changed. }
        PascalFiles.CaseSensitive := true;
        PascalFiles.Sort;

        for PascalFile in PascalFiles do
        begin
          if LowerCase(ExtractFileExt(PascalFile)) = '.pas' then
          begin
            { Normalize paths to Pascal files to use backslashes
              (seems more standard in Delphi IDE, which is Windows-only). }
            UnitFileForDelphi := SReplaceChars(PascalFile, '/', '\');
            IncludedFiles := SAppendPart(IncludedFiles, NL,
              '        <DCCReference Include="' + UnitFileForDelphi + '"/>');
          end;
        end;
      finally FreeAndNil(PascalFiles) end;
      *)

      { Add lines like this to define files:
          <None Include="README.md"/>
          <None Include="play_animation_standalone.dpr"/>
        This allows to easily open them from IDE, also it seems "welcome page"
        (see below) has to be listed here to work. }
      if WelcomePageFile <> '' then
        IncludedFiles := SAppendPart(IncludedFiles, NL,
          '        <None Include="' + WelcomePageFile + '"/>');
      if StandaloneSource <> '' then
        IncludedFiles := SAppendPart(IncludedFiles, NL,
          '        <None Include="' + StandaloneSource + '"/>');
      Macros.Add('DPROJ_INCLUDED_FILES', IncludedFiles);
    end;

    { Define welcome page by XML content like this:

        <WelcomePageFile Path="README.md"/>
        <WelcomePageFolder/>
    }
    if WelcomePageFile <> '' then
    begin
      WelcomePageXml := Format(
        '                <WelcomePageFile Path="%s"/>' + NL +
        '                <WelcomePageFolder/>', [WelcomePageFile])
    end else
      WelcomePageXml := '';
    Macros.Add('DPROJ_WELCOME_PAGE', WelcomePageXml);

    { As an optimization, do not calculate DPROJ_DEPLOY_FILES
      macro value when not needed. }
    if Pos('${DPROJ_DEPLOY_FILES}', Source) <> 0 then
      Macros.Add('DPROJ_DEPLOY_FILES', DelphiDprojDeployFiles);
  end;

var
  NonEmptyAuthor, StandaloneSource: string;
  Macros: TStringStringMap;
begin
  if Author = '' then
    NonEmptyAuthor := 'Unknown Author'
  else
    NonEmptyAuthor := Author;

  StandaloneSource := ExplicitStandaloneFile('.dpr');

  Macros := TStringStringMap.Create;
  try
    Macros.Add('DOLLAR'          , '$');
    Macros.Add('VERSION_MAJOR'   , IntToStr(Version.Items[0]));
    Macros.Add('VERSION_MINOR'   , IntToStr(Version.Items[1]));
    Macros.Add('VERSION_RELEASE' , IntToStr(Version.Items[2]));
    Macros.Add('VERSION_BUILD'   , IntToStr(Version.Items[3]));
    Macros.Add('VERSION'         , Manifest.Version.DisplayValue);
    Macros.Add('VERSION_CODE'    , IntToStr(Manifest.Version.Code));
    Macros.Add('NAME'            , Name);
    Macros.Add('NAME_PASCAL'     , NamePascal);
    Macros.Add('QUALIFIED_NAME'  , QualifiedName);
    Macros.Add('CAPTION'         , Caption);
    Macros.Add('AUTHOR'          , NonEmptyAuthor);
    Macros.Add('EXECUTABLE_NAME' , ExecutableName);
    { TODO: Right now Delphi IDE compiles exe with basename matching
      DeleteFileExt(StandaloneSource), not our ExecutableName
      (which is sometimes different, e.g. doesn't contain "_standalone" suffix,
      has - instead of _).
      So we have DELPHI_EXECUTABLE_NAME.
      In the long run, it would be ideal to only have EXECUTABLE_NAME
      and make Delphi IDE build the same as Delphi command-line and FPC. }
    Macros.Add('DELPHI_EXECUTABLE_NAME', DeleteFileExt(StandaloneSource));
    Macros.Add('GAME_UNITS'      , Manifest.GameUnits);
    Macros.Add('SEARCH_PATHS'          , MakePathsStr(Manifest.SearchPaths, false));
    Macros.Add('ABSOLUTE_SEARCH_PATHS' , MakePathsStr(Manifest.SearchPaths, true));
    Macros.Add('LIBRARY_PATHS'         , MakePathsStr(Manifest.LibraryPaths, false));
    { Using this is important in ../data/custom_editor_template/castle_editor.lpi ,
      otherwise with FPC 3.3.1 (rev 40292) doing "castle-engine editor"
      fails when the project uses some libraries (like mORMot's .o files in static/). }
    Macros.Add('ABSOLUTE_LIBRARY_PATHS', MakePathsStr(Manifest.LibraryPaths, true));
    Macros.Add('CASTLE_ENGINE_PATH'    , CastleEnginePath);
    Macros.Add('EXTRA_COMPILER_OPTIONS', Manifest.ExtraCompilerOptions.Text);
    Macros.Add('EXTRA_COMPILER_OPTIONS_ABSOLUTE', Manifest.ExtraCompilerOptionsAbsolute.Text);
    Macros.Add('DEFINES_SEMICOLON_SEPARATED', DefinesSemicolonSeparated);
    Macros.Add('DEFINES_AS_COMPILER_OPTIONS', DefinesAsCompilerOptions);
    Macros.Add('EDITOR_UNITS'          , Manifest.EditorUnits);
    Macros.Add('EXPLICIT_STANDALONE_SOURCE', StandaloneSource);
    // Suitable name for Pascal "program" declaration
    Macros.Add('PASCAL_PROGRAM_NAME', TCastleManifest.StandaloneSourceToProgramName(StandaloneSource));
    Macros.Add('DELPHI_SEARCH_PATHS', DelphiSearchPaths);
    Macros.Add('ICO_PATH', IcoPath);
    Macros.Add('PROJECT_GUID', ProjectGuid);
    Macros.Add('APPLE_BUNDLE_SIGNATURE', Copy(ExecutableName + '????', 1, 4));
    Macros.Add('APPLE_ASSOCIATE_DOCUMENT_TYPES', AssociateDocumentTypes.ToPListSection(QualifiedName, Name));

    AddMacrosAndroid(Macros);
    AddMacrosIOS(Macros);
    AddMacrosLazarusProject(Macros);
    AddMacrosDproj(Macros, StandaloneSource);

    Result := ToolMacros.ReplaceMacros(Macros, Source);
  finally FreeAndNil(Macros) end;
end;

function TCastleProject.WelcomePage: String;

  procedure TryWelcomePage(const RelativeFileName: String);
  begin
    if FileExists(Path + RelativeFileName) then
      Result := RelativeFileName;
  end;

begin
  Result := '';

  TryWelcomePage('README.md');
  if Result <> '' then Exit;

  TryWelcomePage('README.txt');
  if Result <> '' then Exit;
end;

procedure TCastleProject.ExtractTemplate(const TemplatePath, DestinationPath: string;
  const OverrideExisting: boolean);
var
  TemplateFilesCount: Cardinal;
begin
  ExtractTemplateOverrideExisting := OverrideExisting;
  ExtractTemplateDestinationPath := InclPathDelim(DestinationPath);
  ExtractTemplateDir := ExclPathDelim(UriToFilenameSafe('castle-data:/' +TemplatePath));
  if not DirectoryExists(ExtractTemplateDir) then
    raise Exception.Create('Cannot find template in "' + ExtractTemplateDir + '". ' + SErrDataDir);

  TemplateFilesCount := FindFiles(ExtractTemplateDir, '*', false,
    @ExtractTemplateFoundFile, [ffRecursive]);
  if Verbose then
    Writeln(Format('Copied template "%s" (%d files) to "%s"',
      [TemplatePath, TemplateFilesCount, DestinationPath]));
end;

procedure TCastleProject.ExtractTemplateFoundFile(const FileInfo: TFileInfo; var StopSearch: boolean);
var
  DestinationRelativeFileName, DestinationFileName: string;
begin
  DestinationRelativeFileName := PrefixRemove(InclPathDelim(ExtractTemplateDir),
    FileInfo.AbsoluteName, true);

  if IsWild(DestinationRelativeFileName, '*setup_sdk.sh', true) or
     IsWild(DestinationRelativeFileName, '*~', true) or
     SameFileName(ExtractFileName(DestinationRelativeFileName), '.DS_Store') or
     SameFileName(ExtractFileName(DestinationRelativeFileName), 'thumbs.db') then
  begin
    // if Verbose then
    //   Writeln('Ignoring template file: ' + DestinationRelativeFileName);
    Exit;
  end;

  StringReplaceAllVar(DestinationRelativeFileName, 'cge_project_name', Name);

  DestinationFileName := ExtractTemplateDestinationPath + DestinationRelativeFileName;

  ExtractTemplateFile(FileInfo.AbsoluteName, DestinationFileName,
    DestinationRelativeFileName,
    ExtractTemplateOverrideExisting);
end;

procedure TCastleProject.ExtractTemplateFile(
  const SourceFileName, DestinationFileName, DestinationRelativeFileName: string;
  const OverrideExisting: boolean);
var
  DestinationRelativeFileNameSlashes, Contents, Ext: string;
  BinaryFile: boolean;
begin
  { Do not copy README.adoc, most services define it and would just overwrite each other.
    It is for informational purposes in CGE sources.
    Similarly do not copy CastleEngineService.xml, most services define it,
    it is internal info for build tool. }
  if SameText(DestinationRelativeFileName, 'README.adoc') or
     SameText(DestinationRelativeFileName, 'CastleEngineService.xml') then
    Exit;

  if (not OverrideExisting) and RegularFileExists(DestinationFileName) then
  begin
    DestinationRelativeFileNameSlashes := StringReplace(
      DestinationRelativeFileName, '\', '/', [rfReplaceAll]);

    if SameText(DestinationRelativeFileNameSlashes, Name + '/AppDelegate.m') then
      MergeIOSAppDelegate(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'Podfile') then
      MergeIOSPodfile(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, Name + '/' + Name + '-Info.plist') then
      MergeIOSInfoPlist(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/src/main/AndroidManifest.xml') then
      MergeAndroidManifest(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/src/main/CMakeLists.txt') then
      MergeAppend(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/src/main/res/values/strings.xml') then
      MergeStringsXml(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/src/main/java/io/castleengine/MainActivity.java') then
      MergeAndroidMainActivity(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/src/main/jni/Android.mk') or
       SameText(DestinationRelativeFileNameSlashes, 'app/src/main/custom-proguard-project.txt') then
      MergeAppend(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'app/build.gradle') then
      MergeBuildGradle(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
    if SameText(DestinationRelativeFileNameSlashes, 'build.gradle') then
      MergeBuildGradle(SourceFileName, DestinationFileName, @ReplaceMacros)
    else
      WritelnWarning('Template not overwriting custom ' + DestinationRelativeFileName);

    Exit;
  end;

  Ext := ExtractFileExt(SourceFileName);
  BinaryFile := SameText(Ext, '.so') or SameText(Ext, '.jar');
  CheckForceDirectories(ExtractFilePath(DestinationFileName));

  try
    if BinaryFile then
    begin
      CheckCopyFile(SourceFileName, DestinationFileName);
    end else
    begin
      Contents := FileToString(FilenameToUriSafe(SourceFileName));
      Contents := ReplaceMacros(Contents);
      StringToFile(FilenameToUriSafe(DestinationFileName), Contents);
    end;
  except
    on E: EFOpenError do
    begin
      Writeln('Cannot open a template file.');
      Writeln(SErrDataDir);
      raise;
    end;
  end;
end;

function TCastleProject.OutputPath: string;
begin
  if OutputPathBase = '' then
    Result := Path
  else
  begin
    Result := InclPathDelim(ExpandFileName(OutputPathBase));
    CheckForceDirectories(Result);
  end;
end;

procedure TCastleProject.CopyData(OutputDataPath: string; const TargetPlatform: TCastlePlatform);
var
  I: Integer;
  FileFrom, FileTo: string;
  Files: TCastleStringList;
begin
  OutputDataPath := InclPathDelim(OutputDataPath);
  ForceDirectories(OutputDataPath);

  Files := PackageFiles(true, TargetPlatform);
  try
    for I := 0 to Files.Count - 1 do
    begin
      FileFrom := DataPath + Files[I];
      FileTo := OutputDataPath + Files[I];
      SmartCopyFile(FileFrom, FileTo);
      if Verbose then
        Writeln('Packaging data file: ' + Files[I]);
    end;
  finally FreeAndNil(Files) end;

  GenerateDataInformation(OutputDataPath);
end;

function TCastleProject.PackageOutput(const FileName: String): Boolean;
var
  OS: TOS;
  CPU: TCPU;
  PackageFormat: TPackageFormatNoDefault;
  HasVersion: Boolean;
begin
  for OS in TOS do
    for CPU in TCPU do
      // TODO: This will not exclude output of packaging with pfDirectory
      for PackageFormat in TPackageFormatNoDefault do
        for HasVersion in Boolean do
          if OSCPUSupported[OS, CPU] then
            if SameFileName(FileName, PackageName(OS, CPU, PackageFormat, HasVersion)) then
              Exit(true);

  for HasVersion in Boolean do
    // list all package formats allowed for source packages now
    if SameFileName(FileName, SourcePackageName(HasVersion, pfZip)) or
       SameFileName(FileName, SourcePackageName(HasVersion, pfTarGz)) then
      Exit(true);

  if { avoid Android packages }
     IsWild(FileName, Name + '*-android-debug.apk', true) or
     IsWild(FileName, Name + '*-android-debug.aab', true) or
     IsWild(FileName, Name + '*-android-release.apk', true) or
     IsWild(FileName, Name + '*-android-release.aab', true) or
     { do not pack AndroidSigningProperties.txt (AndroidAntProperties.txt is older name for it) with private stuff }
     SameFileName(FileName, 'AndroidAntProperties.txt') or
     SameFileName(FileName, 'AndroidSigningProperties.txt') then
    Exit(true);

  Result := false;
end;

function TCastleProject.IOSHasLaunchImageStoryboard: Boolean;
begin
  Result :=
    (FLaunchImageStoryboardWidth <> 0) and
    (FLaunchImageStoryboardHeight <> 0);
end;

{ shortcut methods to acces Manifest.Xxx ------------------------------------- }

function TCastleProject.Version: TProjectVersion;
begin
  Result := Manifest.Version;
end;

function TCastleProject.ManifestCompiler: TCompiler;
begin
  Result := Manifest.Compiler;
end;

function TCastleProject.QualifiedName: string;
begin
  Result := Manifest.QualifiedName;
end;

function TCastleProject.Dependencies: TDependencies;
begin
  Result := Manifest.Dependencies;
end;

function TCastleProject.Name: string;
begin
  Result := Manifest.Name;
end;

function TCastleProject.Path: string;
begin
  Result := Manifest.Path;
end;

function TCastleProject.DataExists: Boolean;
begin
  Result := Manifest.DataExists;
end;

function TCastleProject.DataPath: string;
begin
  Result := Manifest.DataPath;
end;

function TCastleProject.Caption: string;
begin
  Result := Manifest.Caption;
end;

function TCastleProject.Author: string;
begin
  Result := Manifest.Author;
end;

function TCastleProject.ExecutableName: string;
begin
  Result := Manifest.ExecutableName;
end;

function TCastleProject.FullscreenImmersive: boolean;
begin
  Result := Manifest.FullscreenImmersive;
end;

function TCastleProject.ScreenOrientation: TScreenOrientation;
begin
  Result := Manifest.ScreenOrientation;
end;

function TCastleProject.AndroidCompileSdkVersion: Cardinal;
begin
  Result := Manifest.AndroidCompileSdkVersion;
end;

function TCastleProject.AndroidMinSdkVersion: Cardinal;
begin
  Result := Manifest.AndroidMinSdkVersion;
end;

function TCastleProject.AndroidTargetSdkVersion: Cardinal;
begin
  Result := Manifest.AndroidTargetSdkVersion;
end;

function TCastleProject.Icons: TImageFileNames;
begin
  Result := Manifest.Icons;
end;

function TCastleProject.LaunchImages: TImageFileNames;
begin
  Result := Manifest.LaunchImages;
end;

function TCastleProject.AndroidServices: TServiceList;
begin
  Result := Manifest.AndroidServices;
end;

function TCastleProject.IOSServices: TServiceList;
begin
  Result := Manifest.IOSServices;
end;

function TCastleProject.AssociateDocumentTypes: TAssociatedDocTypeList;
begin
  Result := Manifest.AssociateDocumentTypes;
end;

function TCastleProject.LocalizedAppNames: TLocalizedAppNameList;
begin
  Result := Manifest.LocalizedAppNames;
end;

function TCastleProject.LaunchImageStoryboard: TLaunchImageStoryboard;
begin
  Result := Manifest.LaunchImageStoryboard;
end;

end.
