// Copyright (C) 2024 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import {time} from '../base/time';

export interface TraceInfo {
  readonly traceTitle: string; // File name and size of the current trace.
  readonly traceUrl: string; // URL of the Trace.

  readonly start: time;
  readonly end: time;

  // This is the ts value at the time of the Unix epoch.
  // Normally some large negative value, because the unix epoch is normally in
  // the past compared to ts=0.
  readonly unixOffset: time;

  // Represents the reported timezone in minutes from UTC.
  readonly tzOffMin: number;

  // The number of import/analysis errors present in the `stats` table.
  readonly importErrors: number;

  // The trace type inferred by TraceProcessor (e.g. 'proto', 'json, ...).
  // See TraceTypeToString() in src/trace_processor/util/trace_type.cc for
  // all the available types.
  readonly traceType?: string;

  // True if the trace contains any ftrace data (sched or other ftrace events).
  readonly hasFtrace: boolean;

  // The UUID of the trace. This is generated by TraceProcessor by either
  // looking at the TraceUuid packet emitted by traced or, as a fallback, by
  // hashing the first KB of the trace. This can be an empty string in rare
  // cases (e.g., opening an empty trace).
  readonly uuid: string;

  // Wheteher the current trace has been successfully stored into cache storage.
  readonly cached: boolean;

  // Returns true if the current trace can be downloaded via getTraceFile().
  // The trace isn't downloadable in the following cases:
  // - It comes from a source (e.g. HTTP+RPC) that doesn't support re-download
  //   due to technical limitations.
  // - Download is disabled because the trace was pushed via postMessage and
  //   the caller has asked to disable downloads.
  readonly downloadable: boolean;
}
