/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.log;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.amoro.log.Converter;
import org.apache.amoro.log.LogData;
import org.apache.amoro.log.TimeFormats;
import org.apache.amoro.log.data.LogArrayData;
import org.apache.amoro.log.data.LogMapData;
import org.apache.amoro.shade.guava32.com.google.common.base.Preconditions;
import org.apache.amoro.shade.jackson2.com.fasterxml.jackson.databind.JsonNode;
import org.apache.amoro.shade.jackson2.com.fasterxml.jackson.databind.node.ArrayNode;
import org.apache.amoro.shade.jackson2.com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.amoro.shade.jackson2.com.fasterxml.jackson.databind.node.TextNode;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;

public class JsonToLogDataConverters<T>
implements Serializable {
    private static final long serialVersionUID = 647419880134661188L;
    private final boolean failOnMissingField = false;
    LogData.Factory<T> factory;
    LogArrayData.Factory arrayFactory;
    LogMapData.Factory mapFactory;

    public JsonToLogDataConverters(LogData.Factory<T> factory, LogArrayData.Factory arrayFactory, LogMapData.Factory mapFactory) {
        this.factory = (LogData.Factory)Preconditions.checkNotNull(factory);
        this.arrayFactory = (LogArrayData.Factory)Preconditions.checkNotNull((Object)arrayFactory);
        this.mapFactory = (LogMapData.Factory)Preconditions.checkNotNull((Object)mapFactory);
    }

    public JsonToLogDataConverter<T> createConverter(Type type) {
        return JsonToLogDataConverters.wrapIntoNullableConverter(this.createNotNullConverter(type));
    }

    private JsonToLogDataConverter<T> createNotNullConverter(Type type) {
        switch (type.typeId()) {
            case BOOLEAN: {
                return this::convertToBoolean;
            }
            case INTEGER: {
                return this::convertToInt;
            }
            case LONG: {
                return this::convertToLong;
            }
            case FLOAT: {
                return this::convertToFloat;
            }
            case DOUBLE: {
                return this::convertToDouble;
            }
            case DATE: {
                return this::convertToDate;
            }
            case TIME: {
                return this::convertToTime;
            }
            case TIMESTAMP: {
                Types.TimestampType timestamp = (Types.TimestampType)type;
                if (timestamp.shouldAdjustToUTC()) {
                    return this::convertToTimestampWithLocalZone;
                }
                return this::convertToTimestamp;
            }
            case STRING: {
                return this::convertToString;
            }
            case UUID: 
            case FIXED: 
            case BINARY: {
                return this::convertToBytes;
            }
            case DECIMAL: {
                return this::convertDecimal;
            }
            case LIST: {
                return this.createListConverter(type);
            }
            case MAP: {
                return this.createMapConverter(type);
            }
            case STRUCT: {
                return this.createStructConverter(type);
            }
        }
        throw new UnsupportedOperationException("Not Support to parse type: " + type);
    }

    private JsonToLogDataConverter<T> createStructConverter(Type type) {
        List fields = type.asNestedType().asStructType().fields();
        Type[] fieldTypes = (Type[])fields.stream().map(Types.NestedField::type).toArray(Type[]::new);
        String[] fieldNames = (String[])fields.stream().map(Types.NestedField::name).toArray(String[]::new);
        List fieldConverters = Arrays.stream(fieldTypes).map(this::createConverter).collect(Collectors.toList());
        return (jsonNode, context) -> {
            ObjectNode node = (ObjectNode)jsonNode;
            int arity = fieldNames.length;
            Object[] struct = new Object[arity];
            for (int i = 0; i < arity; ++i) {
                Object convertedField;
                String fieldName = fieldNames[i];
                JsonNode field = node.get(fieldName);
                struct[i] = convertedField = this.convertField((JsonToLogDataConverter)fieldConverters.get(i), fieldName, field, (Void)context);
            }
            return this.factory.createActualValue(struct, fieldTypes);
        };
    }

    private JsonToLogDataConverter<T> createMapConverter(Type type) {
        Types.MapType map = type.asNestedType().asMapType();
        Types.NestedField keyField = map.field(map.keyId());
        Types.NestedField valueField = map.field(map.valueId());
        JsonToLogDataConverter<T> keyConverter = this.createConverter(keyField.type());
        JsonToLogDataConverter<T> valueConverter = this.createConverter(valueField.type());
        return (jsonNode, context) -> {
            Iterator fields = jsonNode.fields();
            HashMap<Object, Object> result = new HashMap<Object, Object>();
            while (fields.hasNext()) {
                Map.Entry entry = (Map.Entry)fields.next();
                Object key = keyConverter.convert(TextNode.valueOf((String)((String)entry.getKey())), context);
                Object value = valueConverter.convert((JsonNode)entry.getValue(), context);
                key = this.convertSecondTimeIfNecessary(keyField.type(), key);
                value = this.convertSecondTimeIfNecessary(valueField.type(), value);
                result.put(key, value);
            }
            return this.mapFactory.create(result);
        };
    }

    private JsonToLogDataConverter<T> createListConverter(Type type) {
        Types.ListType list = type.asNestedType().asListType();
        Types.NestedField elementField = list.field(list.elementId());
        JsonToLogDataConverter<T> elementConverter = this.createConverter(elementField.type());
        Type elementType = elementField.type();
        return (jsonNode, context) -> {
            ArrayNode node = (ArrayNode)jsonNode;
            Object[] array = null;
            for (int i = 0; i < node.size(); ++i) {
                JsonNode innerNode = node.get(i);
                Object value = elementConverter.convert(innerNode, context);
                Object flinkValue = this.convertSecondTimeIfNecessary(elementType, value);
                if (flinkValue == null) continue;
                if (array == null) {
                    Class<?> flinkValueClass = flinkValue.getClass();
                    array = (Object[])Array.newInstance(flinkValueClass, node.size());
                }
                array[i] = flinkValue;
            }
            array = array == null ? new Object[node.size()] : array;
            return this.arrayFactory.create(array);
        };
    }

    private Object convertSecondTimeIfNecessary(Type type, Object object) {
        return this.factory.convertIfNecessary(type, object);
    }

    private Object convertDecimal(JsonNode jsonNode, Void context) {
        if (jsonNode.isBigDecimal()) {
            return jsonNode.decimalValue();
        }
        return new BigDecimal(jsonNode.asText());
    }

    private Object convertToBytes(JsonNode jsonNode, Void context) {
        try {
            return jsonNode.binaryValue();
        }
        catch (IOException e) {
            throw new JsonParseException("Unable to deserialize byte array.", e);
        }
    }

    private Object convertToString(JsonNode jsonNode, Void context) {
        if (jsonNode.isContainerNode()) {
            return jsonNode.toString();
        }
        return jsonNode.asText();
    }

    private Object convertToTimestampWithLocalZone(JsonNode jsonNode, Void context) {
        TemporalAccessor parsedTimestampWithLocalZone = TimeFormats.ISO8601_TIMESTAMP_WITH_LOCAL_TIMEZONE_FORMAT.parse(jsonNode.asText());
        LocalTime localTime = parsedTimestampWithLocalZone.query(TemporalQueries.localTime());
        LocalDate localDate = parsedTimestampWithLocalZone.query(TemporalQueries.localDate());
        return LocalDateTime.of(localDate, localTime).toInstant(ZoneOffset.UTC);
    }

    private Object convertToTimestamp(JsonNode jsonNode, Void context) {
        TemporalAccessor parsedTimestamp = TimeFormats.SQL_TIMESTAMP_FORMAT.parse(jsonNode.asText());
        LocalTime localTime = parsedTimestamp.query(TemporalQueries.localTime());
        LocalDate localDate = parsedTimestamp.query(TemporalQueries.localDate());
        return LocalDateTime.of(localDate, localTime);
    }

    private boolean convertToBoolean(JsonNode jsonNode, Void context) {
        if (jsonNode.isBoolean()) {
            return jsonNode.asBoolean();
        }
        return Boolean.parseBoolean(jsonNode.asText().trim());
    }

    private int convertToInt(JsonNode jsonNode, Void context) {
        if (jsonNode.canConvertToInt()) {
            return jsonNode.asInt();
        }
        return Integer.parseInt(jsonNode.asText().trim());
    }

    private long convertToLong(JsonNode jsonNode, Void context) {
        if (jsonNode.canConvertToLong()) {
            return jsonNode.asLong();
        }
        return Long.parseLong(jsonNode.asText().trim());
    }

    private float convertToFloat(JsonNode jsonNode, Void context) {
        if (jsonNode.isDouble()) {
            return (float)jsonNode.asDouble();
        }
        return Float.parseFloat(jsonNode.asText().trim());
    }

    private double convertToDouble(JsonNode jsonNode, Void context) {
        if (jsonNode.isDouble()) {
            return jsonNode.asDouble();
        }
        return Double.parseDouble(jsonNode.asText().trim());
    }

    private int convertToDate(JsonNode jsonNode, Void context) {
        LocalDate date = DateTimeFormatter.ISO_LOCAL_DATE.parse(jsonNode.asText()).query(TemporalQueries.localDate());
        return (int)date.toEpochDay();
    }

    private long convertToTime(JsonNode jsonNode, Void context) {
        TemporalAccessor parsedTime = TimeFormats.SQL_TIME_FORMAT.parse(jsonNode.asText());
        LocalTime localTime = parsedTime.query(TemporalQueries.localTime());
        return localTime.toNanoOfDay();
    }

    private Object convertField(JsonToLogDataConverter<T> fieldConverter, String fieldName, JsonNode field, Void context) {
        if (field == null) {
            return null;
        }
        return fieldConverter.convert(field, context);
    }

    private static <T> JsonToLogDataConverter<T> wrapIntoNullableConverter(JsonToLogDataConverter<T> converter) {
        return (source, context) -> {
            if (source == null || source.isNull() || source.isMissingNode()) {
                return null;
            }
            return converter.convert(source, context);
        };
    }

    static interface JsonToLogDataConverter<T>
    extends Converter<JsonNode, Object, Void, T> {
    }

    private static final class JsonParseException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public JsonParseException(String message) {
            super(message);
        }

        public JsonParseException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

